/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier;

import io.confluent.kafka.availability.FilesWrapper;
import io.confluent.kafka.availability.ThreadCountersManager;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import kafka.tier.domain.AbstractTierMetadata;
import kafka.tier.state.OffsetAndEpoch;
import kafka.tier.topic.TierTopicManagerConfig;
import kafka.tier.topic.TierTopicUtils;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.storage.internals.log.LogDirFailureChannel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TierTopicManagerCommitter {
    public static final VersionInfo CURRENT_VERSION = VersionInfo.VERSION_1;
    private static final Logger log = LoggerFactory.getLogger(TierTopicManagerCommitter.class);
    private static final int NO_EPOCH = -1;
    private static final String SEPARATOR = " ";
    private final TierTopicManagerConfig config;
    private final LogDirFailureChannel logDirFailureChannel;
    private final Map<Integer, OffsetAndEpoch> partitionToPosition = new ConcurrentHashMap<Integer, OffsetAndEpoch>();

    public TierTopicManagerCommitter(TierTopicManagerConfig config, LogDirFailureChannel logDirFailureChannel) {
        if (config.logDirs.size() != 1) {
            throw new UnsupportedOperationException("TierTopicManager does not currently support multiple logdirs.");
        }
        this.config = config;
        this.logDirFailureChannel = logDirFailureChannel;
        this.clearTempFiles();
        this.loadOffsets();
    }

    public void updatePosition(int partition, OffsetAndEpoch updateTo, AbstractTierMetadata entry) {
        OffsetAndEpoch current = this.partitionToPosition.getOrDefault(partition, OffsetAndEpoch.EMPTY);
        if (current.offset() >= updateTo.offset()) {
            throw new IllegalStateException(String.format("Illegal offset in %s with current position %s for event: %s", updateTo, current, entry));
        }
        if (current.epoch().isPresent() && updateTo.epoch().isPresent() && current.epoch().get() > updateTo.epoch().get()) {
            throw new IllegalStateException(String.format("Illegal epoch in %s with current position %s for event: %s", updateTo, current, entry));
        }
        this.partitionToPosition.put(partition, updateTo);
        log.debug("Committer position updated {}:{}", (Object)partition, (Object)updateTo);
    }

    public void replacePositions(Map<String, Map<Integer, OffsetAndEpoch>> replacePartitionToPosition) throws IOException {
        Map<String, Map<Integer, OffsetAndEpoch>> logDirTargetPositions = this.readOffsetsForLogDirs();
        logDirTargetPositions.forEach((logDir, offsets) -> offsets.putAll((Map)replacePartitionToPosition.get(logDir)));
        for (Map.Entry<String, Map<Integer, OffsetAndEpoch>> entry : logDirTargetPositions.entrySet()) {
            String logDir2 = entry.getKey();
            Map<Integer, OffsetAndEpoch> targetPartitionToPosition = entry.getValue();
            log.info("For logDir {}, flushing new tier.offsets file to disk with rewound offsets {}", (Object)logDir2, (Object)TierTopicUtils.rewindPositionsAsString(targetPartitionToPosition));
            this.writeOffsetsForLogDir(logDir2, CURRENT_VERSION, targetPartitionToPosition);
        }
        HashMap<Integer, OffsetAndEpoch> localReplacePartitionToPosition = new HashMap<Integer, OffsetAndEpoch>();
        replacePartitionToPosition.forEach((logDir, offsets) -> localReplacePartitionToPosition.putAll((Map<Integer, OffsetAndEpoch>)offsets));
        log.info("Updating committer's local state with {}.", (Object)TierTopicUtils.rewindPositionsAsString(localReplacePartitionToPosition));
        this.partitionToPosition.putAll(localReplacePartitionToPosition);
    }

    public OffsetAndEpoch positionFor(int partitionId) {
        return this.partitionToPosition.get(partitionId);
    }

    public synchronized Map<Integer, OffsetAndEpoch> takePositionsSnapshot() {
        return new HashMap<Integer, OffsetAndEpoch>(this.partitionToPosition);
    }

    public void writePositionsSnapshot(Map<Integer, OffsetAndEpoch> positions) {
        this.writeOffsets(CURRENT_VERSION, positions);
    }

    static Map<Integer, OffsetAndEpoch> earliestOffsets(List<Map<Integer, OffsetAndEpoch>> diskOffsets) {
        if (diskOffsets.stream().map(Map::keySet).collect(Collectors.toSet()).size() != 1) {
            return Collections.emptyMap();
        }
        HashMap<Integer, OffsetAndEpoch> minimum = new HashMap<Integer, OffsetAndEpoch>();
        for (Map<Integer, OffsetAndEpoch> offsets : diskOffsets) {
            log.debug("Loading offsets from logdir {}.", diskOffsets);
            for (Map.Entry<Integer, OffsetAndEpoch> entry : offsets.entrySet()) {
                minimum.compute(entry.getKey(), (k, v) -> {
                    if (v == null || ((OffsetAndEpoch)entry.getValue()).offset() < v.offset()) {
                        return (OffsetAndEpoch)entry.getValue();
                    }
                    return v;
                });
            }
        }
        log.debug("Minimum offsets found {}.", minimum);
        return minimum;
    }

    public Map<String, ByteBuffer> readOffsetsForLogDirsToByteBuf() throws IOException {
        HashMap<String, ByteBuffer> logDirOffsets = new HashMap<String, ByteBuffer>();
        for (String logDir : this.config.logDirs) {
            ByteBuffer logDirPositions = (ByteBuffer)ThreadCountersManager.wrapIOChecked(() -> {
                try {
                    byte[] fileAsBytes = Files.readAllBytes(Paths.get(TierTopicManagerCommitter.commitPath(logDir), new String[0]));
                    return ByteBuffer.wrap(fileAsBytes);
                }
                catch (FileNotFoundException fnf) {
                    log.info("TierTopicManagerCommitter offsets not found. This is expected if this is the first time starting up with tiered storage.", (Throwable)fnf);
                    return null;
                }
                catch (IOException ioe) {
                    log.error("Error loading TierTopicManagerCommitter offsets. Setting logdir offline.", (Throwable)ioe);
                    this.logDirFailureChannel.maybeAddOfflineLogDir(logDir, String.format("Failed to commit tier offsets to logdir %s.", logDir), ioe);
                    throw ioe;
                }
                catch (Exception ex) {
                    log.warn("Exception encountered when reading tier checkpoint file.", (Throwable)ex);
                    throw ex;
                }
            });
            logDirOffsets.put(logDir, logDirPositions);
        }
        return logDirOffsets;
    }

    public Map<String, Map<Integer, OffsetAndEpoch>> readOffsetsForLogDirs() throws IOException {
        HashMap<String, Map<Integer, OffsetAndEpoch>> logDirOffsets = new HashMap<String, Map<Integer, OffsetAndEpoch>>();
        for (String logDir : this.config.logDirs) {
            logDirOffsets.put(logDir, TierTopicManagerCommitter.readCommittedOffsetsForLogDir(logDir, this.logDirFailureChannel));
        }
        return logDirOffsets;
    }

    private static String commitPath(String logDir) {
        return logDir + "/tier.offsets";
    }

    private static String commitTempFilename(String logDir) {
        return TierTopicManagerCommitter.commitPath(logDir) + ".tmp";
    }

    private void clearTempFiles() {
        for (String logDir : this.config.logDirs) {
            try {
                FilesWrapper.deleteIfExists((Path)Paths.get(TierTopicManagerCommitter.commitTempFilename(logDir), new String[0]));
            }
            catch (IOException ioe) {
                this.logDirFailureChannel.maybeAddOfflineLogDir(logDir, String.format("Failed to delete temporary tier offsets in logdir %s.", logDir), ioe);
            }
        }
    }

    static Map<Integer, OffsetAndEpoch> readCommittedOffsetsForLogDir(String logDir, LogDirFailureChannel logDirFailureChannel) throws IOException {
        return (Map)ThreadCountersManager.wrapIOChecked(() -> {
            try (FileReader fr = new FileReader(TierTopicManagerCommitter.commitPath(logDir));){
                Map<Integer, OffsetAndEpoch> map;
                try (BufferedReader br = new BufferedReader(fr);){
                    String versionLine = br.readLine();
                    map = TierTopicManagerCommitter.readPayload(br, TierTopicManagerCommitter.readVersion(versionLine));
                }
                return map;
            }
            catch (FileNotFoundException fnf) {
                log.info("TierTopicManagerCommitter offsets not found. This is expected if this is the first time starting up with tiered storage.", (Throwable)fnf);
            }
            catch (IOException ioe) {
                log.error("Error loading TierTopicManagerCommitter offsets. Setting logdir offline.", (Throwable)ioe);
                logDirFailureChannel.maybeAddOfflineLogDir(logDir, String.format("Failed to commit tier offsets to logdir %s.", logDir), ioe);
                throw ioe;
            }
            catch (Exception ex) {
                log.warn("Exception encountered when reading tier checkpoint file.", (Throwable)ex);
                throw ex;
            }
            return Collections.emptyMap();
        });
    }

    static Map<Integer, OffsetAndEpoch> readCommittedOffsetsForLogDirIgnoreException(String logDir, LogDirFailureChannel logDirFailureChannel) {
        Map<Integer, OffsetAndEpoch> committedOffsets = Collections.emptyMap();
        try {
            committedOffsets = TierTopicManagerCommitter.readCommittedOffsetsForLogDir(logDir, logDirFailureChannel);
        }
        catch (IOException iOException) {
        }
        catch (Exception ignored) {
            log.warn("Resetting tier offsets...");
        }
        return committedOffsets;
    }

    private static VersionInfo readVersion(String line) {
        return VersionInfo.toVersionInfo(Integer.parseInt(line));
    }

    private static Map<Integer, OffsetAndEpoch> readPayload(BufferedReader br, VersionInfo versionInfo) throws IOException {
        String line;
        HashMap<Integer, OffsetAndEpoch> committedPositions = new HashMap<Integer, OffsetAndEpoch>();
        while ((line = br.readLine()) != null) {
            String[] values = line.split(SEPARATOR);
            if (values.length == versionInfo.numFields) {
                OffsetAndEpoch previousPosition;
                int deserializedEpoch;
                int partitionId = Integer.parseInt(values[0]);
                long offset = Long.parseLong(values[1]);
                Optional<Integer> epoch = Optional.empty();
                if (versionInfo.version > VersionInfo.VERSION_0.version && (deserializedEpoch = Integer.parseInt(values[2])) != -1) {
                    epoch = Optional.of(deserializedEpoch);
                }
                if ((previousPosition = committedPositions.put(partitionId, new OffsetAndEpoch(offset, epoch))) == null) continue;
                throw new IllegalStateException("Found duplicate positions for partition " + partitionId);
            }
            throw new IllegalStateException("Committed offsets found in incorrect format on line " + line);
        }
        return committedPositions;
    }

    private void loadOffsets() {
        Map<Integer, OffsetAndEpoch> earliest = TierTopicManagerCommitter.earliestOffsets(this.config.logDirs.stream().map(logDir -> TierTopicManagerCommitter.readCommittedOffsetsForLogDirIgnoreException(logDir, this.logDirFailureChannel)).collect(Collectors.toList()));
        this.partitionToPosition.clear();
        this.partitionToPosition.putAll(earliest);
    }

    void writeOffsets(VersionInfo versionInfo, Map<Integer, OffsetAndEpoch> offsets) {
        ThreadCountersManager.wrapIOVoid(() -> {
            for (String logDir : this.config.logDirs) {
                try {
                    this.writeOffsetsForLogDir(logDir, versionInfo, offsets);
                }
                catch (IOException iOException) {}
            }
        });
    }

    private void writeOffsetsForLogDir(String logDir, VersionInfo versionInfo, Map<Integer, OffsetAndEpoch> offsets) throws IOException {
        try {
            try (FileWriter fw = new FileWriter(TierTopicManagerCommitter.commitTempFilename(logDir));
                 BufferedWriter bw = new BufferedWriter(fw);){
                bw.write(String.valueOf(versionInfo.version));
                bw.newLine();
                for (Map.Entry<Integer, OffsetAndEpoch> entry : offsets.entrySet()) {
                    int partitionId = entry.getKey();
                    long offset = entry.getValue().offset();
                    int epoch = entry.getValue().epoch().orElse(-1);
                    bw.write(partitionId + SEPARATOR + offset);
                    if (versionInfo.version > VersionInfo.VERSION_0.version) {
                        bw.write(SEPARATOR + epoch);
                    }
                    bw.newLine();
                }
            }
            Utils.atomicMoveWithFallback((Path)Paths.get(TierTopicManagerCommitter.commitTempFilename(logDir), new String[0]), (Path)Paths.get(TierTopicManagerCommitter.commitPath(logDir), new String[0]));
        }
        catch (IOException ioe) {
            this.logDirFailureChannel.maybeAddOfflineLogDir(logDir, String.format("Failed to commit tier offsets to logdir %s.", logDir), ioe);
            throw ioe;
        }
    }

    static enum VersionInfo {
        VERSION_0(0, 2),
        VERSION_1(1, 3);

        private static final Map<Integer, VersionInfo> VERSION_MAP;
        final int version;
        final int numFields;

        private VersionInfo(int version, int numFields) {
            this.version = version;
            this.numFields = numFields;
        }

        public static VersionInfo toVersionInfo(int version) {
            VersionInfo versionInfo = VERSION_MAP.get(version);
            if (versionInfo == null) {
                throw new IllegalStateException("Unknown version " + version);
            }
            return versionInfo;
        }

        static {
            VERSION_MAP = new HashMap<Integer, VersionInfo>();
            for (VersionInfo versionInfo : VersionInfo.values()) {
                VersionInfo oldVersion = VERSION_MAP.put(versionInfo.version, versionInfo);
                if (oldVersion == null) continue;
                throw new ExceptionInInitializerError("Found duplicate version " + versionInfo.version);
            }
        }
    }
}

