/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.backupObjectLifecycle;

import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.MultiObjectDeleteException;
import com.azure.storage.blob.models.BlobErrorCode;
import com.azure.storage.blob.models.BlobStorageException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;
import kafka.tier.TopicIdPartition;
import kafka.tier.backupObjectLifecycle.LifecycleManager;
import kafka.tier.backupObjectLifecycle.ObjectStoreUtilsContext;
import kafka.tier.backupObjectLifecycle.RetryPolicy;
import kafka.tier.backupObjectLifecycle.TierObjectStoreBatchOperationException;
import kafka.tier.domain.TierPartitionForceRestore;
import kafka.tier.exceptions.TierObjectStoreFatalException;
import kafka.tier.exceptions.TierObjectStoreRetriableException;
import kafka.tier.fetcher.TierStateFetcher;
import kafka.tier.store.MockInMemoryTierObjectStore;
import kafka.tier.store.S3VersionInformation;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.TierObjectStoreResponse;
import kafka.tier.store.VersionInformation;
import kafka.tier.store.objects.FragmentType;
import kafka.tier.store.objects.ObjectType;
import kafka.tier.store.objects.metadata.BackupObjectsListMetadata;
import kafka.tier.store.objects.metadata.ObjectStoreMetadata;
import kafka.tier.store.objects.metadata.TierStateRestoreSnapshotMetadata;
import kafka.utils.CoreUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ObjectStoreUtils {
    private static final Logger log = LoggerFactory.getLogger(ObjectStoreUtils.class);
    public static final int DEFAULT_BATCH_SIZE_TO_SUBMIT = 100;

    public static Map<String, List<VersionInformation>> verifyObjectNotLive(ObjectStoreUtilsContext ctx, TopicIdPartition topicIdPartition, UUID objectId, RetryPolicy retryPolicy) throws InterruptedException {
        log.debug("LifecycleManager verifying if the segment " + CoreUtils.uuidToBase64(objectId) + " from topicIdPartition " + String.valueOf(topicIdPartition) + " is live");
        Map<String, List<VersionInformation>> versionLists = new HashMap<String, List<VersionInformation>>();
        boolean done = false;
        int numRetries = 0;
        while (!done) {
            ctx.isCancelled();
            try {
                switch (ctx.store.getBackend()) {
                    case AzureBlockBlob: {
                        versionLists = ObjectStoreUtils.verifyAzureBlockBlobObjectNotLive(ctx.store, topicIdPartition, objectId);
                        break;
                    }
                    case GCS: {
                        versionLists = ObjectStoreUtils.verifyGCSObjectNotLive(ctx.store, topicIdPartition, objectId);
                        break;
                    }
                    case S3: {
                        versionLists = ObjectStoreUtils.verifyS3ObjectNotLive(ctx.store, topicIdPartition, objectId);
                        break;
                    }
                    case Mock: {
                        versionLists = ObjectStoreUtils.verifyMockObjectNotLive(ctx.store, topicIdPartition, objectId);
                        break;
                    }
                    default: {
                        log.warn("CLM does not support " + String.valueOf((Object)ctx.store.getBackend()) + " object store yet");
                    }
                }
                done = true;
            }
            catch (TierObjectStoreRetriableException e) {
                log.warn("ObjectStore returned exception in response to listObjects call " + e.getMessage());
                if (++numRetries >= retryPolicy.maxRetries()) {
                    throw new TierObjectStoreFatalException("Exhausted max retries for object store list call. ", (Throwable)((Object)e));
                }
                Thread.sleep(retryPolicy.backOffMs(numRetries));
            }
            log.debug("Exiting the list loop. done " + done + " numRetries: " + numRetries);
        }
        return versionLists;
    }

    private static Map<String, List<VersionInformation>> verifyMockObjectNotLive(TierObjectStore tierObjectStore, TopicIdPartition topicIdPartition, UUID objectId) {
        String keyPrefix = TierObjectStore.DataTypePathPrefix.TOPIC.prefix + "/" + CoreUtils.uuidToBase64(objectId) + "/" + topicIdPartition.topicIdAsBase64() + "/" + topicIdPartition.partition();
        Map<String, List<VersionInformation>> versionLists = tierObjectStore.listObject(keyPrefix, true);
        HashMap<String, List<VersionInformation>> returnVersionLists = new HashMap<String, List<VersionInformation>>();
        for (Map.Entry<String, List<VersionInformation>> entry : versionLists.entrySet()) {
            String key = entry.getKey();
            boolean entryDeleted = false;
            returnVersionLists.put(key, new LinkedList());
            for (VersionInformation e : entry.getValue()) {
                if (e.getVersionId().equals(MockInMemoryTierObjectStore.deleteMarker.getVersionId())) {
                    entryDeleted = true;
                    continue;
                }
                ((List)returnVersionLists.get(key)).add(e);
            }
            if (entryDeleted) continue;
            String msg = "CLM consumed a Segment Delete Complete marker for " + key + " but the blob has not been deleted at object store. Could be an error at some Tiered Storage component. CLM will not delete any blob under this segment.";
            log.error(msg);
            throw new IllegalArgumentException(msg);
        }
        return returnVersionLists;
    }

    private static Map<String, List<VersionInformation>> verifyAzureBlockBlobObjectNotLive(TierObjectStore tierObjectStore, TopicIdPartition topicIdPartition, UUID objectId) {
        String keyPrefix = TierObjectStore.DataTypePathPrefix.TOPIC.prefix() + "/" + CoreUtils.uuidToBase64(objectId) + "/" + topicIdPartition.topicIdAsBase64() + "/" + topicIdPartition.partition();
        Map<String, List<VersionInformation>> versionLists = tierObjectStore.listObject(keyPrefix, true);
        if (versionLists.isEmpty()) {
            return versionLists;
        }
        Map<String, List<VersionInformation>> objectPresenceCheck = tierObjectStore.listObject(keyPrefix, false);
        objectPresenceCheck.forEach((key, versions) -> log.error("LifecycleManager consumed a SegmentDeleteComplete event pointing to a blob that is still live at the object store.\n" + key + "\nThis could either be due to an error at some Tiered Storage component or that this segment is being \nrestored as part of recovery. LifecycleManager will not process this deletion and remove the object from its deletion list."));
        if (!objectPresenceCheck.isEmpty()) {
            throw new IllegalArgumentException("Objects under the prefix " + keyPrefix + " have not been deleted. LifecycleManager will not process them.");
        }
        return versionLists;
    }

    private static Map<String, List<VersionInformation>> verifyGCSObjectNotLive(TierObjectStore tierObjectStore, TopicIdPartition topicIdPartition, UUID objectId) {
        String keyPrefix = TierObjectStore.DataTypePathPrefix.TOPIC.prefix + "/" + CoreUtils.uuidToBase64(objectId) + "/" + topicIdPartition.topicIdAsBase64() + "/" + topicIdPartition.partition();
        Map<String, List<VersionInformation>> versionLists = tierObjectStore.listObject(keyPrefix, true);
        Map<String, List<VersionInformation>> objectPresenceCheck = tierObjectStore.listObject(keyPrefix, false);
        if (!objectPresenceCheck.isEmpty()) {
            for (Map.Entry<String, List<VersionInformation>> entry : objectPresenceCheck.entrySet()) {
                String msg = "LifecycleManager consumed a SegmentDeleteComplete event pointing to a blob that is still live at the object store.\n" + entry.getKey() + "\nThis could either be due to an error at some Tiered Storage component or that this segment is being \nrestored as part of recovery. LifecycleManager will not process this deletion and remove the object from its deletion list.";
                log.error(msg);
            }
            throw new IllegalArgumentException("Objects under the prefix " + keyPrefix + " have not been deleted. LifecycleManager will not process them.");
        }
        return versionLists;
    }

    private static Map<String, List<VersionInformation>> verifyS3ObjectNotLive(TierObjectStore tierObjectStore, TopicIdPartition topicIdPartition, UUID objectId) {
        String keyPrefix = TierObjectStore.DataTypePathPrefix.TOPIC.prefix + "/" + CoreUtils.uuidToBase64(objectId) + "/" + topicIdPartition.topicIdAsBase64() + "/" + topicIdPartition.partition();
        Map<String, List<VersionInformation>> versionLists = tierObjectStore.listObject(keyPrefix, true);
        log.debug("S3TierObjectStore returned " + versionLists.size() + " blobs under the prefix " + keyPrefix);
        for (Map.Entry<String, List<VersionInformation>> entry : versionLists.entrySet()) {
            List<VersionInformation> versions = entry.getValue();
            boolean isObjectDeleted = false;
            for (VersionInformation version : versions) {
                if (!((S3VersionInformation)version).isObjectDeleted()) continue;
                isObjectDeleted = true;
                break;
            }
            if (isObjectDeleted) continue;
            String msg = "LifecycleManager consumed a SegmentDeleteComplete event pointing to a blob that is still live at the object store.\n" + entry.getKey() + "\nThis could either be due to an error at some Tiered Storage component or that this segment is being \nrestored as part of recovery. LifecycleManager will not process this deletion and remove the object from its deletion list.";
            log.error(msg);
            throw new IllegalArgumentException("Objects under the prefix " + keyPrefix + " have not been deleted. LifecycleManager will not process them.");
        }
        return versionLists;
    }

    public static Set<String> backupObjectListNames(ObjectStoreUtilsContext ctx, String clusterId) throws InterruptedException {
        BackupObjectsListMetadata backupObjectsListMetadata = new BackupObjectsListMetadata(clusterId, "", "");
        Map<String, List<VersionInformation>> objectLists = ObjectStoreUtils.listObject(ctx, backupObjectsListMetadata.generateKeyPrefix(""), false, LifecycleManager.DEFAULT_RETRY_POLICY);
        return objectLists.keySet().stream().filter(key -> key.endsWith(ObjectType.BACKUP_OBJECTS_LIST.suffix())).collect(Collectors.toSet());
    }

    public static Map<String, List<VersionInformation>> listObject(ObjectStoreUtilsContext ctx, String keyPrefix, Boolean getVersionInfo, RetryPolicy retryPolicy) throws InterruptedException {
        int numRetries = 0;
        while (true) {
            ctx.isCancelled();
            try {
                return ctx.store.listObject(keyPrefix, getVersionInfo);
            }
            catch (TierObjectStoreRetriableException e) {
                if (++numRetries >= retryPolicy.maxRetries()) {
                    throw new TierObjectStoreFatalException("Failed to list " + keyPrefix + " after " + numRetries + " tries.", (Throwable)((Object)e));
                }
                log.warn("Retry listObject call. Retry # " + numRetries + ". " + String.valueOf((Object)e));
                Thread.sleep(retryPolicy.backOffMs(numRetries));
                continue;
            }
            break;
        }
    }

    public static void putBuffer(ObjectStoreUtilsContext ctx, ObjectStoreMetadata objectMetadata, ByteBuffer buf, ObjectType objectType, RetryPolicy retryPolicy) throws InterruptedException {
        int numRetries = 0;
        while (true) {
            ctx.isCancelled();
            try {
                ctx.store.putBuffer(objectMetadata, buf, objectType);
                return;
            }
            catch (TierObjectStoreFatalException e) {
                throw e;
            }
            catch (IOException | RuntimeException e) {
                Exception cause = (Exception)e.getCause();
                if (ObjectStoreUtils.isEncryptionKeyStateInvalid(cause) || ++numRetries >= retryPolicy.maxRetries()) {
                    throw new TierObjectStoreFatalException("Failed to put buffer to object store after " + numRetries + " tries.", cause);
                }
                log.warn("Retry putBuf call. Retry # " + numRetries + ". " + String.valueOf(e));
                Thread.sleep(retryPolicy.backOffMs(numRetries));
                continue;
            }
            break;
        }
    }

    public static boolean isEncryptionKeyStateInvalid(Exception e) {
        if (e instanceof AmazonS3Exception) {
            String errorCode = ((AmazonS3Exception)e).getErrorCode();
            return Objects.equals(errorCode, "KMS.KMSInvalidStateException") || Objects.equals(errorCode, "KMS.DisabledException");
        }
        return false;
    }

    public static TierObjectStoreResponse getObjectStoreFragment(ObjectStoreUtilsContext ctx, ObjectStoreMetadata metadata, FragmentType fragmentType, RetryPolicy retryPolicy) throws InterruptedException {
        int numRetries = 0;
        while (true) {
            ctx.isCancelled();
            try {
                return ctx.store.getObjectStoreFragment(metadata, fragmentType);
            }
            catch (TierObjectStoreFatalException e) {
                throw e;
            }
            catch (IOException | RuntimeException e) {
                Exception cause = (Exception)e.getCause();
                if (ObjectStoreUtils.isEncryptionKeyStateInvalid(cause) || ObjectStoreUtils.isObjectNotFound(e, ctx.store.getBackend()) || ++numRetries >= retryPolicy.maxRetries()) {
                    throw new TierObjectStoreFatalException("Failed to get object from object store after " + numRetries + " tries.", cause);
                }
                log.warn("Retry getObject call. Retry # " + numRetries + ". " + String.valueOf(e));
                Thread.sleep(retryPolicy.backOffMs(numRetries));
                continue;
            }
            break;
        }
    }

    private static boolean isObjectNotFound(Exception e, TierObjectStore.Backend backend) {
        if (backend == TierObjectStore.Backend.Mock && e instanceof TierObjectStoreRetriableException && e.getMessage().startsWith("Key not found:")) {
            return true;
        }
        return e instanceof BlobStorageException && ((BlobStorageException)e).getErrorCode().equals((Object)BlobErrorCode.BLOB_NOT_FOUND);
    }

    public static void deleteVersionedObjectsAsync(ObjectStoreUtilsContext ctx, List<TierObjectStore.KeyAndVersion> keys, ThreadPoolExecutor executor) throws TierObjectStoreBatchOperationException, InterruptedException {
        ArrayList futures = new ArrayList();
        ConcurrentHashMap<TierObjectStore.KeyAndVersion, Throwable> errors = new ConcurrentHashMap<TierObjectStore.KeyAndVersion, Throwable>();
        int startIndex = 0;
        while (startIndex < keys.size()) {
            ctx.isCancelled();
            int endIndex = Math.min(startIndex + 100, keys.size());
            List<TierObjectStore.KeyAndVersion> subList = keys.subList(startIndex, endIndex);
            subList.forEach(key -> futures.add(CompletableFuture.runAsync(() -> ctx.store.deleteVersions(Collections.singletonList(key)), executor).exceptionally(ex -> {
                if (ex.getCause() instanceof TierObjectStoreRetriableException && ex.getCause().getCause() instanceof BlobStorageException) {
                    BlobStorageException be = (BlobStorageException)ex.getCause().getCause();
                    if (!BlobErrorCode.BLOB_NOT_FOUND.equals((Object)be.getErrorCode())) {
                        errors.put((TierObjectStore.KeyAndVersion)key, ex.getCause());
                    }
                    return null;
                }
                errors.put((TierObjectStore.KeyAndVersion)key, ex.getCause());
                return null;
            })));
            CompletableFuture<Void> allFutures = CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
            allFutures.join();
            futures.clear();
            startIndex = endIndex;
        }
        if (!errors.isEmpty()) {
            throw new TierObjectStoreBatchOperationException(errors);
        }
    }

    public static void deleteVersions(ObjectStoreUtilsContext ctx, List<TierObjectStore.KeyAndVersion> objectsToDelete, ThreadPoolExecutor executor, RetryPolicy retryPolicy) throws InterruptedException {
        ArrayList<TierObjectStore.KeyAndVersion> copy = new ArrayList<TierObjectStore.KeyAndVersion>(objectsToDelete);
        int numRetries = 0;
        while (!copy.isEmpty() && numRetries++ < retryPolicy.maxRetries()) {
            ctx.isCancelled();
            try {
                if (ctx.store.getBackend() == TierObjectStore.Backend.AzureBlockBlob) {
                    ObjectStoreUtils.deleteVersionedObjectsAsync(ctx, copy, executor);
                } else {
                    ctx.store.deleteVersions(copy);
                }
                return;
            }
            catch (TierObjectStoreRetriableException e) {
                if (e.getCause() instanceof MultiObjectDeleteException) {
                    MultiObjectDeleteException throwable = (MultiObjectDeleteException)e.getCause();
                    copy.clear();
                    throwable.getErrors().forEach(error -> {
                        if (!error.getCode().trim().equals("NoSuchVersion")) {
                            copy.add(new TierObjectStore.KeyAndVersion(error.getKey(), error.getVersionId()));
                        }
                    });
                }
                log.warn("Retry versioned objects deletion. Retry # " + numRetries + ". " + String.valueOf((Object)e));
            }
            catch (TierObjectStoreBatchOperationException e) {
                Map<TierObjectStore.KeyAndVersion, Throwable> errors = e.getErrors();
                Iterator iter = copy.iterator();
                while (iter.hasNext()) {
                    TierObjectStore.KeyAndVersion key = (TierObjectStore.KeyAndVersion)iter.next();
                    if (errors.containsKey(key)) {
                        Throwable ex = errors.get(key);
                        if (!(ex instanceof TierObjectStoreFatalException)) continue;
                        throw (TierObjectStoreFatalException)ex;
                    }
                    iter.remove();
                }
                log.warn("Retry versioned objects deletion. Retry # " + numRetries + ". " + String.valueOf(e));
            }
            Thread.sleep(retryPolicy.backOffMs(numRetries));
        }
    }

    public static ByteBuffer fetchRecoverSnapshot(ObjectStoreUtilsContext ctx, TierPartitionForceRestore forceRestoreEvent, RetryPolicy retryPolicy) throws InterruptedException {
        int numRetries = 0;
        while (true) {
            ctx.isCancelled();
            try {
                TierStateFetcher tierStateFetcher = new TierStateFetcher(1, ctx.store);
                return tierStateFetcher.fetchRecoverSnapshot(new TierStateRestoreSnapshotMetadata(forceRestoreEvent));
            }
            catch (TierObjectStoreRetriableException e) {
                if (++numRetries >= retryPolicy.maxRetries()) {
                    throw new TierObjectStoreFatalException("Failed to fetch recover snapshot " + String.valueOf(forceRestoreEvent) + " after " + numRetries + " tries.", (Throwable)((Object)e));
                }
                log.warn("Retriable error fetch recover snapshot, metadata {}. Backing off for {} ms. Retry count: {}", new Object[]{forceRestoreEvent, retryPolicy.backOffMs(numRetries), numRetries, e});
                Thread.sleep(retryPolicy.backOffMs(numRetries));
                continue;
            }
            break;
        }
    }

    public static class DeletionRecord {
        private final UUID objectId;
        private final TopicIdPartition topicIdPartition;
        private final Long creationTime;

        public DeletionRecord(UUID objectId, TopicIdPartition tp, Long creationTime) {
            this.objectId = objectId;
            this.topicIdPartition = tp;
            this.creationTime = creationTime;
        }

        public String getTopicName() {
            return this.topicIdPartition.topic();
        }

        public Long getCreationTime() {
            return this.creationTime;
        }

        public TopicIdPartition getTopicIdPartition() {
            return this.topicIdPartition;
        }

        public UUID getObjectId() {
            return this.objectId;
        }
    }
}

