/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.tools.recovery;

import org.apache.kafka.common.metadata.MetadataRecordType;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.image.MetadataDelta;
import org.apache.kafka.image.MetadataImage;
import org.apache.kafka.image.MetadataProvenance;
import org.apache.kafka.metadata.MetadataEncryptorFactory;
import org.apache.kafka.raft.Batch;
import org.apache.kafka.server.common.ApiMessageAndVersion;
import org.slf4j.Logger;

public class MetadataBatchLoader {
    private final Logger log;
    private final MetadataEncryptorFactory encryptorFactory;
    private MetadataImage image;
    private MetadataDelta delta;
    private long lastOffset;
    private int lastEpoch;
    private long lastContainedLogTimeMs;
    private MetadataProvenance provenance;
    private int numBatches;
    private TransactionState transactionState;

    public MetadataBatchLoader(LogContext logContext, MetadataEncryptorFactory encryptorFactory) {
        this.log = logContext.logger(MetadataBatchLoader.class);
        this.encryptorFactory = encryptorFactory;
        this.resetToImage(MetadataImage.EMPTY);
    }

    private void resetToImage(MetadataImage image) {
        this.image = image;
        this.delta = new MetadataDelta.Builder().setImage(image).setMetadataEncryptorFactory(this.encryptorFactory).build();
        this.transactionState = TransactionState.NO_TRANSACTION;
        this.lastOffset = image.provenance().lastContainedOffset();
        this.lastEpoch = image.provenance().lastContainedEpoch();
        this.lastContainedLogTimeMs = image.provenance().lastContainedLogTimeMs();
        this.numBatches = 0;
    }

    public MetadataImage getImage() {
        return this.image;
    }

    public void loadBatch(Batch<ApiMessageAndVersion> batch) {
        this.loadBatchWithinOffsets(batch, 0L, Long.MAX_VALUE);
    }

    public void loadBatchWithinOffsets(Batch<ApiMessageAndVersion> batch, long startingOffset, long endingOffset) {
        int indexWithinBatch = 0;
        long currOffset = batch.baseOffset() - 1L;
        this.lastContainedLogTimeMs = batch.appendTimestamp();
        this.lastEpoch = batch.epoch();
        for (ApiMessageAndVersion record : batch.records()) {
            try {
                if (++currOffset < startingOffset) continue;
                if (currOffset >= endingOffset) break;
                this.replay(record);
            }
            catch (Throwable e) {
                throw new RuntimeException("Error loading metadata log record from offset " + (batch.baseOffset() + (long)indexWithinBatch), e);
            }
            if (this.transactionState == TransactionState.STARTED_TRANSACTION && (indexWithinBatch > 0 || this.numBatches > 0)) {
                this.provenance = new MetadataProvenance(this.lastOffset, this.lastEpoch, this.lastContainedLogTimeMs, indexWithinBatch == 0);
                if (this.log.isDebugEnabled()) {
                    this.log.debug("handleCommit: Generated a metadata delta between {} and {} from {} batch(es).", new Object[]{this.image.offset(), this.provenance.lastContainedOffset(), this.numBatches});
                }
                this.applyDeltaAndUpdate(this.delta);
                this.transactionState = TransactionState.STARTED_TRANSACTION;
            }
            this.lastOffset = batch.baseOffset() + (long)indexWithinBatch;
            ++indexWithinBatch;
        }
        this.lastOffset = batch.lastOffset();
        ++this.numBatches;
    }

    public void maybeFlushBatches() {
        this.provenance = new MetadataProvenance(this.lastOffset, this.lastEpoch, this.lastContainedLogTimeMs, true);
        switch (this.transactionState.ordinal()) {
            case 1: 
            case 2: {
                this.log.debug("handleCommit: not publishing since a transaction starting at {} is still in progress. {} batch(es) processed so far.", (Object)this.image.offset(), (Object)this.numBatches);
                break;
            }
            case 4: {
                this.log.debug("handleCommit: publishing empty delta between {} and {} from {} batch(es) since a transaction was aborted", new Object[]{this.image.offset(), this.provenance.lastContainedOffset(), this.numBatches});
                this.applyDeltaAndUpdate(new MetadataDelta.Builder().setImage(this.image).setMetadataEncryptorFactory(this.encryptorFactory).build());
                break;
            }
            case 0: 
            case 3: {
                if (this.log.isDebugEnabled()) {
                    this.log.debug("handleCommit: Generated a metadata delta between {} and {} from {} batch(es).", new Object[]{this.image.offset(), this.provenance.lastContainedOffset(), this.numBatches});
                }
                this.applyDeltaAndUpdate(this.delta);
            }
        }
    }

    private void replay(ApiMessageAndVersion record) {
        MetadataRecordType type = MetadataRecordType.fromId((short)record.message().apiKey());
        switch (type) {
            case BEGIN_TRANSACTION_RECORD: {
                if (this.transactionState == TransactionState.STARTED_TRANSACTION || this.transactionState == TransactionState.CONTINUED_TRANSACTION) {
                    throw new RuntimeException("Encountered BeginTransactionRecord while already in a transaction");
                }
                this.transactionState = TransactionState.STARTED_TRANSACTION;
                break;
            }
            case END_TRANSACTION_RECORD: {
                if (this.transactionState == TransactionState.CONTINUED_TRANSACTION || this.transactionState == TransactionState.STARTED_TRANSACTION) {
                    this.transactionState = TransactionState.ENDED_TRANSACTION;
                    break;
                }
                throw new RuntimeException("Encountered EndTransactionRecord without having seen a BeginTransactionRecord");
            }
            case ABORT_TRANSACTION_RECORD: {
                if (this.transactionState == TransactionState.CONTINUED_TRANSACTION || this.transactionState == TransactionState.STARTED_TRANSACTION) {
                    this.transactionState = TransactionState.ABORTED_TRANSACTION;
                    break;
                }
                throw new RuntimeException("Encountered AbortTransactionRecord without having seen a BeginTransactionRecord");
            }
            default: {
                switch (this.transactionState.ordinal()) {
                    case 1: {
                        this.transactionState = TransactionState.CONTINUED_TRANSACTION;
                        break;
                    }
                    case 3: 
                    case 4: {
                        this.transactionState = TransactionState.NO_TRANSACTION;
                        break;
                    }
                }
                this.delta.replay(record.message(), record.version());
            }
        }
    }

    private void applyDeltaAndUpdate(MetadataDelta delta) {
        try {
            this.image = delta.apply(this.provenance);
        }
        catch (Throwable e) {
            throw new RuntimeException("Error generating new metadata image from metadata delta between offset " + this.image.offset() + " and " + this.provenance.lastContainedOffset(), e);
        }
        this.resetToImage(this.image);
    }

    static enum TransactionState {
        NO_TRANSACTION,
        STARTED_TRANSACTION,
        CONTINUED_TRANSACTION,
        ENDED_TRANSACTION,
        ABORTED_TRANSACTION;

    }
}

