/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.hub.platform;

import io.confluent.connect.hub.cli.ExitCode;
import io.confluent.connect.hub.exceptions.ConfluentHubClientException;
import io.confluent.connect.hub.io.Storage;
import io.confluent.connect.hub.platform.Installation;
import io.confluent.connect.hub.platform.InstallationType;
import io.confluent.connect.hub.utils.IOUtils;
import io.confluent.connect.hub.utils.NamingUtils;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.Pair;

public class PlatformInspector {
    private static final String BIN_ABS_PATH_OPTION_NAME = "bin.abs.path";
    private static final String UNABLE_TO_SCAN_PROCESSES_MSG = "Unable to list processes";
    private static final String UNABLE_TO_PARSE_PS_MSG = "Unable to parse output of ps command";
    private static final String UNKNOWN_INSTALLATION_TYPE_MSG = "Unknown installation type";
    private static final String DETECTED_CONNECT_PROCESS_MSG = "Detected Connect process: ";
    private static final String CLIENT_INSTALLATION_TYPE_MSG_FORMAT = "Client's installation type is: %s";
    private static final String ABS_PATH_IS_NOT_SET_MSG = "Property bin.abs.path is not set.";
    private static final String NO_INSTALLATION_DETECTED_MSG = "Unable to detect Confluent Platform installation. Specify --component-dir and --worker-configs explicitly.";
    private static final String CONNECT_PROCESS_REGEX = "org.apache.kafka.connect.cli.Connect(Distributed|Standalone)";
    private static final String ARCHIVE_CONFLUENT_CURRENT_CONFIG = "/connect/connect.properties";
    private static final String STANDARD_STANDALONE_CONFIG = "/etc/kafka/connect-standalone.properties";
    private static final String STANDARD_DISTRIBUTED_CONFIG = "/etc/kafka/connect-distributed.properties";
    private static final String STANDARD_SCHEMA_REGISTRY_STANDALONE_CONFIG = "/etc/schema-registry/connect-avro-standalone.properties";
    private static final String STANDARD_SCHEMA_REGISTRY_DISTRIBUTED_CONFIG = "/etc/schema-registry/connect-avro-distributed.properties";
    private static final String CONFLUENT_CURRENT_ENV_VAR = "CONFLUENT_CURRENT";
    private static final String TMPDIR_ENV_VAR = "TMPDIR";
    public static final String COMPONENT_DIR_ARCHIVE_DEFAULT = "share/confluent-hub-components";
    public static final String COMPONENT_DIR_PACKAGE_DEFAULT = "/usr/share/confluent-hub-components";
    private static final String PACKAGE_SHARE_JAVA_BASE_PATH = "/usr";
    private static final String ARCHIVE_SHARE_JAVA_PATH = "share/java/confluent-common";
    private static final String TEMP_DIR_DEFAULT = "/tmp/";
    private static final String BIN_ABS_PATH_PACKAGE = "/usr/bin";
    private static final String CONFLUENT_CURRENT_PROPERTIES_FILE_NAME = "confluent.current";
    private static final String STANDARD_CONFIG_TYPE = "Standard";
    private static final String CURRENT_CONFIG_TYPE = "Based on CONFLUENT_CURRENT";
    private static final String DAEMON_OPTION = "-daemon";
    private final Runtime runtime;
    private static final String CONFLUENT_HOME_NAME = "CONFLUENT_HOME";
    private static final String CONNECT_SCRIPT_PATH = "/usr/bin/connect-distributed";
    private static final String CURRENT_DIR_INSTALLATION_MSG = "(found in the current directory)";
    private static final String STANDARD_INSTALLATION_MSG = "(installed rpm/deb package)";
    private static final String BASED_ON_CONFLUENT_HOME_INSTALLATION_MSG = "(based on $CONFLUENT_HOME)";
    private static final String CLIENT_INSTALLATION_MSG = "(where this tool is installed)";
    private Storage storage;

    public PlatformInspector(Storage storage, Runtime runtime) {
        this.storage = storage;
        this.runtime = runtime;
    }

    private Installation getClientInstallation() {
        String binAbsPath = this.getBinAbsPath();
        if (BIN_ABS_PATH_PACKAGE.equals(binAbsPath)) {
            IOUtils.trace(String.format(CLIENT_INSTALLATION_TYPE_MSG_FORMAT, new Object[]{InstallationType.PACKAGE}), new Object[0]);
            return new Installation(InstallationType.PACKAGE, "/");
        }
        IOUtils.trace(String.format(CLIENT_INSTALLATION_TYPE_MSG_FORMAT, new Object[]{InstallationType.ARCHIVE}), new Object[0]);
        return new Installation(InstallationType.ARCHIVE, this.storage.getParentDirectoryPath(binAbsPath));
    }

    public List<Pair<String, String>> detectConfigs(Installation installation) {
        ArrayList<Pair<String, String>> result = new ArrayList<Pair<String, String>>();
        result.addAll(this.detectDistributionConfigs(installation));
        result.addAll(this.detectProcessesConfigs());
        List<Pair<String, String>> existingDetectedConfigs = result.stream().filter(config -> {
            if (this.storage.exists((String)config.getValue())) {
                return true;
            }
            IOUtils.trace("{} {} was detected but it doesn't exist, excluding it", config.getKey(), config.getValue());
            return false;
        }).collect(Collectors.toList());
        return existingDetectedConfigs;
    }

    private List<Pair<String, String>> detectProcessesConfigs() {
        try {
            String command = "ps ax| grep -E \"org.apache.kafka.connect.cli.Connect(Distributed|Standalone)\" | grep -v grep; test ${PIPESTATUS[0]} -eq 0";
            Process ps = this.runtime.exec(new String[]{"/bin/bash", "-c", command});
            ps.waitFor();
            if (ps.exitValue() != 0) {
                String stderr = org.apache.commons.io.IOUtils.toString((InputStream)ps.getErrorStream(), (Charset)Charset.defaultCharset());
                IOUtils.warn(stderr, new Object[0]);
                throw new ConfluentHubClientException(stderr, ExitCode.UNKNOWN_ERROR);
            }
            BufferedReader br = new BufferedReader(new InputStreamReader(ps.getInputStream(), StandardCharsets.UTF_8));
            ArrayList<Pair<String, String>> result = new ArrayList<Pair<String, String>>();
            br.lines().forEach(l -> {
                String[] split = l.split(" ");
                if (split.length < 2) {
                    throw new ConfluentHubClientException(UNABLE_TO_PARSE_PS_MSG, ExitCode.UNKNOWN_ERROR);
                }
                boolean isFirst = true;
                boolean reachedArgs = false;
                String processId = null;
                for (String s : split) {
                    if (isFirst) {
                        processId = s;
                        isFirst = false;
                        continue;
                    }
                    if (s.matches(CONNECT_PROCESS_REGEX)) {
                        reachedArgs = true;
                        continue;
                    }
                    if (!reachedArgs || DAEMON_OPTION.equals(s)) continue;
                    result.add(Pair.of((Object)String.format("Used by Connect process with PID %s", processId), (Object)s));
                    break;
                }
                IOUtils.trace(DETECTED_CONNECT_PROCESS_MSG + l, new Object[0]);
            });
            return result;
        }
        catch (IOException e) {
            throw new ConfluentHubClientException(UNABLE_TO_SCAN_PROCESSES_MSG, e, ExitCode.UNKNOWN_ERROR);
        }
        catch (InterruptedException e) {
            throw new ConfluentHubClientException(UNABLE_TO_SCAN_PROCESSES_MSG, e, ExitCode.UNKNOWN_ERROR);
        }
    }

    private List<Pair<String, String>> detectDistributionConfigs(Installation installation) {
        ArrayList<Pair<String, String>> result = new ArrayList<Pair<String, String>>();
        switch (installation.getInstallationType()) {
            case ARCHIVE: {
                result.add(Pair.of((Object)STANDARD_CONFIG_TYPE, (Object)NamingUtils.getArchiveItemAbsolutePath(installation.getInstallationPath(), STANDARD_DISTRIBUTED_CONFIG)));
                result.add(Pair.of((Object)STANDARD_CONFIG_TYPE, (Object)NamingUtils.getArchiveItemAbsolutePath(installation.getInstallationPath(), STANDARD_STANDALONE_CONFIG)));
                result.add(Pair.of((Object)STANDARD_CONFIG_TYPE, (Object)NamingUtils.getArchiveItemAbsolutePath(installation.getInstallationPath(), STANDARD_SCHEMA_REGISTRY_DISTRIBUTED_CONFIG)));
                result.add((Pair<String, String>)Pair.of((Object)STANDARD_CONFIG_TYPE, (Object)NamingUtils.getArchiveItemAbsolutePath(installation.getInstallationPath(), STANDARD_SCHEMA_REGISTRY_STANDALONE_CONFIG)));
                this.getConfluentCurrent().ifPresent(s -> result.add(Pair.of((Object)CURRENT_CONFIG_TYPE, (Object)(s + ARCHIVE_CONFLUENT_CURRENT_CONFIG))));
                break;
            }
            case PACKAGE: {
                result.add(Pair.of((Object)STANDARD_CONFIG_TYPE, (Object)STANDARD_DISTRIBUTED_CONFIG));
                result.add(Pair.of((Object)STANDARD_CONFIG_TYPE, (Object)STANDARD_STANDALONE_CONFIG));
                result.add(Pair.of((Object)STANDARD_CONFIG_TYPE, (Object)STANDARD_SCHEMA_REGISTRY_DISTRIBUTED_CONFIG));
                result.add(Pair.of((Object)STANDARD_CONFIG_TYPE, (Object)STANDARD_SCHEMA_REGISTRY_STANDALONE_CONFIG));
                break;
            }
            default: {
                throw new IllegalArgumentException(UNKNOWN_INSTALLATION_TYPE_MSG);
            }
        }
        return result;
    }

    private String getBinAbsPath() {
        String binAbsPath = System.getProperty(BIN_ABS_PATH_OPTION_NAME);
        if (binAbsPath == null || binAbsPath.isEmpty()) {
            throw new ConfluentHubClientException(ABS_PATH_IS_NOT_SET_MSG, ExitCode.INVALID_OPTS_OR_ARGS);
        }
        return binAbsPath;
    }

    private Optional<String> getConfluentCurrent() {
        String tempDir = this.getEnvVar(TMPDIR_ENV_VAR, TEMP_DIR_DEFAULT);
        String confluentCurrentVar = this.getEnvVar(CONFLUENT_CURRENT_ENV_VAR, tempDir);
        String path = confluentCurrentVar + CONFLUENT_CURRENT_PROPERTIES_FILE_NAME;
        if (!this.storage.exists(path)) {
            return Optional.empty();
        }
        List<String> lines = this.storage.readLines(path);
        if (lines.size() == 1) {
            String confluentCurrent = lines.get(0);
            if (this.storage.exists(confluentCurrent)) {
                return Optional.of(confluentCurrent);
            }
            return Optional.empty();
        }
        IOUtils.warn("Invalid content of file " + path + ". It should point to the root of extracted archive with Confluent distribution.", new Object[0]);
        return Optional.empty();
    }

    public String getComponentDir(Installation installation) {
        switch (installation.getInstallationType()) {
            case ARCHIVE: {
                return NamingUtils.getArchiveItemAbsolutePath(installation.getInstallationPath(), COMPONENT_DIR_ARCHIVE_DEFAULT);
            }
            case PACKAGE: {
                return COMPONENT_DIR_PACKAGE_DEFAULT;
            }
        }
        throw new IllegalArgumentException(UNKNOWN_INSTALLATION_TYPE_MSG);
    }

    private String getEnvVar(String name, String defaultValue) {
        String value = System.getenv(name);
        value = value == null || value.isEmpty() ? defaultValue : value;
        return value;
    }

    public Pair<Installation, String> detectInstallation() {
        List<Pair<Installation, String>> installations = this.detectInstallations();
        return installations.get(0);
    }

    public List<Pair<Installation, String>> detectInstallations() {
        Installation clientInstallation;
        String currentDirectory;
        ArrayList<Pair<Installation, String>> result = new ArrayList<Pair<Installation, String>>();
        String confluentHomeValue = this.getEnvVar(CONFLUENT_HOME_NAME, null);
        if (null != confluentHomeValue && this.hasArchiveInstallation(confluentHomeValue)) {
            result.add(Pair.of((Object)new Installation(InstallationType.ARCHIVE, confluentHomeValue), (Object)BASED_ON_CONFLUENT_HOME_INSTALLATION_MSG));
        }
        if (this.hasArchiveInstallation(currentDirectory = System.getProperty("user.dir"))) {
            result.add(Pair.of((Object)new Installation(InstallationType.ARCHIVE, currentDirectory), (Object)CURRENT_DIR_INSTALLATION_MSG));
        }
        if (this.hasPackageInstallation()) {
            result.add(Pair.of((Object)new Installation(InstallationType.PACKAGE, "/"), (Object)STANDARD_INSTALLATION_MSG));
        }
        if (this.hasArchiveInstallation((clientInstallation = this.getClientInstallation()).getInstallationPath()) || clientInstallation.getInstallationPath().equals("/") && this.hasPackageInstallation()) {
            result.add(Pair.of((Object)clientInstallation, (Object)CLIENT_INSTALLATION_MSG));
        }
        if (result.size() == 0) {
            throw new ConfluentHubClientException(NO_INSTALLATION_DETECTED_MSG, ExitCode.INVALID_OPTS_OR_ARGS);
        }
        return result;
    }

    private boolean hasPackageInstallation() {
        return this.storage.exists(CONNECT_SCRIPT_PATH);
    }

    private boolean hasArchiveInstallation(String root) {
        return !PACKAGE_SHARE_JAVA_BASE_PATH.equals(root) && this.storage.exists(NamingUtils.getArchiveItemAbsolutePath(root, ARCHIVE_SHARE_JAVA_PATH));
    }
}

