/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.storage.internals.log;

import io.confluent.kafka.storage.checksum.ChecksumParams;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiPredicate;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.KafkaStorageException;
import org.apache.kafka.common.errors.OffsetOutOfRangeException;
import org.apache.kafka.common.message.FetchResponseData;
import org.apache.kafka.common.record.FileLogInputStream;
import org.apache.kafka.common.record.FileRecords;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.Records;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.server.util.Scheduler;
import org.apache.kafka.storage.internals.log.AbortedTxn;
import org.apache.kafka.storage.internals.log.FetchDataInfo;
import org.apache.kafka.storage.internals.log.LogConfig;
import org.apache.kafka.storage.internals.log.LogDirFailureChannel;
import org.apache.kafka.storage.internals.log.LogFileUtils;
import org.apache.kafka.storage.internals.log.LogOffsetMetadata;
import org.apache.kafka.storage.internals.log.LogOffsetsListener;
import org.apache.kafka.storage.internals.log.LogSegment;
import org.apache.kafka.storage.internals.log.LogSegments;
import org.apache.kafka.storage.internals.log.LogTruncation;
import org.apache.kafka.storage.internals.log.OffsetPosition;
import org.apache.kafka.storage.internals.log.SegmentDeletionReason;
import org.apache.kafka.storage.internals.log.StorageAction;
import org.apache.kafka.storage.internals.log.TxnIndexSearchResult;
import org.apache.kafka.storage.log.metrics.BrokerTopicStats;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocalLog {
    private static final Logger LOG = LoggerFactory.getLogger(LocalLog.class);
    public static final Pattern DELETE_DIR_PATTERN = Pattern.compile("^(\\S+)-(\\S+)\\.(\\S+)-delete");
    public static final Pattern FUTURE_DIR_PATTERN = Pattern.compile("^(\\S+)-(\\S+)\\.(\\S+)-future");
    public static final Pattern STRAY_DIR_PATTERN = Pattern.compile("^(\\S+)-(\\S+)\\.(\\S+)-stray");
    public static final long UNKNOWN_OFFSET = -1L;
    private final AtomicLong lastFlushedTime;
    private final String logIdent;
    private final LogSegments segments;
    private final Scheduler scheduler;
    private final Time time;
    private final TopicPartition topicPartition;
    private final LogDirFailureChannel logDirFailureChannel;
    private final Logger logger;
    private final BrokerTopicStats brokerTopicStats;
    private final ChecksumParams checksumParams;
    private volatile LogOffsetsListener logOffsetsListener;
    private volatile LogOffsetMetadata nextOffsetMetadata;
    private volatile boolean isMemoryMappedBufferClosed = false;
    private volatile String parentDir;
    private volatile LogConfig config;
    private volatile long recoveryPoint;
    private File dir;

    public LocalLog(File dir, LogConfig config, LogSegments segments, long recoveryPoint, LogOffsetMetadata nextOffsetMetadata, Scheduler scheduler, Time time, TopicPartition topicPartition, LogDirFailureChannel logDirFailureChannel, BrokerTopicStats brokerTopicStats, LogOffsetsListener logOffsetsListener, ChecksumParams checksumParams) {
        this.dir = dir;
        this.config = config;
        this.segments = segments;
        this.recoveryPoint = recoveryPoint;
        this.nextOffsetMetadata = nextOffsetMetadata;
        this.scheduler = scheduler;
        this.time = time;
        this.topicPartition = topicPartition;
        this.logDirFailureChannel = logDirFailureChannel;
        this.brokerTopicStats = brokerTopicStats;
        this.logOffsetsListener = logOffsetsListener;
        this.checksumParams = checksumParams;
        this.logIdent = "[LocalLog partition=" + String.valueOf(topicPartition) + ", dir=" + String.valueOf(dir) + "] ";
        this.logger = new LogContext(this.logIdent).logger(LocalLog.class);
        this.lastFlushedTime = new AtomicLong(time.milliseconds());
        this.parentDir = dir.getParent();
        logOffsetsListener.onEndOffsetUpdated(nextOffsetMetadata.messageOffset);
    }

    public void setLogOffsetsListener(LogOffsetsListener listener) {
        this.logOffsetsListener = listener;
    }

    public File dir() {
        return this.dir;
    }

    public Logger logger() {
        return this.logger;
    }

    public LogConfig config() {
        return this.config;
    }

    public LogSegments segments() {
        return this.segments;
    }

    public Scheduler scheduler() {
        return this.scheduler;
    }

    public LogOffsetMetadata nextOffsetMetadata() {
        return this.nextOffsetMetadata;
    }

    public TopicPartition topicPartition() {
        return this.topicPartition;
    }

    public LogDirFailureChannel logDirFailureChannel() {
        return this.logDirFailureChannel;
    }

    public BrokerTopicStats brokerTopicStats() {
        return this.brokerTopicStats;
    }

    public LogOffsetsListener logOffsetsListener() {
        return this.logOffsetsListener;
    }

    public long recoveryPoint() {
        return this.recoveryPoint;
    }

    public Time time() {
        return this.time;
    }

    public String name() {
        return this.dir.getName();
    }

    public String parentDir() {
        return this.parentDir;
    }

    public File parentDirFile() {
        return new File(this.parentDir);
    }

    public boolean isFuture() {
        return this.dir.getName().endsWith("-future");
    }

    public boolean isDeleted() {
        return this.dir.getName().endsWith("-delete");
    }

    public boolean isStray() {
        return this.dir.getName().endsWith("-stray");
    }

    private <T> T maybeHandleIOException(Supplier<String> errorMsgSupplier, StorageAction<T, IOException> function) {
        return LocalLog.maybeHandleIOException(this.logDirFailureChannel, this.parentDir, errorMsgSupplier, function);
    }

    public boolean renameDir(String name) {
        return this.maybeHandleIOException(() -> "Error while renaming dir for " + String.valueOf(this.topicPartition) + " in log dir " + this.dir.getParent(), () -> {
            File renamedDir = new File(this.dir.getParent(), name);
            Utils.atomicMoveWithFallback((Path)this.dir.toPath(), (Path)renamedDir.toPath());
            if (!renamedDir.equals(this.dir)) {
                this.dir = renamedDir;
                this.parentDir = renamedDir.getParent();
                this.segments.updateParentDir(renamedDir);
                return true;
            }
            return false;
        });
    }

    public LogConfig updateConfig(LogConfig newConfig) {
        LogConfig oldConfig = this.config;
        this.config = newConfig;
        return oldConfig;
    }

    public void checkIfMemoryMappedBufferClosed() {
        if (this.isMemoryMappedBufferClosed) {
            throw new KafkaStorageException("The memory mapped buffer for log of " + String.valueOf(this.topicPartition) + " is already closed");
        }
    }

    public void updateRecoveryPoint(long newRecoveryPoint) {
        this.recoveryPoint = newRecoveryPoint;
    }

    public void markFlushed(long offset) {
        this.checkIfMemoryMappedBufferClosed();
        if (offset > this.recoveryPoint) {
            this.updateRecoveryPoint(offset);
            this.lastFlushedTime.set(this.time.milliseconds());
        }
    }

    public long unflushedMessages() {
        return this.logEndOffset() - this.recoveryPoint;
    }

    public void flush(long offset) throws IOException {
        long currentRecoveryPoint = this.recoveryPoint;
        if (currentRecoveryPoint <= offset) {
            Collection<LogSegment> segmentsToFlush = this.segments.values(currentRecoveryPoint, offset);
            for (LogSegment logSegment : segmentsToFlush) {
                logSegment.flush();
            }
            if (segmentsToFlush.stream().anyMatch(s -> s.baseOffset() >= currentRecoveryPoint)) {
                Utils.flushDirIfExists((Path)this.dir.toPath());
            }
        }
    }

    public long lastFlushTime() {
        return this.lastFlushedTime.get();
    }

    public LogOffsetMetadata logEndOffsetMetadata() {
        return this.nextOffsetMetadata;
    }

    public long logEndOffset() {
        return this.nextOffsetMetadata.messageOffset;
    }

    public void updateLogEndOffset(long endOffset) {
        this.nextOffsetMetadata = new LogOffsetMetadata(endOffset, this.segments.activeSegment().baseOffset(), this.segments.activeSegment().size());
        this.logOffsetsListener.onEndOffsetUpdated(this.nextOffsetMetadata.messageOffset);
        if (this.recoveryPoint > endOffset) {
            this.updateRecoveryPoint(endOffset);
        }
    }

    public void closeHandlers() {
        this.segments.closeHandlers();
        this.isMemoryMappedBufferClosed = true;
    }

    public void close() {
        this.maybeHandleIOException(() -> "Error while renaming dir for " + String.valueOf(this.topicPartition) + " in dir " + this.dir.getParent(), () -> {
            this.checkIfMemoryMappedBufferClosed();
            this.segments.close();
            return null;
        });
    }

    public void deleteEmptyDir() {
        this.maybeHandleIOException(() -> "Error while deleting dir for " + String.valueOf(this.topicPartition) + " in dir " + this.dir.getParent(), () -> {
            if (!this.segments.isEmpty()) {
                throw new IllegalStateException("Can not delete directory when " + this.segments.numberOfSegments() + " segments are still present");
            }
            if (!this.isMemoryMappedBufferClosed) {
                throw new IllegalStateException("Can not delete directory when memory mapped buffer for log of " + String.valueOf(this.topicPartition) + " is still open.");
            }
            Utils.delete((File)this.dir);
            return null;
        });
    }

    public List<LogSegment> deleteAllSegments() {
        return this.maybeHandleIOException(() -> String.format("Error while deleting all segments for %s in dir %s", this.topicPartition, this.dir.getParent()), () -> {
            ArrayList<LogSegment> deletableSegments = new ArrayList<LogSegment>(this.segments.values());
            this.removeAndDeleteSegments(this.segments.values(), false, toDelete -> this.logger.info("Deleting segments as the log has been deleted: {}", (Object)toDelete.stream().map(LogSegment::toString).collect(Collectors.joining(", "))));
            this.isMemoryMappedBufferClosed = true;
            return deletableSegments;
        });
    }

    public List<LogSegment> deletableSegments(BiPredicate<LogSegment, Optional<LogSegment>> predicate, int maxNumSegmentsToDelete) {
        if (this.segments.isEmpty() || maxNumSegmentsToDelete <= 0) {
            return new ArrayList<LogSegment>();
        }
        ArrayList<LogSegment> deletable = new ArrayList<LogSegment>();
        Iterator<LogSegment> segmentsIterator = this.segments.values().iterator();
        Optional<LogSegment> segmentOpt = LocalLog.nextOptional(segmentsIterator);
        while (segmentOpt.isPresent() && deletable.size() < maxNumSegmentsToDelete) {
            boolean isLastSegmentAndEmpty;
            LogSegment segment = segmentOpt.get();
            Optional<LogSegment> nextSegmentOpt = LocalLog.nextOptional(segmentsIterator);
            boolean bl = isLastSegmentAndEmpty = nextSegmentOpt.isEmpty() && segment.size() == 0;
            if (predicate.test(segment, nextSegmentOpt) && !isLastSegmentAndEmpty) {
                deletable.add(segment);
                segmentOpt = nextSegmentOpt;
                continue;
            }
            segmentOpt = Optional.empty();
        }
        return deletable;
    }

    private static <T> Optional<T> nextOptional(Iterator<T> iterator) {
        return iterator.hasNext() ? Optional.of(iterator.next()) : Optional.empty();
    }

    public void removeAndDeleteSegments(Collection<LogSegment> segmentsToDelete, boolean asyncDelete, SegmentDeletionReason reason) throws IOException {
        if (!segmentsToDelete.isEmpty()) {
            ArrayList<LogSegment> toDelete = new ArrayList<LogSegment>(segmentsToDelete);
            reason.logReason(toDelete);
            toDelete.forEach(segment -> this.segments.remove(segment.baseOffset()));
            LocalLog.deleteSegmentFiles(toDelete, asyncDelete, this.dir, this.topicPartition, this.config, this.scheduler, this.logDirFailureChannel, this.logIdent);
        }
    }

    public LogSegment createAndDeleteSegment(long newOffset, LogSegment segmentToDelete, boolean asyncDelete, SegmentDeletionReason reason) throws IOException {
        if (newOffset == segmentToDelete.baseOffset()) {
            segmentToDelete.changeFileSuffixes("", ".deleted");
        }
        LogSegment newSegment = LogSegment.open(this.dir, newOffset, this.config, this.time, this.config.initFileSize(), this.config.preallocate, this.checksumParams);
        this.segments.add(newSegment);
        reason.logReason(Collections.singletonList(segmentToDelete));
        if (newOffset != segmentToDelete.baseOffset()) {
            this.segments.remove(segmentToDelete.baseOffset());
        }
        LocalLog.deleteSegmentFiles(Collections.singletonList(segmentToDelete), asyncDelete, this.dir, this.topicPartition, this.config, this.scheduler, this.logDirFailureChannel, this.logIdent);
        return newSegment;
    }

    public LogOffsetMetadata convertToOffsetMetadataOrThrow(long offset) throws IOException {
        FetchDataInfo fetchDataInfo = this.read(offset, 1, false, this.nextOffsetMetadata, false);
        return fetchDataInfo.fetchOffsetMetadata;
    }

    private void maybeLoadRecordsIntoPageCache(FileRecords records) {
        try {
            this.brokerTopicStats.allTopicsStats().segmentSpeculativePrefetchRate().get().mark();
            records.loadIntoPageCache();
        }
        catch (Throwable e) {
            this.logger.warn("Failed to prepare cache for read", e);
        }
    }

    public FetchDataInfo read(long startOffset, int maxLength, boolean minOneMessage, LogOffsetMetadata maxOffsetMetadata, boolean includeAbortedTxns) throws IOException {
        return this.maybeHandleIOException(() -> "Exception while reading from " + String.valueOf(this.topicPartition) + " in dir " + this.dir.getParent(), () -> {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Reading maximum {} bytes at offset {} from log with total length {} bytes", new Object[]{maxLength, startOffset, this.segments.sizeInBytes()});
            }
            LogOffsetMetadata endOffsetMetadata = this.nextOffsetMetadata;
            long endOffset = endOffsetMetadata.messageOffset;
            Optional<LogSegment> segmentOpt = this.segments.floorSegment(startOffset);
            if (startOffset > endOffset || segmentOpt.isEmpty()) {
                throw new OffsetOutOfRangeException("Received request for offset " + startOffset + " for partition " + String.valueOf(this.topicPartition) + ", but we only have log segments upto " + endOffset + ".");
            }
            if (startOffset == maxOffsetMetadata.messageOffset) {
                return LocalLog.emptyFetchDataInfo(maxOffsetMetadata, includeAbortedTxns);
            }
            if (startOffset > maxOffsetMetadata.messageOffset) {
                return LocalLog.emptyFetchDataInfo(this.convertToOffsetMetadataOrThrow(startOffset), includeAbortedTxns);
            }
            FetchDataInfo fetchDataInfo = null;
            while (fetchDataInfo == null && segmentOpt.isPresent()) {
                LogSegment segment = segmentOpt.get();
                long baseOffset = segment.baseOffset();
                Optional<Long> maxPositionOpt = segment.baseOffset() < maxOffsetMetadata.segmentBaseOffset ? Optional.of(Long.valueOf(segment.size())) : (segment.baseOffset() == maxOffsetMetadata.segmentBaseOffset && !maxOffsetMetadata.messageOffsetOnly() ? Optional.of(Long.valueOf(maxOffsetMetadata.relativePositionInSegment)) : Optional.empty());
                fetchDataInfo = segment.read(startOffset, maxLength, maxPositionOpt, minOneMessage);
                if (fetchDataInfo != null) {
                    this.brokerTopicStats.allTopicsStats().segmentReadRate().get().mark();
                    if (segment.baseOffset() != this.segments.activeSegment().baseOffset() && this.config.confluentLogConfig().segmentSpeculativePrefetchEnable && fetchDataInfo.records instanceof FileRecords) {
                        this.maybeLoadRecordsIntoPageCache((FileRecords)fetchDataInfo.records);
                    }
                    if (!includeAbortedTxns) continue;
                    fetchDataInfo = this.addAbortedTransactions(startOffset, segment, fetchDataInfo);
                    continue;
                }
                segmentOpt = this.segments.higherSegment(baseOffset);
            }
            if (fetchDataInfo != null) {
                return fetchDataInfo;
            }
            return new FetchDataInfo(this.nextOffsetMetadata, (Records)MemoryRecords.EMPTY);
        });
    }

    public void append(long lastOffset, MemoryRecords records, long appendTimeMs) throws IOException {
        this.segments.activeSegment().append(lastOffset, records, appendTimeMs);
        this.updateLogEndOffset(lastOffset + 1L);
    }

    FetchDataInfo addAbortedTransactions(long startOffset, LogSegment segment, FetchDataInfo fetchInfo) throws IOException {
        int fetchSize = fetchInfo.records.sizeInBytes();
        OffsetPosition startOffsetPosition = new OffsetPosition(fetchInfo.fetchOffsetMetadata.messageOffset, fetchInfo.fetchOffsetMetadata.relativePositionInSegment);
        long upperBoundOffset = segment.fetchUpperBoundOffset(startOffsetPosition, fetchSize).orElse(this.segments.higherSegment(segment.baseOffset()).map(LogSegment::baseOffset).orElse(this.logEndOffset()));
        ArrayList abortedTransactions = new ArrayList();
        Consumer<List<AbortedTxn>> accumulator = abortedTxns -> {
            for (AbortedTxn abortedTxn : abortedTxns) {
                abortedTransactions.add(abortedTxn.asAbortedTransaction());
            }
        };
        this.collectAbortedTransactions(startOffset, upperBoundOffset, segment, accumulator, false);
        return new FetchDataInfo(fetchInfo.fetchOffsetMetadata, fetchInfo.records, fetchInfo.firstEntryIncomplete, Optional.of(abortedTransactions));
    }

    private void collectAbortedTransactions(long startOffset, long upperBoundOffset, LogSegment startingSegment, Consumer<List<AbortedTxn>> accumulator, boolean shouldValidateChecksum) {
        Iterator<LogSegment> higherSegments = this.segments.higherSegments(startingSegment.baseOffset()).iterator();
        Optional<LogSegment> segmentEntryOpt = Optional.of(startingSegment);
        while (segmentEntryOpt.isPresent()) {
            LogSegment segment = segmentEntryOpt.get();
            TxnIndexSearchResult searchResult = segment.collectAbortedTxns(startOffset, upperBoundOffset, shouldValidateChecksum);
            accumulator.accept(searchResult.abortedTransactions);
            if (searchResult.isComplete) {
                return;
            }
            segmentEntryOpt = LocalLog.nextItem(higherSegments);
        }
    }

    public List<AbortedTxn> collectAbortedTransactions(long logStartOffset, long baseOffset, long upperBoundOffset, boolean shouldValidateChecksum) {
        Optional<LogSegment> segmentEntry = this.segments.floorSegment(baseOffset);
        ArrayList<AbortedTxn> allAbortedTxns = new ArrayList<AbortedTxn>();
        segmentEntry.ifPresent(logSegment -> this.collectAbortedTransactions(logStartOffset, upperBoundOffset, (LogSegment)logSegment, allAbortedTxns::addAll, shouldValidateChecksum));
        return allAbortedTxns;
    }

    public LogSegment roll(Long expectedNextOffset) {
        return this.maybeHandleIOException(() -> "Error while rolling log segment for " + String.valueOf(this.topicPartition) + " in dir " + this.dir.getParent(), () -> {
            long start = this.time.hiResClockMs();
            this.checkIfMemoryMappedBufferClosed();
            long newOffset = Math.max(expectedNextOffset, this.logEndOffset());
            File logFile = LogFileUtils.logFile(this.dir, newOffset, "");
            LogSegment activeSegment = this.segments.activeSegment();
            if (this.segments.contains(newOffset)) {
                if (activeSegment.baseOffset() == newOffset && activeSegment.size() == 0) {
                    this.logger.warn("Trying to roll a new log segment with start offset {}=max(provided offset = {}, LEO = {}) while it already exists and is active with size 0. Size of time index: {}, size of offset index: {}.", new Object[]{newOffset, expectedNextOffset, this.logEndOffset(), activeSegment.timeIndex().entries(), activeSegment.offsetIndex().entries()});
                    LogSegment newSegment = this.createAndDeleteSegment(newOffset, activeSegment, true, toDelete -> this.logger.info("Deleting segments as part of log roll: {}", (Object)toDelete.stream().map(LogSegment::toString).collect(Collectors.joining(", "))));
                    this.updateLogEndOffset(this.nextOffsetMetadata.messageOffset);
                    this.logger.info("Rolled new log segment at offset {} in {} ms.", (Object)newOffset, (Object)(this.time.hiResClockMs() - start));
                    return newSegment;
                }
                throw new KafkaException("Trying to roll a new log segment for topic partition " + String.valueOf(this.topicPartition) + " with start offset " + newOffset + " =max(provided offset = " + expectedNextOffset + ", LEO = " + this.logEndOffset() + ") while it already exists. Existing segment is " + String.valueOf(this.segments.get(newOffset)) + ".");
            }
            if (!this.segments.isEmpty() && newOffset < activeSegment.baseOffset()) {
                throw new KafkaException("Trying to roll a new log segment for topic partition " + String.valueOf(this.topicPartition) + " with start offset " + newOffset + " =max(provided offset = " + expectedNextOffset + ", LEO = " + this.logEndOffset() + ") lower than start offset of the active segment " + String.valueOf(activeSegment));
            }
            if (this.segments.lastSegment().isPresent()) {
                this.segments.lastSegment().get().onBecomeInactiveSegment();
            }
            LogSegment newSegment = LogSegment.open(this.dir, newOffset, this.config, this.time, this.config.initFileSize(), this.config.preallocate, this.checksumParams);
            this.segments.add(newSegment);
            this.updateLogEndOffset(this.nextOffsetMetadata.messageOffset);
            return newSegment;
        });
    }

    public List<LogSegment> truncateFullyAndStartAt(long newOffset) {
        return this.maybeHandleIOException(() -> "Error while truncating the entire log for " + String.valueOf(this.topicPartition) + " in dir " + this.dir.getParent(), () -> {
            this.logger.debug("Truncate and start at offset {}", (Object)newOffset);
            this.checkIfMemoryMappedBufferClosed();
            ArrayList<LogSegment> segmentsToDelete = new ArrayList<LogSegment>(this.segments.values());
            if (!segmentsToDelete.isEmpty()) {
                this.removeAndDeleteSegments(segmentsToDelete.subList(0, segmentsToDelete.size() - 1), true, new LogTruncation(this.logger));
                this.createAndDeleteSegment(newOffset, (LogSegment)segmentsToDelete.get(segmentsToDelete.size() - 1), true, new LogTruncation(this.logger));
            }
            this.updateLogEndOffset(newOffset);
            return segmentsToDelete;
        });
    }

    public Collection<LogSegment> truncateTo(long targetOffset) throws IOException {
        Collection<LogSegment> deletableSegments = this.segments.filter(segment -> segment.baseOffset() > targetOffset);
        this.removeAndDeleteSegments(deletableSegments, true, new LogTruncation(this.logger));
        this.segments.activeSegment().truncateTo(targetOffset);
        this.updateLogEndOffset(targetOffset);
        return deletableSegments;
    }

    public static String logDeleteDirName(TopicPartition topicPartition) {
        return LocalLog.logDirNameWithSuffixCappedLength(topicPartition, "-delete");
    }

    public static String logStrayDirName(TopicPartition topicPartition) {
        return LocalLog.logDirNameWithSuffixCappedLength(topicPartition, "-stray");
    }

    public static String logFutureDirName(TopicPartition topicPartition) {
        return LocalLog.logDirNameWithSuffix(topicPartition, "-future");
    }

    private static String logDirNameWithSuffixCappedLength(TopicPartition topicPartition, String suffix) {
        String uniqueId = UUID.randomUUID().toString().replaceAll("-", "");
        String fullSuffix = "-" + topicPartition.partition() + "." + uniqueId + suffix;
        int prefixLength = Math.min(topicPartition.topic().length(), 255 - fullSuffix.length());
        return topicPartition.topic().substring(0, prefixLength) + fullSuffix;
    }

    private static String logDirNameWithSuffix(TopicPartition topicPartition, String suffix) {
        String uniqueId = UUID.randomUUID().toString().replaceAll("-", "");
        return LocalLog.logDirName(topicPartition) + "." + uniqueId + suffix;
    }

    public static String logDirName(TopicPartition topicPartition) {
        return topicPartition.topic() + "-" + topicPartition.partition();
    }

    private static KafkaException exception(File dir) throws IOException {
        return new KafkaException("Found directory " + dir.getCanonicalPath() + ", '" + dir.getName() + "' is not in the form of topic-partition or topic-partition.uniqueId-delete (if marked for deletion).\nKafka's log directories (and children) should only contain Kafka topic data.");
    }

    public static TopicPartition parseTopicPartitionName(File dir) throws IOException {
        if (dir == null) {
            throw new KafkaException("dir should not be null");
        }
        String dirName = dir.getName();
        if (!dirName.contains("-")) {
            throw LocalLog.exception(dir);
        }
        if (dirName.endsWith("-delete") && !DELETE_DIR_PATTERN.matcher(dirName).matches() || dirName.endsWith("-future") && !FUTURE_DIR_PATTERN.matcher(dirName).matches() || dirName.endsWith("-stray") && !STRAY_DIR_PATTERN.matcher(dirName).matches()) {
            throw LocalLog.exception(dir);
        }
        String name = dirName.endsWith("-delete") || dirName.endsWith("-future") || dirName.endsWith("-stray") ? dirName.substring(0, dirName.lastIndexOf(46)) : dirName;
        try {
            return LocalLog.parseTopicPartitionName(name);
        }
        catch (KafkaException e) {
            throw LocalLog.exception(dir);
        }
    }

    private static KafkaException exception(String topicPartitionString) {
        return new KafkaException(String.format("Input String '%s' is not in the form of topic-partition.", topicPartitionString));
    }

    public static TopicPartition parseTopicPartitionName(String topicPartitionString) throws IOException {
        try {
            int index = topicPartitionString.lastIndexOf(45);
            String topic = topicPartitionString.substring(0, index);
            String partitionString = topicPartitionString.substring(index + 1);
            if (topic.isEmpty() || partitionString.isEmpty()) {
                throw LocalLog.exception(topicPartitionString);
            }
            int partition = Integer.parseInt(partitionString);
            return new TopicPartition(topic, partition);
        }
        catch (Exception e) {
            throw LocalLog.exception(topicPartitionString);
        }
    }

    public static <T> Optional<T> nextItem(Iterator<T> iterator) {
        return iterator.hasNext() ? Optional.of(iterator.next()) : Optional.empty();
    }

    private static FetchDataInfo emptyFetchDataInfo(LogOffsetMetadata fetchOffsetMetadata, boolean includeAbortedTxns) {
        Optional<List<FetchResponseData.AbortedTransaction>> abortedTransactions = includeAbortedTxns ? Optional.of(Collections.emptyList()) : Optional.empty();
        return new FetchDataInfo(fetchOffsetMetadata, (Records)MemoryRecords.EMPTY, false, abortedTransactions);
    }

    public static <T> T maybeHandleIOException(LogDirFailureChannel logDirFailureChannel, String logDir, Supplier<String> errorMsgSupplier, StorageAction<T, IOException> function) {
        if (logDirFailureChannel.hasOfflineLogDir(logDir)) {
            throw new KafkaStorageException("The log dir " + logDir + " is already offline due to a previous IO exception.");
        }
        try {
            return function.execute();
        }
        catch (IOException ioe) {
            String errorMsg = errorMsgSupplier.get();
            logDirFailureChannel.maybeAddOfflineLogDir(logDir, errorMsg, ioe);
            throw new KafkaStorageException(errorMsg, (Throwable)ioe);
        }
    }

    public static void deleteSegmentFiles(Collection<LogSegment> segmentsToDelete, boolean asyncDelete, File dir, TopicPartition topicPartition, LogConfig config, Scheduler scheduler, LogDirFailureChannel logDirFailureChannel, String logPrefix) throws IOException {
        for (LogSegment segment : segmentsToDelete) {
            if (segment.hasSuffix(".deleted")) continue;
            segment.changeFileSuffixes("", ".deleted");
        }
        Runnable deleteSegments = () -> {
            LOG.info("{}Deleting segment files {}", (Object)logPrefix, (Object)segmentsToDelete.stream().map(LogSegment::toString).collect(Collectors.joining(", ")));
            String parentDir = dir.getParent();
            LocalLog.maybeHandleIOException(logDirFailureChannel, parentDir, () -> "Error while deleting segments for " + String.valueOf(topicPartition) + " in dir " + parentDir, () -> {
                for (LogSegment segment : segmentsToDelete) {
                    segment.deleteIfExists();
                }
                return null;
            });
        };
        if (asyncDelete) {
            scheduler.scheduleOnce("delete-file", deleteSegments, config.fileDeleteDelayMs);
        } else {
            deleteSegments.run();
        }
    }

    public static LogSegment createNewCleanedSegment(File dir, LogConfig logConfig, long baseOffset, boolean tiered, ChecksumParams checksumParams) throws IOException {
        String fileSuffix = tiered ? ".tiercleaned" : ".cleaned";
        LogSegment.deleteIfExists(dir, baseOffset, fileSuffix);
        return LogSegment.open(dir, baseOffset, logConfig, Time.SYSTEM, false, logConfig.initFileSize(), logConfig.preallocate, fileSuffix, checksumParams);
    }

    public static SplitSegmentResult splitOverflowedSegment(LogSegment segment, LogSegments existingSegments, File dir, TopicPartition topicPartition, LogConfig config, Scheduler scheduler, LogDirFailureChannel logDirFailureChannel, String logPrefix) throws IOException {
        Utils.require((boolean)LogFileUtils.isLogFile(segment.log().file()), (String)("Cannot split file " + String.valueOf(segment.log().file().getAbsoluteFile())));
        Utils.require((boolean)segment.hasOverflow(), (String)("Split operation is only permitted for segments with overflow, and the problem path is " + String.valueOf(segment.log().file().getAbsoluteFile())));
        LOG.info("{}Splitting overflowed segment {}", (Object)logPrefix, (Object)segment);
        ArrayList<LogSegment> newSegments = new ArrayList<LogSegment>();
        try {
            int bytesAppended;
            FileRecords sourceRecords = segment.log();
            for (int position = 0; position < sourceRecords.sizeInBytes(); position += bytesAppended) {
                FileLogInputStream.FileChannelRecordBatch firstBatch = (FileLogInputStream.FileChannelRecordBatch)sourceRecords.batchesFrom(position).iterator().next();
                LogSegment newSegment = LocalLog.createNewCleanedSegment(dir, config, firstBatch.baseOffset(), false, segment.checksumParams());
                newSegments.add(newSegment);
                bytesAppended = newSegment.appendFromFile(sourceRecords, position);
                if (bytesAppended != 0) continue;
                throw new IllegalStateException("Failed to append records from position " + position + " in " + String.valueOf(segment));
            }
            int totalSizeOfNewSegments = 0;
            for (LogSegment splitSegment : newSegments) {
                splitSegment.onBecomeInactiveSegment();
                splitSegment.flush();
                splitSegment.setLastModified(segment.lastModified());
                totalSizeOfNewSegments += splitSegment.log().sizeInBytes();
            }
            if (totalSizeOfNewSegments != segment.log().sizeInBytes()) {
                throw new IllegalStateException("Inconsistent segment sizes after split before: " + segment.log().sizeInBytes() + " after: " + totalSizeOfNewSegments);
            }
            LOG.info("{}Replacing overflowed segment {} with split segments {}", new Object[]{logPrefix, segment, newSegments});
            List<LogSegment> deletedSegments = LocalLog.replaceSegments(existingSegments, newSegments, Collections.singletonList(segment), dir, topicPartition, config, scheduler, logDirFailureChannel, logPrefix, false);
            return new SplitSegmentResult(deletedSegments, newSegments);
        }
        catch (Exception e) {
            for (LogSegment splitSegment : newSegments) {
                splitSegment.close();
                splitSegment.deleteIfExists();
            }
            throw e;
        }
    }

    public static List<LogSegment> replaceSegments(LogSegments existingSegments, List<LogSegment> newSegments, List<LogSegment> oldSegments, File dir, TopicPartition topicPartition, LogConfig config, Scheduler scheduler, LogDirFailureChannel logDirFailureChannel, String logPrefix, boolean isRecoveredSwapFile) throws IOException {
        ArrayList<LogSegment> sortedNewSegments = new ArrayList<LogSegment>(newSegments);
        sortedNewSegments.sort(Comparator.comparingLong(LogSegment::baseOffset));
        List sortedOldSegments = oldSegments.stream().filter(seg -> existingSegments.contains(seg.baseOffset())).sorted(Comparator.comparingLong(LogSegment::baseOffset)).collect(Collectors.toList());
        ArrayList<LogSegment> reversedSegmentsList = new ArrayList<LogSegment>(sortedNewSegments);
        Collections.reverse(reversedSegmentsList);
        for (LogSegment segment : reversedSegmentsList) {
            if (!isRecoveredSwapFile) {
                segment.changeFileSuffixes(".cleaned", ".swap");
            }
            existingSegments.add(segment);
        }
        Set newSegmentBaseOffsets = sortedNewSegments.stream().map(LogSegment::baseOffset).collect(Collectors.toSet());
        ArrayList<LogSegment> deletedNotReplaced = new ArrayList<LogSegment>();
        for (LogSegment segment : sortedOldSegments) {
            if (segment.baseOffset() != ((LogSegment)sortedNewSegments.get(0)).baseOffset()) {
                existingSegments.remove(segment.baseOffset());
            }
            LocalLog.deleteSegmentFiles(Collections.singletonList(segment), true, dir, topicPartition, config, scheduler, logDirFailureChannel, logPrefix);
            if (newSegmentBaseOffsets.contains(segment.baseOffset())) continue;
            deletedNotReplaced.add(segment);
        }
        for (LogSegment logSegment : sortedNewSegments) {
            logSegment.changeFileSuffixes(".swap", "");
        }
        Utils.flushDir((Path)dir.toPath());
        return deletedNotReplaced;
    }

    public static class SplitSegmentResult {
        public final List<LogSegment> deletedSegments;
        public final List<LogSegment> newSegments;

        public SplitSegmentResult(List<LogSegment> deletedSegments, List<LogSegment> newSegments) {
            this.deletedSegments = deletedSegments;
            this.newSegments = newSegments;
        }
    }
}

