/*
 * Decompiled with CFR 0.152.
 */
package kafka.restore.schedulers;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.confluent.rest.TierRecordMetadataResponse;
import java.io.Closeable;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import kafka.restore.RestoreMetricsManager;
import kafka.restore.configmap.NodeConfig;
import kafka.restore.messages.KafkaFenceRequest;
import kafka.restore.messages.KafkaFenceResponse;
import kafka.restore.messages.KafkaFetchFtpsRequest;
import kafka.restore.messages.KafkaFetchFtpsResponse;
import kafka.restore.messages.KafkaForceRestoreRequest;
import kafka.restore.messages.KafkaForceRestoreResponse;
import kafka.restore.messages.KafkaPreConditionCheckRequest;
import kafka.restore.messages.KafkaPreConditionCheckResponse;
import kafka.restore.messages.KafkaRequest;
import kafka.restore.messages.KafkaTierPartitionStatusRequest;
import kafka.restore.messages.KafkaTierPartitionStatusResponse;
import kafka.restore.messages.KafkaUnfreezeRequest;
import kafka.restore.messages.KafkaUnfreezeResponse;
import kafka.restore.messages.KafkaValidateLogRangeRequest;
import kafka.restore.messages.KafkaValidateLogRangeResponse;
import kafka.restore.messages.MessageResponse;
import kafka.restore.messages.MessageResult;
import kafka.restore.messages.MessageStatusCode;
import kafka.restore.schedulers.AsyncServiceSchedulerResultsReceiver;
import kafka.restore.schedulers.CompletableFutureRetryer;
import kafka.restore.schedulers.Constants;
import kafka.restore.schedulers.FtpsFetcher;
import kafka.restore.schedulers.KafkaConnectionPool;
import kafka.restore.schedulers.PartitionLogRangeValidator;
import kafka.restore.schedulers.PartitionStatusFetcher;
import kafka.restore.schedulers.RetryableException;
import kafka.restore.schedulers.SchedulerUtil;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.util.EntityUtils;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaConnectionPoolImpl
implements KafkaConnectionPool {
    private static final Logger LOGGER = LoggerFactory.getLogger(KafkaConnectionPoolImpl.class);
    private final AsyncServiceSchedulerResultsReceiver resultsReceiver;
    private final int poolSize;
    private String ftpsDirPath;
    private ThreadPoolExecutor threadPool;
    private PoolingHttpClientConnectionManager connectionManager;
    private CloseableHttpClient httpClient;
    private KafkaConnectionPoolState state;
    private int nextUuid = 0;
    private Duration statusQueryRetryWaitInMs = Constants.DEFAULT_WAIT_BETWEEN_IN_MS;
    private CompletableFutureRetryer retries;
    private final RestoreMetricsManager metrics;
    private final Time time;
    private final ObjectMapper mapper = new ObjectMapper();

    public void setStatusQueryRetryWaitInMs(long statusQueryRetryWaitInMs) {
        this.statusQueryRetryWaitInMs = Duration.ofMillis(statusQueryRetryWaitInMs);
    }

    public void setHttpClient(CloseableHttpClient httpClient) {
        this.httpClient = httpClient;
    }

    private int getNextUuid() {
        int uuid = this.nextUuid++;
        return uuid;
    }

    public KafkaConnectionPoolImpl(AsyncServiceSchedulerResultsReceiver resultsReceiver, int poolSize, String ftpsDirPath, RestoreMetricsManager metrics, Time time) {
        if (poolSize < 1) {
            throw new IllegalArgumentException("KafkaConnectionPool must have a pool size of at least one");
        }
        this.resultsReceiver = resultsReceiver;
        this.poolSize = poolSize;
        this.state = KafkaConnectionPoolState.OFF;
        this.ftpsDirPath = ftpsDirPath;
        this.metrics = metrics;
        this.time = time;
        Path path = Paths.get(ftpsDirPath, new String[0]);
        if (!Files.exists(path, new LinkOption[0])) {
            try {
                Files.createDirectory(path, new FileAttribute[0]);
            }
            catch (IOException e) {
                LOGGER.warn(ftpsDirPath + "is not exist, create it also failed, set ftpsDirPath to /tmp");
                this.ftpsDirPath = "/tmp";
            }
        }
    }

    @Override
    public void startUp() {
        if (this.state == KafkaConnectionPoolState.RUNNING) {
            return;
        }
        this.state = KafkaConnectionPoolState.RUNNING;
        this.threadPool = new ThreadPoolExecutor(this.poolSize, this.poolSize, 1L, TimeUnit.MINUTES, new SynchronousQueue<Runnable>(), new ThreadPoolExecutor.CallerRunsPolicy());
        this.retries = new CompletableFutureRetryer(this.threadPool, this.statusQueryRetryWaitInMs, this.time);
        if (this.httpClient == null) {
            this.connectionManager = new PoolingHttpClientConnectionManager();
            this.connectionManager.setMaxTotal(this.poolSize);
            this.httpClient = SchedulerUtil.buildHttpClient(this.connectionManager);
        }
    }

    @Override
    public void shutdown() {
        if (this.state == KafkaConnectionPoolState.OFF) {
            return;
        }
        if (this.threadPool != null) {
            this.threadPool.shutdown();
        }
        try {
            if (this.httpClient != null) {
                this.httpClient.close();
            }
        }
        catch (IOException e) {
            LOGGER.error("Received IOException while closing HTTP Client during shutdown of KafkaConnectionPool.", e);
        }
        if (this.connectionManager != null) {
            this.connectionManager.shutdown();
        }
    }

    @Override
    public void submitKafkaRequest(KafkaRequest kafkaRequest) {
        if (this.state != KafkaConnectionPoolState.RUNNING) {
            throw new IllegalStateException("Cannot submit request to non-running connection pool.");
        }
        if (kafkaRequest instanceof KafkaFetchFtpsRequest) {
            this.submitFetchFtpsRequest((KafkaFetchFtpsRequest)kafkaRequest);
        } else if (kafkaRequest instanceof KafkaFenceRequest || kafkaRequest instanceof KafkaUnfreezeRequest || kafkaRequest instanceof KafkaForceRestoreRequest || kafkaRequest instanceof KafkaPreConditionCheckRequest) {
            this.submitKafkaEventRequest(kafkaRequest);
        } else if (kafkaRequest instanceof KafkaTierPartitionStatusRequest) {
            this.submitTierPartitionStatusRequest((KafkaTierPartitionStatusRequest)kafkaRequest);
        } else if (kafkaRequest instanceof KafkaValidateLogRangeRequest) {
            this.submitKafkaValidateLogRangeRequest((KafkaValidateLogRangeRequest)kafkaRequest);
        } else {
            throw new UnsupportedOperationException("kafkaRequest of type " + String.valueOf(kafkaRequest.getClass()) + " is not of a recognizable request type.");
        }
    }

    protected void submitFetchFtpsRequest(KafkaFetchFtpsRequest request) {
        this.threadPool.execute(() -> {
            FtpsFetcher ftpsFetcher = new FtpsFetcher(this.httpClient, request, this.ftpsDirPath, this.threadPool);
            CompletableFuture ftpsDownloadResponse = this.retries.withTimedRetries(ftpsFetcher::fetchFtpsFile, t -> t.getClass().equals(RetryableException.class), 20, this.metrics != null ? this.metrics.restoreFetchFtpsMs()::record : null, this.metrics != null ? this.metrics::recordRestoreFetchFtpsFailures : null);
            ftpsDownloadResponse.thenCompose(response -> CompletableFuture.runAsync(() -> this.resultsReceiver.reportServiceSchedulerResponse((MessageResponse)response), this.threadPool));
            ftpsDownloadResponse.exceptionally(throwable -> {
                LOGGER.error(String.format("[%s]: fetchFtpsFile call completed with Exception", request.getTopicPartition()), (Throwable)throwable);
                this.reportFetchFtpsResponse(request, MessageStatusCode.EXTERNAL_SERVICE_ERROR, MessageResult.FAILURE);
                return null;
            });
        });
    }

    protected void submitKafkaEventRequest(KafkaRequest request) {
        this.threadPool.execute(() -> {
            Closeable httpResponse = null;
            HttpPost httpPostRequest = null;
            httpPostRequest = SchedulerUtil.buildKafkaHttpRequest(request);
            LOGGER.debug("Send kafka request: " + httpPostRequest.getURI().toString());
            httpResponse = this.httpClient.execute(httpPostRequest);
            if (httpResponse.getStatusLine().getStatusCode() == 200) {
                if (request instanceof KafkaFenceRequest) {
                    this.reportKafkaFenceResponse(request, this.extractTierRecordMetadataResponse((CloseableHttpResponse)httpResponse), MessageStatusCode.OK, MessageResult.SUCCESS);
                } else {
                    this.reportKafkaResponse(request, MessageStatusCode.OK, MessageResult.SUCCESS);
                }
            } else {
                LOGGER.error(String.format("[%s]: http response error status: %s, http response body: %s.", request.getTopicPartition(), httpResponse.getStatusLine().getStatusCode(), httpResponse.getEntity().toString()));
                this.reportKafkaResponse(request, MessageStatusCode.SERVICE_CONNECTION_ERROR, MessageResult.FAILURE);
            }
            if (httpResponse == null) return;
            try {
                httpResponse.close();
                return;
            }
            catch (IOException e) {
                LOGGER.error(String.format("[%s]: Received IOException while closing HTTP response after replying to KafkaTierPartitionEventRequest", request.getTopicPartition()), e);
            }
            return;
            catch (IOException e) {
                LOGGER.error(String.format("[%s]: Received Exception while sending KafkaEventRequest %s", request.getTopicPartition(), httpPostRequest), e);
                this.reportKafkaResponse(request, MessageStatusCode.EXTERNAL_SERVICE_ERROR, MessageResult.FAILURE);
                if (httpResponse == null) return;
                try {
                    httpResponse.close();
                    return;
                }
                catch (IOException e2) {
                    LOGGER.error(String.format("[%s]: Received IOException while closing HTTP response after replying to KafkaTierPartitionEventRequest", request.getTopicPartition()), e2);
                }
                return;
            }
            catch (URISyntaxException e2) {
                LOGGER.error("[{}]: TierPartitionEvent request parameters for request with UUID {}, resulted in invalid URI.", request.getTopicPartition(), request.getUuid(), e2);
                this.reportKafkaResponse(request, MessageStatusCode.BAD_ARGUMENT_ERROR, MessageResult.FAILURE);
                if (httpResponse == null) return;
                {
                    catch (Throwable throwable) {
                        if (httpResponse == null) throw throwable;
                        try {
                            httpResponse.close();
                            throw throwable;
                        }
                        catch (IOException e3) {
                            LOGGER.error(String.format("[%s]: Received IOException while closing HTTP response after replying to KafkaTierPartitionEventRequest", request.getTopicPartition()), e3);
                        }
                        throw throwable;
                    }
                }
                try {
                    httpResponse.close();
                    return;
                }
                catch (IOException e4) {
                    LOGGER.error(String.format("[%s]: Received IOException while closing HTTP response after replying to KafkaTierPartitionEventRequest", request.getTopicPartition()), e4);
                }
                return;
            }
        });
    }

    protected void submitTierPartitionStatusRequest(KafkaTierPartitionStatusRequest request) {
        this.threadPool.execute(() -> {
            List<NodeConfig> brokers = request.getReplicaBrokerIDs();
            long count = brokers.stream().map(brokerId -> {
                PartitionStatusFetcher fetcher = new PartitionStatusFetcher(this.httpClient, request.getTopic(), request.getPartition(), (NodeConfig)brokerId, request.getExpectedStatus(), this.threadPool);
                return this.retries.withRetries(fetcher::fetchPartitionStatus, t -> t.getClass().equals(RetryableException.class), 20);
            }).map(completableFuture -> {
                try {
                    return (Integer)completableFuture.join();
                }
                catch (CompletionException ex) {
                    return -1;
                }
            }).filter(status -> status.intValue() == request.getExpectedStatus()).count();
            if (count == (long)brokers.size()) {
                LOGGER.debug("report success");
                this.reportKafkaResponse(request, MessageStatusCode.OK, MessageResult.SUCCESS);
            } else {
                LOGGER.error(String.format("[%s]: Check partition status failed, %s brokers status as %s but expecting %s", request.getTopicPartition(), count, request.getExpectedStatus(), brokers.size()));
                this.reportKafkaResponse(request, MessageStatusCode.EXTERNAL_SERVICE_ERROR, MessageResult.FAILURE);
            }
        });
    }

    protected void submitKafkaValidateLogRangeRequest(KafkaValidateLogRangeRequest request) {
        this.threadPool.execute(() -> {
            List<NodeConfig> brokers = request.getReplicaBrokers();
            long count = brokers.stream().map(broker -> {
                PartitionLogRangeValidator logRangeValidator = new PartitionLogRangeValidator(this.httpClient, request.getTopic(), request.getPartition(), (NodeConfig)broker, request.getLogStartOffset(), request.getLogEndOffset(), this.threadPool);
                return this.retries.withRetries(logRangeValidator::validateLogRange, t -> t.getClass().equals(RetryableException.class), 20);
            }).map(completableFuture -> {
                try {
                    return (Integer)completableFuture.join();
                }
                catch (CompletionException ex) {
                    return -1;
                }
            }).filter(status -> status == 1).count();
            if (count == (long)brokers.size()) {
                LOGGER.debug("report success");
                this.reportKafkaResponse(request, MessageStatusCode.OK, MessageResult.SUCCESS);
            } else {
                LOGGER.error(String.format("[%s]: Validate log range failed, %s brokers log range match as (%s-%s) but expecting %s.", request.getTopicPartition(), count, request.getLogStartOffset(), request.getLogEndOffset(), brokers.size()));
                this.reportKafkaResponse(request, MessageStatusCode.EXTERNAL_SERVICE_ERROR, MessageResult.FAILURE);
            }
        });
    }

    protected void reportFetchFtpsResponse(KafkaFetchFtpsRequest request, MessageStatusCode statusCode, MessageResult result, String ftpsFileName) {
        KafkaFetchFtpsResponse response = new KafkaFetchFtpsResponse(this.getNextUuid(), request.getTopic(), request.getPartition(), request.getUuid(), statusCode, result, ftpsFileName);
        this.resultsReceiver.reportServiceSchedulerResponse(response);
    }

    protected void reportFetchFtpsResponse(KafkaFetchFtpsRequest request, MessageStatusCode statusCode, MessageResult result) {
        this.reportFetchFtpsResponse(request, statusCode, result, null);
    }

    protected void reportKafkaResponse(KafkaRequest request, MessageStatusCode statusCode, MessageResult result) {
        if (request instanceof KafkaFenceRequest) {
            this.reportKafkaFenceResponse(request, null, statusCode, result);
        } else if (request instanceof KafkaForceRestoreRequest) {
            this.reportKafkaForceRestoreResponse(request, statusCode, result);
        } else if (request instanceof KafkaUnfreezeRequest) {
            this.reportKafkaUnfreezeResponse(request, statusCode, result);
        } else if (request instanceof KafkaValidateLogRangeRequest) {
            this.reportKafkaValidateLogRangeResponse(request, statusCode, result);
        } else if (request instanceof KafkaTierPartitionStatusRequest) {
            this.reportTierPartitionStatusResponse(request, statusCode, result);
        } else if (request instanceof KafkaPreConditionCheckRequest) {
            this.reportKafkaPreConditionCheckResponse(request, statusCode, result);
        } else {
            LOGGER.warn("Unknown kafka request type: " + request.toString());
        }
    }

    private void reportKafkaFenceResponse(KafkaRequest request, TierRecordMetadataResponse tierRecordMetadataResponse, MessageStatusCode statusCode, MessageResult result) {
        KafkaFenceResponse response = new KafkaFenceResponse(this.getNextUuid(), request.getTopic(), request.getPartition(), request.getUuid(), tierRecordMetadataResponse, statusCode, result);
        this.resultsReceiver.reportServiceSchedulerResponse(response);
    }

    private void reportKafkaForceRestoreResponse(KafkaRequest request, MessageStatusCode statusCode, MessageResult result) {
        KafkaForceRestoreResponse response = new KafkaForceRestoreResponse(this.getNextUuid(), request.getTopic(), request.getPartition(), request.getUuid(), statusCode, result);
        this.resultsReceiver.reportServiceSchedulerResponse(response);
    }

    private void reportKafkaUnfreezeResponse(KafkaRequest request, MessageStatusCode statusCode, MessageResult result) {
        KafkaUnfreezeResponse response = new KafkaUnfreezeResponse(this.getNextUuid(), request.getTopic(), request.getPartition(), request.getUuid(), statusCode, result);
        this.resultsReceiver.reportServiceSchedulerResponse(response);
    }

    protected void reportKafkaValidateLogRangeResponse(KafkaRequest request, MessageStatusCode statusCode, MessageResult result) {
        KafkaValidateLogRangeResponse response = new KafkaValidateLogRangeResponse(this.getNextUuid(), request.getTopic(), request.getPartition(), request.getUuid(), statusCode, result);
        this.resultsReceiver.reportServiceSchedulerResponse(response);
    }

    protected void reportTierPartitionStatusResponse(KafkaRequest request, MessageStatusCode statusCode, MessageResult result) {
        KafkaTierPartitionStatusResponse response = new KafkaTierPartitionStatusResponse(this.getNextUuid(), request.getTopic(), request.getPartition(), request.getUuid(), statusCode, result);
        this.resultsReceiver.reportServiceSchedulerResponse(response);
    }

    protected void reportKafkaPreConditionCheckResponse(KafkaRequest request, MessageStatusCode statusCode, MessageResult result) {
        KafkaPreConditionCheckResponse response = new KafkaPreConditionCheckResponse(this.getNextUuid(), request.getTopic(), request.getPartition(), request.getUuid(), statusCode, result);
        this.resultsReceiver.reportServiceSchedulerResponse(response);
    }

    private TierRecordMetadataResponse extractTierRecordMetadataResponse(CloseableHttpResponse response) throws IOException {
        String responseBody = EntityUtils.toString(response.getEntity());
        JsonNode attributeNode = this.mapper.readTree(responseBody).get("data").get("attributes");
        String topicName = attributeNode.get("user_topic_partition").asText();
        long offset = attributeNode.get("offset").asLong();
        long timestamp = attributeNode.get("timestamp").asLong();
        int tierPartition = attributeNode.get("tier_partition").asInt();
        return new TierRecordMetadataResponse(topicName, timestamp, offset, tierPartition);
    }

    private static enum KafkaConnectionPoolState {
        RUNNING,
        OFF;

    }
}

