/*
 * Decompiled with CFR 0.152.
 */
package kafka.server.resource;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import kafka.metrics.LinuxDiskMetricsCollector;
import kafka.server.BrokerReconfigurable;
import kafka.server.KafkaConfig;
import kafka.server.resource.ResourceUsageListener;
import kafka.server.resource.ResourceUsageManager;
import kafka.server.resource.TierArchiverDiskThroughputListener;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.utils.KafkaThread;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.Tuple2;
import scala.collection.Set;
import scala.jdk.javaapi.CollectionConverters;

public class DiskIOManager
implements ResourceUsageManager,
Runnable,
BrokerReconfigurable {
    private static final Logger LOGGER = LoggerFactory.getLogger(DiskIOManager.class);
    private static final long CHECK_INTERVAL_MS = 5000L;
    public static final long INVALID_THROUGHPUT_THRESHOLD = 1000000000L;
    private static final int HEADROOM_MULTIPLIER_TO_EXIT_THROTTLED_MODE = 3;
    private volatile long limit;
    private volatile long headroom;
    private volatile double rate = 0.0;
    private volatile boolean isThrottled = false;
    private volatile long diskThroughputQuotaForTierArchive;
    private volatile long diskThroughputQuotaForTierArchiveThrottled;
    private final Thread thread = new KafkaThread("DiskIOManagerThread", (Runnable)this, false);
    private volatile boolean shutdown = false;
    private final LinuxDiskMetricsCollector linuxDiskMetricsCollector;
    private final int sectorSizeInBytes;
    private final String deviceName;
    private final List<ResourceUsageListener> listeners;
    private final Metrics metrics;
    public static final MetricName DISK_THROUGHPUT_BYTE_PER_SEC = new MetricName("DiskThroughputBytePerSec", "DiskIOManager", "Disk read/write throughput (bytes/sec)", new HashMap<String, String>());
    public static final MetricName DISK_THROUGHPUT_LIMIT_BYTE_PER_SEC = new MetricName("DiskThroughputLimitBytePerSec", "DiskIOManager", "Disk read/write throughput limit (bytes/sec)", new HashMap<String, String>());
    public static Set<String> reconfigurableConfigs = CollectionConverters.asScala(new HashSet<String>(Arrays.asList(KafkaConfig.DiskThroughputLimitBytePerSecProp(), KafkaConfig.DiskThroughputHeadroomBytePerSecProp(), KafkaConfig.DiskThroughputQuotaForTierArchiveBytePerSecProp(), KafkaConfig.DiskThroughputThrottledQuotaForTierArchiveBytePerSecProp())));

    DiskIOManager(Metrics metrics, KafkaConfig config, LinuxDiskMetricsCollector linuxDiskMetricsCollector, int sectorSizeInBytes, String deviceName) {
        this.metrics = metrics;
        this.limit = config.confluentConfig().diskThroughputLimitBytePerSec();
        this.headroom = config.confluentConfig().diskThroughputHeadroomBytePerSec();
        this.diskThroughputQuotaForTierArchive = config.confluentConfig().diskThroughputQuotaTierArchiveBytePerSec();
        this.diskThroughputQuotaForTierArchiveThrottled = config.confluentConfig().diskThroughputThrottledQuotaTierArchiveBytePerSec();
        this.linuxDiskMetricsCollector = linuxDiskMetricsCollector;
        this.sectorSizeInBytes = sectorSizeInBytes;
        this.deviceName = deviceName;
        this.listeners = new LinkedList<ResourceUsageListener>();
        LOGGER.info("DiskIOManager initialized successfully - disk throughput limit={} bytes/sec, headroom={} bytes/sec, sectorSize={} bytes, deviceName={}", this.limit, this.headroom, sectorSizeInBytes, deviceName);
    }

    public static Optional<DiskIOManager> maybeInitDiskIOManager(Metrics metrics, KafkaConfig config, Optional<LinuxDiskMetricsCollector> linuxDiskMetricsCollectorOpt) {
        if (!config.confluentConfig().diskIOManagerEnable().booleanValue()) {
            LOGGER.info("Skip DiskIOManager init: {} = false", (Object)KafkaConfig.DiskIOManagerEnableProp());
            return Optional.empty();
        }
        try {
            DiskIOManager.validateConfig(config);
        }
        catch (ConfigException e) {
            LOGGER.error("Skip DiskIOManager init: invalid configuration", e);
            return Optional.empty();
        }
        if (!linuxDiskMetricsCollectorOpt.isPresent()) {
            LOGGER.warn("Skip DiskIOManager init: fail to get LinuxDiskMetricsCollector instance");
            return Optional.empty();
        }
        if (linuxDiskMetricsCollectorOpt.get().devices().size() != 1) {
            LOGGER.warn("Skip DiskIOManager init: DiskIOManager currently doesn't support multiple disks");
            return Optional.empty();
        }
        String deviceName = (String)((Tuple2)linuxDiskMetricsCollectorOpt.get().devices().head())._1();
        int sectorSize = DiskIOManager.readSectorSize(deviceName);
        if (sectorSize < 0) {
            LOGGER.warn("Skip DiskIOManager init: failed to get sector size from hw_sector_size");
            return Optional.empty();
        }
        return Optional.of(new DiskIOManager(metrics, config, linuxDiskMetricsCollectorOpt.get(), sectorSize, deviceName));
    }

    private static int readSectorSize(String deviceName) {
        Path sectorSizeFilePath = Paths.get("/sys", "block", deviceName, "queue", "hw_sector_size");
        try {
            if (!sectorSizeFilePath.toFile().exists()) {
                return -1;
            }
            List<String> lines = Files.readAllLines(sectorSizeFilePath);
            if (lines.size() != 1) {
                return -1;
            }
            return Integer.parseInt(lines.get(0).trim());
        }
        catch (Exception e) {
            LOGGER.error("Failed to get sector size from {}", (Object)sectorSizeFilePath);
            return -1;
        }
    }

    public void startup() {
        if (this.shutdown) {
            return;
        }
        this.thread.start();
        this.setupMetrics();
    }

    @Override
    public void run() {
        block3: {
            try {
                while (!this.shutdown) {
                    this.updateUsage();
                    this.applyQuota();
                    Thread.sleep(5000L);
                }
            }
            catch (Exception e) {
                if (this.shutdown) break block3;
                LOGGER.error("Fatal exception in DiskIOManager", e);
            }
        }
    }

    public void shutdown() {
        this.shutdown = true;
        try {
            this.thread.join();
        }
        catch (InterruptedException e) {
            LOGGER.error("DiskIOManager shutdown interrupted", e);
        }
        this.removeMetrics();
    }

    @Override
    public void updateUsage() {
        double sectorsReadPerSec = this.linuxDiskMetricsCollector.metricRate(this.deviceName, 2);
        double sectorsWrittenPerSec = this.linuxDiskMetricsCollector.metricRate(this.deviceName, 6);
        if (sectorsReadPerSec < 0.0 || sectorsWrittenPerSec < 0.0) {
            return;
        }
        this.rate = (sectorsWrittenPerSec + sectorsReadPerSec) * (double)this.sectorSizeInBytes;
    }

    @Override
    public void applyQuota() {
        long limit = this.limit;
        long headroom = this.headroom;
        double rate = this.rate;
        if (!this.isThrottled) {
            if (rate > (double)(limit - headroom)) {
                LOGGER.info("DiskIOManager entering throttled mode");
                this.applyQuota(true);
                this.isThrottled = true;
            }
        } else if (rate < (double)(limit - headroom * 3L)) {
            LOGGER.info("DiskIOManager exiting throttled mode");
            this.applyQuota(false);
            this.isThrottled = false;
        }
    }

    private void applyQuota(boolean setThrottledQuota) {
        for (ResourceUsageListener listener : this.listeners) {
            if (!(listener instanceof TierArchiverDiskThroughputListener)) continue;
            listener.setQuota(setThrottledQuota ? (double)this.diskThroughputQuotaForTierArchiveThrottled : (double)this.diskThroughputQuotaForTierArchive);
        }
    }

    @Override
    public void registerListener(ResourceUsageListener listener) {
        this.listeners.add(listener);
    }

    public boolean isThrottled() {
        return this.isThrottled;
    }

    private void setupMetrics() {
        this.metrics.addMetric(DISK_THROUGHPUT_BYTE_PER_SEC, (mConfig, currentTimeMs) -> this.rate);
        this.metrics.addMetric(DISK_THROUGHPUT_LIMIT_BYTE_PER_SEC, (mConfig, currentTimeMs) -> {
            long limit = this.limit;
            if (limit >= 1000000000L) {
                return 0.0;
            }
            return limit;
        });
    }

    private void removeMetrics() {
        this.metrics.removeMetric(DISK_THROUGHPUT_BYTE_PER_SEC);
        this.metrics.removeMetric(DISK_THROUGHPUT_LIMIT_BYTE_PER_SEC);
    }

    @Override
    public Set<String> reconfigurableConfigs() {
        return reconfigurableConfigs;
    }

    @Override
    public void validateReconfiguration(KafkaConfig newConfig) {
        DiskIOManager.validateConfig(newConfig);
    }

    private static void validateConfig(KafkaConfig config) {
        long limit = config.confluentConfig().diskThroughputLimitBytePerSec();
        long headroom = config.confluentConfig().diskThroughputHeadroomBytePerSec();
        long diskThroughputQuotaForTierArchive = config.confluentConfig().diskThroughputQuotaTierArchiveBytePerSec();
        long diskThroughputQuotaForTierArchiveThrottled = config.confluentConfig().diskThroughputThrottledQuotaTierArchiveBytePerSec();
        if (limit < 0L || headroom < 0L || headroom * 3L >= limit || diskThroughputQuotaForTierArchiveThrottled < 0L || diskThroughputQuotaForTierArchive < 0L) {
            throw new ConfigException(String.format("Invalid DiskIOManager configuration, limit: %d, headroom: %d, tier archiver quota: %d (non-throttled mode), %d (throttled mode)", limit, headroom, diskThroughputQuotaForTierArchive, diskThroughputQuotaForTierArchiveThrottled));
        }
    }

    @Override
    public void reconfigure(KafkaConfig oldConfig, KafkaConfig newConfig) {
        this.limit = newConfig.confluentConfig().diskThroughputLimitBytePerSec();
        this.headroom = newConfig.confluentConfig().diskThroughputHeadroomBytePerSec();
        this.diskThroughputQuotaForTierArchive = newConfig.confluentConfig().diskThroughputQuotaTierArchiveBytePerSec();
        this.diskThroughputQuotaForTierArchiveThrottled = newConfig.confluentConfig().diskThroughputThrottledQuotaTierArchiveBytePerSec();
    }
}

