/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.backupObjectLifecycle;

import io.confluent.kafka.backupRestore.objectLifecycle.serdes.LifecycleManagerState;
import io.confluent.kafka.backupRestore.objectLifecycle.serdes.TopicRetentionData;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import kafka.server.InternalAdmin;
import kafka.tier.TopicIdPartition;
import kafka.tier.backupObjectLifecycle.LifecycleManagerConfig;
import kafka.tier.backupObjectLifecycle.LifecycleManagerMetrics;
import kafka.tier.backupObjectLifecycle.NameAndId;
import kafka.tier.backupObjectLifecycle.ObjectStoreUtils;
import kafka.tier.backupObjectLifecycle.ObjectStoreUtilsContext;
import kafka.tier.backupObjectLifecycle.RetryPolicy;
import kafka.tier.backupObjectLifecycle.StateManager;
import kafka.tier.backupObjectLifecycle.StateManagerConfig;
import kafka.tier.backupObjectLifecycle.TierTopicReader;
import kafka.tier.backupObjectLifecycle.TierTopicReaderConfig;
import kafka.tier.exceptions.WrappedInterruptedException;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.VersionInformation;
import kafka.tier.store.objects.FragmentType;
import kafka.tier.store.objects.ObjectType;
import kafka.tier.store.objects.metadata.BackupObjectsListMetadata;
import kafka.tier.store.objects.metadata.TierTopicSnapshotMetadata;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.internals.Topic;
import org.apache.kafka.common.message.DescribeConfigsResponseData;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.util.ShutdownableThread;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.collection.immutable.List;

public class LifecycleManager
extends ShutdownableThread {
    private final TierObjectStore tierObjectStore;
    private final LifecycleManagerConfig config;
    private final Supplier<Boolean> isTierTopicAvailable;
    private final Supplier<Boolean> canCLMRun;
    private final Time time;
    private final LifecycleManagerMetrics lifecycleManagerMetrics;
    private long[] endOffsets;
    private long lastCLMRuntimeInMs = 0L;
    private DeletionCounters deletionCounters;
    private int maxBackupForSegmentsFromLiveTopics;
    private ThreadPoolExecutor executor;
    private ThreadPoolExecutor listObjectExecutor;
    private static final Logger log = LoggerFactory.getLogger(LifecycleManager.class);
    public static final int DEFAULT_CLM_RUN_FREQUENCY_IN_HOURS = 6;
    static final int DEFAULT_RETRIES_FOR_TIER_TOPIC_MANAGER = 20;
    static final int MAX_OBJECTS_IN_BACKUP_LIST = 10000;
    static final long MAX_DELETE_RECORDS_TO_PROCESS_PER_ITERATION = 10000L;
    static final long DEFAULT_CLM_THREAD_BACKOFF_INTERVAL_MS = 2000L;
    static final int DEFAULT_DELETION_BATCH_SIZE_PER_ITERATION = 1000;
    static final int DEFAULT_TOPIC_CONFIG_BATCH_SIZE_PER_ITERATION = 500;
    static final long ONE_DAY_IN_MS = 86400000L;
    public static final String DATE_PATTERN = "yyyyMMdd";
    private static final int MAX_RETRIES_FOR_CONSUMER_APIS = 10;
    private static final int MAX_RETRIES_FOR_CONFIG_REQUESTS = 10;
    private static final int MAX_RETRIES_OBJECT_STORE_CALLS = 10;
    private static final long RETRY_BACKOFF_MAX_MS = 30000L;
    private static final long RETRY_BACKOFF_MIN_MS = 2000L;
    static final RetryPolicy DEFAULT_RETRY_POLICY = new RetryPolicy(10, 30000L, 2000L);

    public LifecycleManager(TierObjectStore tierObjectStore, LifecycleManagerConfig config, Supplier<Boolean> isTierTopicAvailable, Supplier<Boolean> canCLMRun, Time time, Metrics metrics) {
        super("CustomLifecycleManager", true);
        this.tierObjectStore = tierObjectStore;
        this.config = config;
        this.isTierTopicAvailable = isTierTopicAvailable;
        this.canCLMRun = canCLMRun;
        this.time = time;
        int clmRunsPerDay = 24 / (config.customLifecycleManagerFrequencyInMs.get() / 1000 / 60 / 60);
        this.lifecycleManagerMetrics = new LifecycleManagerMetrics(metrics, clmRunsPerDay);
        this.endOffsets = new long[this.config.tierMetadataNumPartitions.shortValue()];
        this.deletionCounters = new DeletionCounters();
    }

    @Override
    public void run() {
        try {
            this.waitForTierTopicToBeAvailable();
        }
        catch (Exception e) {
            log.info("CustomLifecycleManager got an exception " + e.getMessage());
        }
        super.run();
    }

    @Override
    public void shutdown() throws InterruptedException {
        this.lifecycleManagerMetrics.removeMetrics();
        super.shutdown();
    }

    @Override
    public void doWork() {
        block5: {
            try {
                if (this.config.customLifecycleManagerEnabled.get().booleanValue()) {
                    log.debug("CustomLifecycleManager sleep for " + String.valueOf(this.config.minDelayInMs.get()) + " ms before attempting next run. Last run was at " + String.valueOf(new Date(this.lastCLMRuntimeInMs)));
                }
                Thread.sleep(this.config.minDelayInMs.get());
                this.initializeMetrics();
                this.canCLMRunElseThrow();
                if (this.time.milliseconds() - this.lastCLMRuntimeInMs > (long)this.config.customLifecycleManagerFrequencyInMs.get().intValue()) {
                    this.lastCLMRuntimeInMs = this.time.milliseconds();
                    this.manageLifecycleForBackedUpSegments();
                    this.manageLifecycleForTierTopicSnapshots();
                }
            }
            catch (InterruptedException e) {
                log.info("CustomLifecycleManager was interrupted. Is Shutdown initiated: " + this.isShutdownInitiated() + " Exception: " + e.getMessage());
            }
            catch (Exception e) {
                if (!this.config.customLifecycleManagerEnabled.get().booleanValue()) break block5;
                log.info("CustomLifecycleManager got an exception " + e.getMessage());
            }
        }
    }

    public void manageLifecycleForTierTopicSnapshots() {
        long timeMs = this.time.milliseconds();
        long retentionMs = TimeUnit.HOURS.toMillis(this.config.tierTopicSnapshotRetentionHours.get().intValue());
        long retentionCutoffTimeMs = timeMs - retentionMs;
        log.info("LifecycleManager tier topic snapshot deletion retentionCutoffTimeMs: " + retentionCutoffTimeMs + " timeMs: " + timeMs + " retentionMs: " + retentionMs);
        String snapshotPrefix = TierTopicSnapshotMetadata.pathPrefix("");
        Map<String, java.util.List<VersionInformation>> snapshots = this.tierObjectStore.listObject(snapshotPrefix, false);
        java.util.List<TierObjectStore.KeyAndVersion> snapshotsToDelete = snapshots.keySet().stream().map(TierTopicSnapshotMetadata::fromPath).filter(snapshotMetadata -> snapshotMetadata.snapshotObject().endTimestampMs() < retentionCutoffTimeMs).map(snapshotMetadata -> new TierObjectStore.KeyAndVersion(snapshotMetadata.toFragmentLocation("", FragmentType.TIER_TOPIC_SNAPSHOT).get().objectPath())).collect(Collectors.toList());
        if (snapshotsToDelete.isEmpty()) {
            log.info("No tier topic snapshots to be deleted");
        } else {
            StringBuilder sb = new StringBuilder("Following " + snapshotsToDelete.size() + " tier topic snapshots are to be deleted\n");
            snapshotsToDelete.forEach(kv -> sb.append(kv.key()).append(", "));
            log.info(sb.toString());
            this.tierObjectStore.deleteVersions(snapshotsToDelete);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void manageLifecycleForBackedUpSegments() {
        HashMap<String, Map<TopicIdPartition, java.util.List<BlobMetadata>>> dateToDeletableBlobs = new HashMap<String, Map<TopicIdPartition, java.util.List<BlobMetadata>>>();
        TierTopicReader tierTopicReader = null;
        this.markCLMActive();
        long currTimeMs = this.time.milliseconds();
        try {
            this.canCLMRunElseThrow();
            if (this.tierObjectStore.getBackend() == TierObjectStore.Backend.AzureBlockBlob) {
                this.executor = (ThreadPoolExecutor)Executors.newFixedThreadPool(this.config.threadPoolSize.get());
            }
            this.listObjectExecutor = (ThreadPoolExecutor)Executors.newFixedThreadPool(this.config.listObjectThreadPoolSize.get());
            Optional<LifecycleManagerState> stateOpt = this.lifecycleManagerState(currTimeMs);
            Map<NameAndId, Integer> topicToBackupRetentionInDays = this.backupRetentionForAllTopics();
            this.canCLMRunElseThrow();
            log.info("CustomLifecycleManager processing deletions for today and previous days");
            Map<String, java.util.List<String>> deletionListsCurrentOrPastDays = this.getAllDeletionListsToProcessNow(currTimeMs);
            for (Map.Entry<String, java.util.List<String>> entry : deletionListsCurrentOrPastDays.entrySet()) {
                String date = entry.getKey();
                for (String name : entry.getValue()) {
                    this.canCLMRunElseThrow();
                    StateManagerConfig stateManagerconfig = new StateManagerConfig(this.tierObjectStore, this.config.clusterId, this.canCLMRun, this::isShuttingDownOrInterrupted);
                    StateManager.loadDeletionList(stateManagerconfig, date, name, dateToDeletableBlobs);
                    this.processRetentionIncreases(currTimeMs, topicToBackupRetentionInDays, dateToDeletableBlobs);
                    java.util.List<TierObjectStore.KeyAndVersion> blobsEligibleForDeletionNow = this.retrieveObjectsEligibleForDeletion(currTimeMs, dateToDeletableBlobs, this.listObjectExecutor);
                    this.maybeSendDeleteRequests(blobsEligibleForDeletionNow);
                    Thread.sleep(2000L);
                }
            }
            Map<NameAndId, ReductionInRetention> retentionReductions = this.determineRetentionPeriodReductions(currTimeMs, stateOpt.orElse(null), topicToBackupRetentionInDays);
            tierTopicReader = this.tierTopicReader(stateOpt.orElse(null));
            boolean currentDayBlobsAlreadyLoaded = true;
            do {
                this.canCLMRunElseThrow();
                java.util.List<ObjectStoreUtils.DeletionRecord> segments = tierTopicReader.deletedSegments();
                log.info("CustomLifecycleManager processing new segments and checkpointing state. Num segments: " + segments.size());
                this.sortNewlyDeletedSegments(segments, topicToBackupRetentionInDays, retentionReductions, dateToDeletableBlobs);
                java.util.List<TierObjectStore.KeyAndVersion> blobsEligibleForDeletionNow = this.retrieveObjectsEligibleForDeletion(currTimeMs, dateToDeletableBlobs, this.listObjectExecutor);
                this.maybeSendDeleteRequests(blobsEligibleForDeletionNow);
                this.endOffsets = tierTopicReader.currentPositions();
                stateOpt = Optional.of(this.cleanupAndCheckpoint(currTimeMs, stateOpt.orElse(null), topicToBackupRetentionInDays, retentionReductions, deletionListsCurrentOrPastDays, dateToDeletableBlobs, currentDayBlobsAlreadyLoaded));
                dateToDeletableBlobs.clear();
                currentDayBlobsAlreadyLoaded = false;
                this.canCLMRunElseThrow();
                Thread.sleep(2000L);
            } while (tierTopicReader.hasMoreRecordsToConsume());
            this.recordLastRunStatus(true, Optional.empty());
        }
        catch (Exception e) {
            this.recordLastRunStatus(false, Optional.of(e));
        }
        finally {
            if (this.tierObjectStore.getBackend() == TierObjectStore.Backend.AzureBlockBlob) {
                this.executor.shutdownNow();
            }
            this.listObjectExecutor.shutdownNow();
            if (tierTopicReader != null) {
                tierTopicReader.maybeCloseConsumer();
            }
            this.markCLMInactive();
            this.updateMetrics(currTimeMs);
            log.debug("CustomLifecycleManager duration of last run " + (this.time.milliseconds() - currTimeMs) / 1000L + " seconds");
        }
    }

    public Optional<LifecycleManagerState> lifecycleManagerState(Long currTimeMs) throws InterruptedException, ParseException {
        log.info("Getting lifecycle manager state");
        LifecycleManagerState state = null;
        SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_PATTERN);
        String currentDateStr = StateManager.convertToDateKey(currTimeMs);
        Date currentDate = dateFormat.parse(currentDateStr);
        int maxLookBackInDays = this.maxLookBackPeriodInDays();
        try {
            state = StateManager.getState(new StateManagerConfig(this.tierObjectStore, this.config.clusterId, this.canCLMRun, this::isShuttingDownOrInterrupted));
            String lastRunDateStr = StateManager.convertToDateKey(state.lastRunTimestamp());
            Date lastRunDate = dateFormat.parse(lastRunDateStr);
            long timeDifferenceInMs = Math.abs(currentDate.getTime() - lastRunDate.getTime());
            long timeDifferenceInDays = TimeUnit.DAYS.convert(timeDifferenceInMs, TimeUnit.MILLISECONDS);
            if (timeDifferenceInDays > (long)maxLookBackInDays) {
                log.info("Discarding LifecycleManagerState older than " + maxLookBackInDays + " days. Current time: " + String.valueOf(new Date(currTimeMs)) + " Last run timestamp: " + String.valueOf(new Date(state.lastRunTimestamp())));
                return Optional.empty();
            }
        }
        catch (StateManager.LifecycleManagerVersionException e) {
            log.info("Version mismatch for LifecycleManagerState. Discard existing state files. " + e.getMessage());
            StateManagerConfig stateManagerConfig = new StateManagerConfig(this.tierObjectStore, this.config.clusterId, this.canCLMRun, this::isShuttingDownOrInterrupted);
            StateManager.deleteAllStateFiles(stateManagerConfig, this.executor);
            return Optional.empty();
        }
        catch (InterruptedException e) {
            throw e;
        }
        catch (Exception e) {
            log.warn("LifecycleManagerState from the previous run does not exist or has been corrupted. " + e.getMessage());
            return Optional.empty();
        }
        log.info("CustomLifecycleManager successfully retrieved the LifecycleManagerState from last run at: " + String.valueOf(new Date(state.lastRunTimestamp())));
        return Optional.of(state);
    }

    private boolean shouldReportFailure(Optional<Exception> ex) {
        if (ex.isPresent()) {
            if (ex.get() instanceof InterruptedException) {
                return false;
            }
            Throwable cause = ex.get().getCause();
            if (cause != null && ObjectStoreUtils.isEncryptionKeyStateInvalid((Exception)cause)) {
                return false;
            }
        }
        return this.canCLMRun.get();
    }

    private int maxLookBackPeriodInDays() {
        Integer maxBackupInDays = this.config.maxBackupInDays.get();
        TreeMap<Long, Integer> retentionToBackup = LifecycleManagerConfig.parseRetentionToBackupConfig(this.config.topicRetentionToBackupInDays.get());
        return Math.max(maxBackupInDays, retentionToBackup.get(-1L));
    }

    public LifecycleManagerState cleanupAndCheckpoint(Long currTimeMs, LifecycleManagerState state, Map<NameAndId, Integer> topicToBackupRetentionInDays, Map<NameAndId, ReductionInRetention> retentionChanges, Map<String, java.util.List<String>> deletionListsCurrentOrPastDays, Map<String, Map<TopicIdPartition, java.util.List<BlobMetadata>>> dateToDeletableBlobs, Boolean currentDayBlobsAlreadyLoaded) throws IOException, ParseException, InterruptedException {
        SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_PATTERN);
        String currentDateStr = StateManager.convertToDateKey(currTimeMs);
        Date currentDate = dateFormat.parse(currentDateStr);
        Map<String, String> dateToLatestDeletionList = StateManager.loadLatestDeletionListNamesFrom(currentDate, state);
        for (Map.Entry<String, Map<TopicIdPartition, java.util.List<BlobMetadata>>> entry : dateToDeletableBlobs.entrySet()) {
            String nextName;
            String dateStr = entry.getKey();
            Date date = dateFormat.parse(dateStr);
            Map<TopicIdPartition, java.util.List<BlobMetadata>> tpIdToBlobsMap = entry.getValue();
            if (date.before(currentDate)) {
                throw new RuntimeException("LifecycleManager must have deleted blobs from prior dates by now");
            }
            if (tpIdToBlobsMap.isEmpty()) continue;
            String name = dateToLatestDeletionList.getOrDefault(dateStr, null);
            if (name != null && (date.after(currentDate) || date.equals(currentDate) && !currentDayBlobsAlreadyLoaded.booleanValue())) {
                StateManager.loadDeletionList(new StateManagerConfig(this.tierObjectStore, this.config.clusterId, this.canCLMRun, this::isShuttingDownOrInterrupted), dateStr, name, dateToDeletableBlobs);
            }
            if ((nextName = name) == null || date.equals(currentDate) && currentDayBlobsAlreadyLoaded.booleanValue()) {
                nextName = StateManager.generateNextObjectsListName(null);
            }
            this.writeToBackupObjectsList(dateStr, nextName, tpIdToBlobsMap, dateToLatestDeletionList);
        }
        if (currentDayBlobsAlreadyLoaded.booleanValue()) {
            boolean wasThereAnyBlobUploadedForCurrentDay = dateToDeletableBlobs.getOrDefault(currentDateStr, null) != null;
            String lastUploadedListForCurrentDate = null;
            if (!wasThereAnyBlobUploadedForCurrentDay) {
                dateToLatestDeletionList.remove(currentDateStr);
            } else {
                lastUploadedListForCurrentDate = dateToLatestDeletionList.getOrDefault(currentDateStr, null);
            }
            this.cleanupOldState(currTimeMs, deletionListsCurrentOrPastDays, lastUploadedListForCurrentDate);
        }
        ByteBuffer buf = StateManager.serializeState(currTimeMs, this.endOffsets, topicToBackupRetentionInDays, retentionChanges, dateToLatestDeletionList);
        LifecycleManagerState updatedState = StateManager.deserializeState(buf);
        StateManagerConfig stateManagerConfig = new StateManagerConfig(this.tierObjectStore, this.config.clusterId, this.canCLMRun, this::isShuttingDownOrInterrupted);
        StateManager.putStateBufToObjectStore(stateManagerConfig, buf);
        return updatedState;
    }

    private void writeToBackupObjectsList(String date, String name, Map<TopicIdPartition, java.util.List<BlobMetadata>> tpIdToBlobs, Map<String, String> dateToLatestDeletionList) throws IOException, InterruptedException {
        HashMap<TopicIdPartition, java.util.List<BlobMetadata>> partitionedData = new HashMap<TopicIdPartition, java.util.List<BlobMetadata>>();
        long count = 0L;
        for (Map.Entry<TopicIdPartition, java.util.List<BlobMetadata>> tpIdToBlobsEntry : tpIdToBlobs.entrySet()) {
            TopicIdPartition tpId = tpIdToBlobsEntry.getKey();
            java.util.List<BlobMetadata> blobs = tpIdToBlobsEntry.getValue();
            for (BlobMetadata blob : blobs) {
                partitionedData.putIfAbsent(tpId, new ArrayList());
                ((java.util.List)partitionedData.get(tpId)).add(blob);
                if (++count < 10000L) continue;
                log.debug("Upload the backup objects list with " + count + " objects");
                ByteBuffer buf = StateManager.serializeBackupObjectsList(partitionedData);
                StateManagerConfig stateManagerConfig = new StateManagerConfig(this.tierObjectStore, this.config.clusterId, this.canCLMRun, this::isShuttingDownOrInterrupted);
                StateManager.putBackedUpObjectsListBufToObjectStore(stateManagerConfig, buf, date, name);
                partitionedData.clear();
                dateToLatestDeletionList.put(date, name);
                name = StateManager.generateNextObjectsListName(name);
                count = 0L;
            }
        }
        if (count > 0L) {
            log.debug("Upload the backup objects list with " + count + " objects");
            ByteBuffer buf = StateManager.serializeBackupObjectsList(partitionedData);
            StateManagerConfig stateManagerConfig = new StateManagerConfig(this.tierObjectStore, this.config.clusterId, this.canCLMRun, this::isShuttingDownOrInterrupted);
            StateManager.putBackedUpObjectsListBufToObjectStore(stateManagerConfig, buf, date, name);
            dateToLatestDeletionList.put(date, name);
        }
    }

    private void cleanupOldState(Long currTimeMs, Map<String, java.util.List<String>> deletionListsCurrentOrPastDays, String lastUploadedListForCurrDate) throws InterruptedException {
        String currentDateStr = StateManager.convertToDateKey(currTimeMs);
        if (deletionListsCurrentOrPastDays.containsKey(currentDateStr) && lastUploadedListForCurrDate != null) {
            java.util.List<String> oldListsForCurrentDate = deletionListsCurrentOrPastDays.get(currentDateStr);
            ArrayList disposableListsForCurrentDate = new ArrayList();
            int n = Integer.parseInt(lastUploadedListForCurrDate);
            for (String fileName : oldListsForCurrentDate) {
                int index = Integer.parseInt(fileName);
                if (index <= n) continue;
                disposableListsForCurrentDate.add(fileName);
            }
            deletionListsCurrentOrPastDays.put(currentDateStr, disposableListsForCurrentDate);
        }
        ArrayList<TierObjectStore.KeyAndVersion> keys = new ArrayList<TierObjectStore.KeyAndVersion>();
        for (Map.Entry entry : deletionListsCurrentOrPastDays.entrySet()) {
            String dateStr = (String)entry.getKey();
            for (String name : (java.util.List)entry.getValue()) {
                BackupObjectsListMetadata metadata = new BackupObjectsListMetadata(this.config.clusterId, dateStr, name);
                String keyPath = metadata.toFragmentLocation("", FragmentType.BACKUP_OBJECTS_LIST).get().objectPath();
                keys.add(new TierObjectStore.KeyAndVersion(keyPath));
            }
        }
        if (!keys.isEmpty()) {
            StringBuilder builder = new StringBuilder("Deleting the following backup object lists: ");
            keys.forEach(k -> builder.append(k.key()).append(", "));
            log.debug(builder.toString());
            ObjectStoreUtilsContext objectStoreUtilsContext = new ObjectStoreUtilsContext(this.tierObjectStore, this.canCLMRun, this::isShuttingDownOrInterrupted);
            ObjectStoreUtils.deleteVersions(objectStoreUtilsContext, keys, this.executor, DEFAULT_RETRY_POLICY);
        }
    }

    public Boolean isShuttingDownOrInterrupted() {
        return Thread.interrupted() || this.isShutdownInitiated();
    }

    private void sortNewlyDeletedSegments(java.util.List<ObjectStoreUtils.DeletionRecord> deletedSegments, Map<NameAndId, Integer> topicToBackupRetentionInDays, Map<NameAndId, ReductionInRetention> retentionChanges, Map<String, Map<TopicIdPartition, java.util.List<BlobMetadata>>> dateToDeletionLists) {
        log.debug("Number of deleted objects to process " + deletedSegments.size());
        long segmentsFromDeletedTopics = 0L;
        for (ObjectStoreUtils.DeletionRecord segment : deletedSegments) {
            TopicIdPartition topicIdPartition = segment.getTopicIdPartition();
            Optional<Integer> backupRetentionInDaysOpt = this.backupPeriodForTopic(topicToBackupRetentionInDays, topicIdPartition.topic(), topicIdPartition.kafkaTopicId(), true);
            if (!backupRetentionInDaysOpt.isPresent()) {
                ++segmentsFromDeletedTopics;
            }
            int backupRetentionInDays = -1;
            long minDeletionTimestamp = -1L;
            Optional<ReductionInRetention> retentionChangeOpt = this.retentionChangeForTopic(retentionChanges, topicIdPartition.topic(), topicIdPartition.kafkaTopicId());
            if (retentionChangeOpt.isPresent()) {
                backupRetentionInDays = retentionChangeOpt.get().lastNotedRetentionValueInDays;
                minDeletionTimestamp = retentionChangeOpt.get().minDeletionTimestamp;
            } else {
                backupRetentionInDays = backupRetentionInDaysOpt.orElseGet(this.config.maxBackupInDays);
            }
            Long backupRetentionInMs = TimeUnit.DAYS.toMillis(backupRetentionInDays);
            long permanentDeletionTimeInMs = segment.getCreationTime() + backupRetentionInMs;
            if (retentionChangeOpt.isPresent() && permanentDeletionTimeInMs < minDeletionTimestamp) {
                permanentDeletionTimeInMs = minDeletionTimestamp;
            }
            this.addToDeletionList(topicIdPartition, segment.getObjectId(), permanentDeletionTimeInMs, backupRetentionInDays, dateToDeletionLists);
        }
        this.updateBackupCost(deletedSegments.size(), segmentsFromDeletedTopics);
    }

    private Map<String, java.util.List<String>> getAllDeletionListsToProcessNow(Long currTimeMs) throws ParseException, InterruptedException {
        SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_PATTERN);
        String currentDateStr = StateManager.convertToDateKey(currTimeMs);
        Date currentDate = dateFormat.parse(currentDateStr);
        BackupObjectsListMetadata backupObjectsListMetadata = new BackupObjectsListMetadata(this.config.clusterId, "", "");
        ObjectStoreUtilsContext ctx = new ObjectStoreUtilsContext(this.tierObjectStore, this.canCLMRun, this::isShuttingDownOrInterrupted);
        Set<String> backupObjectsListNames = ObjectStoreUtils.backupObjectListNames(ctx, this.config.clusterId);
        HashMap<String, java.util.List<String>> dateToAllObjectLists = new HashMap<String, java.util.List<String>>();
        log.debug("Names of all backup object lists: " + String.valueOf(backupObjectsListNames));
        for (String key : backupObjectsListNames) {
            String dateStr = backupObjectsListMetadata.getDateFromKey(key);
            Date date = dateFormat.parse(dateStr);
            String name = backupObjectsListMetadata.getListNameFromKey(key);
            if (!date.before(currentDate) && !date.equals(currentDate)) continue;
            log.debug("Deletion list for current or past day: " + key + " " + dateStr + " " + name);
            dateToAllObjectLists.putIfAbsent(dateStr, new ArrayList());
            ((java.util.List)dateToAllObjectLists.get(dateStr)).add(name);
        }
        return dateToAllObjectLists;
    }

    private void processRetentionIncreases(Long currTimeMs, Map<NameAndId, Integer> topicToBackupRetentionDays, Map<String, Map<TopicIdPartition, java.util.List<BlobMetadata>>> dateToDeletionLists) throws ParseException {
        Object date;
        HashMap tempLists = new HashMap();
        SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_PATTERN);
        String currentDateStr = StateManager.convertToDateKey(currTimeMs);
        Date currentDate = dateFormat.parse(currentDateStr);
        Iterator<Map.Entry<String, Map<TopicIdPartition, java.util.List<BlobMetadata>>>> dateToDeletionListsIter = dateToDeletionLists.entrySet().iterator();
        long countRetentionIncreases = 0L;
        while (dateToDeletionListsIter.hasNext()) {
            Map.Entry<String, Map<TopicIdPartition, java.util.List<BlobMetadata>>> dateToDeletionListsEntry = dateToDeletionListsIter.next();
            String dateStr = dateToDeletionListsEntry.getKey();
            date = dateFormat.parse(dateStr);
            if (((Date)date).after(currentDate)) continue;
            Map<TopicIdPartition, java.util.List<BlobMetadata>> tpIdToBlobsMap = dateToDeletionListsEntry.getValue();
            Iterator<Map.Entry<TopicIdPartition, java.util.List<BlobMetadata>>> tpIdToBlobsIter = tpIdToBlobsMap.entrySet().iterator();
            while (tpIdToBlobsIter.hasNext()) {
                Map.Entry<TopicIdPartition, java.util.List<BlobMetadata>> tpIdToBlobs = tpIdToBlobsIter.next();
                TopicIdPartition tpId = tpIdToBlobs.getKey();
                Iterator<BlobMetadata> blobsIter = tpIdToBlobs.getValue().iterator();
                Integer currentRetentionDays = this.backupPeriodForTopic(topicToBackupRetentionDays, tpId.topic(), tpId.kafkaTopicId(), true).orElseGet(this.config.maxBackupInDays);
                while (blobsIter.hasNext()) {
                    BlobMetadata blob = blobsIter.next();
                    if (blob.retentionDays >= currentRetentionDays) continue;
                    int delta = currentRetentionDays - blob.retentionDays;
                    Long newTimeForDeletion = blob.timeForDeletionMs + (long)delta * 86400000L;
                    String newDateForDeletion = StateManager.convertToDateKey(newTimeForDeletion);
                    tempLists.putIfAbsent(newDateForDeletion, new HashMap());
                    Map futureList = (Map)tempLists.get(newDateForDeletion);
                    futureList.putIfAbsent(tpId, new ArrayList());
                    ((java.util.List)futureList.get(tpId)).add(new BlobMetadata(blob.objectId, newTimeForDeletion, currentRetentionDays));
                    blobsIter.remove();
                    ++countRetentionIncreases;
                }
                if (!tpIdToBlobs.getValue().isEmpty()) continue;
                tpIdToBlobsIter.remove();
            }
            if (!dateToDeletionListsEntry.getValue().isEmpty()) continue;
            dateToDeletionListsIter.remove();
        }
        log.info("Number of blobs with increased retention " + countRetentionIncreases);
        for (Map.Entry entry : tempLists.entrySet()) {
            date = (String)entry.getKey();
            Map tpIdToBlobsTempMap = (Map)entry.getValue();
            dateToDeletionLists.putIfAbsent((String)date, new HashMap());
            Map<TopicIdPartition, java.util.List<BlobMetadata>> tpIdToBlobsMap = dateToDeletionLists.get(date);
            for (Map.Entry entry2 : tpIdToBlobsTempMap.entrySet()) {
                TopicIdPartition tpId = (TopicIdPartition)entry2.getKey();
                java.util.List blobs = (java.util.List)entry2.getValue();
                tpIdToBlobsMap.putIfAbsent(tpId, new ArrayList());
                tpIdToBlobsMap.get(tpId).addAll(blobs);
            }
        }
    }

    public java.util.List<TierObjectStore.KeyAndVersion> retrieveObjectsEligibleForDeletion(Long currTimeMs, Map<String, Map<TopicIdPartition, java.util.List<BlobMetadata>>> dateToDeletableBlobs, ThreadPoolExecutor listObjectExecutor) throws Exception {
        java.util.List<TierObjectStore.KeyAndVersion> deletionList = Collections.synchronizedList(new ArrayList());
        SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_PATTERN);
        String currentDateStr = StateManager.convertToDateKey(currTimeMs);
        Date currentDate = dateFormat.parse(currentDateStr);
        Iterator<Map.Entry<String, Map<TopicIdPartition, java.util.List<BlobMetadata>>>> dateToDeletableSegmentBlobsIter = dateToDeletableBlobs.entrySet().iterator();
        log.debug("Checking objects eligibility for deletion");
        while (dateToDeletableSegmentBlobsIter.hasNext()) {
            Map.Entry<String, Map<TopicIdPartition, java.util.List<BlobMetadata>>> dateToDeletableSegmentBlobsEntry = dateToDeletableSegmentBlobsIter.next();
            String dateStr = dateToDeletableSegmentBlobsEntry.getKey();
            Date date = dateFormat.parse(dateStr);
            if (date.after(currentDate)) continue;
            Map<TopicIdPartition, java.util.List<BlobMetadata>> tpIdToSegmentBlobs = dateToDeletableSegmentBlobsEntry.getValue();
            Iterator<Map.Entry<TopicIdPartition, java.util.List<BlobMetadata>>> tpIdToSegmentBlobsIter = tpIdToSegmentBlobs.entrySet().iterator();
            while (tpIdToSegmentBlobsIter.hasNext()) {
                ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>();
                ConcurrentHashMap errors = new ConcurrentHashMap();
                Map.Entry<TopicIdPartition, java.util.List<BlobMetadata>> tpIdToSegmentBlobsEntry = tpIdToSegmentBlobsIter.next();
                TopicIdPartition tpId = tpIdToSegmentBlobsEntry.getKey();
                java.util.List<BlobMetadata> segmentBlobsList = tpIdToSegmentBlobsEntry.getValue();
                ObjectStoreUtilsContext ctx = new ObjectStoreUtilsContext(this.tierObjectStore, this.canCLMRun, this::isShuttingDownOrInterrupted);
                for (BlobMetadata segmentBlob : segmentBlobsList) {
                    CompletionStage future = ((CompletableFuture)CompletableFuture.supplyAsync(() -> {
                        try {
                            log.debug("Verifying live status of blob {}", (Object)segmentBlob.objectId);
                            if (segmentBlob.timeForDeletionMs > currTimeMs) {
                                return null;
                            }
                            return ObjectStoreUtils.verifyObjectNotLive(ctx, tpId, UUID.fromString(segmentBlob.objectId), DEFAULT_RETRY_POLICY);
                        }
                        catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                            throw new WrappedInterruptedException(e);
                        }
                    }, listObjectExecutor).thenAccept(blobsWithVersions -> this.appendVersionsToDeletionList((Map<String, java.util.List<VersionInformation>>)blobsWithVersions, segmentBlob, currTimeMs, deletionList))).exceptionally(ex -> this.handleListObjectException(ex.getCause(), segmentBlob, errors));
                    futures.add(future);
                }
                CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).join();
                futures.clear();
                Iterator<BlobMetadata> blobIter = tpIdToSegmentBlobsEntry.getValue().iterator();
                while (blobIter.hasNext()) {
                    BlobMetadata key = blobIter.next();
                    if (errors.containsKey(key)) {
                        throw (Exception)errors.get(key);
                    }
                    blobIter.remove();
                }
                if (!tpIdToSegmentBlobsEntry.getValue().isEmpty()) continue;
                tpIdToSegmentBlobsIter.remove();
            }
            if (!dateToDeletableSegmentBlobsEntry.getValue().isEmpty()) continue;
            dateToDeletableSegmentBlobsIter.remove();
        }
        log.info("Number of objects eligible for immediate deletion: {}", (Object)deletionList.size());
        return deletionList;
    }

    public void initializeMetrics() {
        this.lifecycleManagerMetrics.durationOfLastRunMetric.update(0L);
        this.lifecycleManagerMetrics.numDataSegmentsDeletedMetric.update(0L);
        this.lifecycleManagerMetrics.numDataSegmentsDeletedInDueTimeMetric.update(0L);
        this.lifecycleManagerMetrics.netDelayInDeletionOfOneDataBlobInSecMetric.update(0L);
        this.lifecycleManagerMetrics.numDataSegmentsDeletedBeforeBucketPolicyMetric.update(0L);
        this.lifecycleManagerMetrics.numObjectsDeletedMetric.update(0L);
        this.lifecycleManagerMetrics.numObjectsDeletedBeforeBucketPolicyMetric.update(0L);
        this.lifecycleManagerMetrics.weightedSavingsCurrentRunMetric.update(0.0);
        this.lifecycleManagerMetrics.weightedBackupCostCurrentRunMetric.update(0.0);
        this.deletionCounters.initializeCounters();
    }

    private void markCLMActive() {
        this.lifecycleManagerMetrics.activeStateIndicatorMetric.increment();
    }

    private void markCLMInactive() {
        this.lifecycleManagerMetrics.activeStateIndicatorMetric.decrement();
    }

    private void recordLastRunStatus(boolean lastRunSucceeded, Optional<Exception> optException) {
        if (lastRunSucceeded) {
            this.lifecycleManagerMetrics.failureMetric.update(0);
            this.lifecycleManagerMetrics.ignoredFailureMetric.update(0);
            log.info("Last run of backup object LifecycleManager has completed successfully");
            return;
        }
        StringBuilder errorMessage = new StringBuilder("Last run of backup object LifecycleManager has failed.");
        if (this.shouldReportFailure(optException)) {
            errorMessage.append(" ").append("Ignorable error: false");
            this.lifecycleManagerMetrics.failureMetric.increment();
            this.lifecycleManagerMetrics.ignoredFailureMetric.update(0);
        } else {
            errorMessage.append(" ").append("Ignorable error: true");
            this.lifecycleManagerMetrics.ignoredFailureMetric.increment();
            this.lifecycleManagerMetrics.failureMetric.update(0);
        }
        optException.ifPresent(ex -> {
            String cause = ex.getCause() != null ? "Cause: " + ex.getCause().getMessage() : "";
            errorMessage.append("\n").append(ex.getMessage()).append("\n").append(cause);
        });
        log.error(errorMessage.toString());
    }

    private synchronized void updateDeletionCounters(Long currTimeMs, String blobKey, Integer retention, Long timeForDeletionMs) {
        boolean isSegment;
        ++this.deletionCounters.numObjectsDeleted;
        boolean bl = isSegment = blobKey.endsWith("segment") || blobKey.endsWith("segment-with-metadata");
        if (isSegment) {
            ++this.deletionCounters.numDataSegmentsDeleted;
            long runFrequencyMs = Math.max((long)this.config.customLifecycleManagerFrequencyInMs.get().intValue(), this.config.minDelayInMs.get());
            if (runFrequencyMs >= currTimeMs - timeForDeletionMs) {
                ++this.deletionCounters.numSegmentsDeletedInDueTime;
            }
            long midPointExpectedLastRunTimeMs = currTimeMs - runFrequencyMs / 2L;
            this.deletionCounters.netDelayInDeletionOfOneDataBlobInMs += midPointExpectedLastRunTimeMs - timeForDeletionMs;
        }
        if (retention < this.maxBackupForSegmentsFromLiveTopics) {
            ++this.deletionCounters.numObjectsDeletedBeforeBucketPolicy;
            if (isSegment) {
                ++this.deletionCounters.numDataSegmentsDeletedBeforeBucketPolicy;
                this.deletionCounters.spaceSavings100MBEachDay += (long)(this.maxBackupForSegmentsFromLiveTopics - retention);
            }
        }
    }

    private void updateBackupCost(long totalNumDeletedSegments, long segmentsFromDeletedTopics) {
        this.deletionCounters.backupCost100MBEachDay += (totalNumDeletedSegments - segmentsFromDeletedTopics) * (long)this.maxBackupForSegmentsFromLiveTopics;
        this.deletionCounters.backupCost100MBEachDay += segmentsFromDeletedTopics * (long)this.config.maxBackupInDays.get().intValue();
    }

    private void updateMetrics(long startTimeMs) {
        long endTimeMs = this.time.milliseconds();
        long durationOfLastRun = (endTimeMs - startTimeMs) / 1000L;
        this.lifecycleManagerMetrics.durationOfLastRunMetric.update(durationOfLastRun);
        this.lifecycleManagerMetrics.durationOfRunSensor.record(durationOfLastRun);
        this.lifecycleManagerMetrics.numDataSegmentsDeletedMetric.update(this.deletionCounters.numDataSegmentsDeleted);
        this.lifecycleManagerMetrics.numDataSegmentsDeletedInDueTimeMetric.update(this.deletionCounters.numSegmentsDeletedInDueTime);
        this.lifecycleManagerMetrics.netDelayInDeletionOfOneDataBlobInSecMetric.update(this.deletionCounters.netDelayInDeletionOfOneDataBlobInMs / 1000L);
        this.lifecycleManagerMetrics.numDataSegmentsDeletedBeforeBucketPolicyMetric.update(this.deletionCounters.numDataSegmentsDeletedBeforeBucketPolicy);
        this.lifecycleManagerMetrics.numObjectsDeletedMetric.update(this.deletionCounters.numObjectsDeleted);
        this.lifecycleManagerMetrics.numObjectsDeletedBeforeBucketPolicyMetric.update(this.deletionCounters.numObjectsDeletedBeforeBucketPolicy);
        double spaceSavingsPerMonthPerGB = (double)this.deletionCounters.spaceSavings100MBEachDay / 30.0 / 10.0;
        this.lifecycleManagerMetrics.weightedSavingsCurrentRunMetric.update(spaceSavingsPerMonthPerGB);
        this.lifecycleManagerMetrics.weightedSavingsSensor.record(spaceSavingsPerMonthPerGB);
        double backupCostPerMonthPerGB = (double)this.deletionCounters.backupCost100MBEachDay / 30.0 / 10.0;
        this.lifecycleManagerMetrics.weightedBackupCostCurrentRunMetric.update(backupCostPerMonthPerGB);
    }

    private Map<NameAndId, Integer> backupRetentionForAllTopics() throws InterruptedException {
        log.info("Getting backup retention for all topics");
        Map<NameAndId, Long> topicToRetentionMs = this.getKafkaTopicRetentionMs();
        return this.getBackupRetentionInDaysForAllTopics(topicToRetentionMs);
    }

    public Map<NameAndId, ReductionInRetention> determineRetentionPeriodReductions(Long currentTime, LifecycleManagerState state, Map<NameAndId, Integer> backupRetentionInDays) throws ParseException {
        HashMap<NameAndId, ReductionInRetention> retentionChanges = new HashMap<NameAndId, ReductionInRetention>();
        if (state == null) {
            return retentionChanges;
        }
        for (int i = 0; i < state.retentionDataLength(); ++i) {
            ReductionInRetention change;
            TopicRetentionData topicState = state.retentionData(i);
            NameAndId topic = new NameAndId(topicState.topic(), Uuid.fromString(Objects.requireNonNull(topicState.topicId())));
            Optional<Integer> desiredRetentionInDaysOpt = this.backupPeriodForTopic(backupRetentionInDays, topic.name(), topic.id(), false);
            if (!desiredRetentionInDaysOpt.isPresent()) continue;
            int desiredRetentionInDays = desiredRetentionInDaysOpt.get();
            int lastNotedRetentionInDays = topicState.currentRetentionInDays();
            Long minDeletionTimestamp = topicState.ongoingReductionMinDeletionTimestamp();
            Long changeRecordTimestamp = topicState.reductionRecordTimestamp();
            if (lastNotedRetentionInDays == desiredRetentionInDays && changeRecordTimestamp == -1L || lastNotedRetentionInDays < desiredRetentionInDays) continue;
            if (changeRecordTimestamp == -1L) {
                minDeletionTimestamp = currentTime + (long)lastNotedRetentionInDays * 86400000L;
                minDeletionTimestamp = LifecycleManager.getLastMillisecondForTheDay(minDeletionTimestamp);
                change = new ReductionInRetention(topic, lastNotedRetentionInDays, desiredRetentionInDays, minDeletionTimestamp, currentTime, false);
            } else {
                long daysSinceLastRun = this.durationInDays(changeRecordTimestamp, currentTime);
                if (daysSinceLastRun > 0L) {
                    if (lastNotedRetentionInDays > desiredRetentionInDays) {
                        boolean changeHasCompleted = lastNotedRetentionInDays - (int)daysSinceLastRun < desiredRetentionInDays;
                        lastNotedRetentionInDays = Math.max(lastNotedRetentionInDays - (int)daysSinceLastRun, desiredRetentionInDays);
                        change = new ReductionInRetention(topic, lastNotedRetentionInDays, desiredRetentionInDays, minDeletionTimestamp, currentTime, changeHasCompleted);
                    } else {
                        change = new ReductionInRetention(topic, lastNotedRetentionInDays, desiredRetentionInDays, minDeletionTimestamp, currentTime, true);
                    }
                } else {
                    change = new ReductionInRetention(topic, lastNotedRetentionInDays, desiredRetentionInDays, minDeletionTimestamp, changeRecordTimestamp, false);
                }
            }
            retentionChanges.put(topic, change);
        }
        log.info("Number of topics whose backup retention period has reduced " + retentionChanges.size());
        retentionChanges.forEach((key, value) -> log.debug("Reduction in backup object retention period for " + value.toString()));
        return retentionChanges;
    }

    private void maybeSendDeleteRequests(java.util.List<TierObjectStore.KeyAndVersion> blobsEligibleForDeletionNow) throws InterruptedException {
        int startIndex = 0;
        int endIndex = Math.min(startIndex + 1000, blobsEligibleForDeletionNow.size());
        while (startIndex < endIndex) {
            this.canCLMRunElseThrow();
            java.util.List<TierObjectStore.KeyAndVersion> subset = blobsEligibleForDeletionNow.subList(startIndex, endIndex);
            log.info(subset.size() + " backed-up objects are to be deleted\n");
            if (log.isTraceEnabled()) {
                StringBuilder sb = new StringBuilder("Following " + subset.size() + " backed-up objects are to be deleted\n");
                subset.forEach(kv -> sb.append(kv.key()).append(", "));
                log.trace(sb.toString());
            }
            ObjectStoreUtilsContext ctx = new ObjectStoreUtilsContext(this.tierObjectStore, this.canCLMRun, this::isShuttingDownOrInterrupted);
            ObjectStoreUtils.deleteVersions(ctx, subset, this.executor, DEFAULT_RETRY_POLICY);
            startIndex = endIndex;
            endIndex = Math.min(startIndex + 1000, blobsEligibleForDeletionNow.size());
        }
    }

    private static Long getLastMillisecondForTheDay(Long timeInMs) throws ParseException {
        SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_PATTERN);
        long timePlusOneDayInMs = timeInMs + 86400000L;
        Date timePlusOneDay = new Date(timePlusOneDayInMs);
        String zeroHoursTimePlusOneDayStr = dateFormat.format(timePlusOneDay);
        Date zeroHoursTimePlusOneDay = dateFormat.parse(zeroHoursTimePlusOneDayStr);
        return zeroHoursTimePlusOneDay.getTime() - 1L;
    }

    private String extractPrefix(String fullName) {
        if (fullName == null) {
            return null;
        }
        String ext = "." + ObjectType.BACKUP_OBJECTS_LIST.suffix();
        int indexOfExt = fullName.indexOf(ext);
        if (indexOfExt == -1) {
            throw new RuntimeException(fullName + " is not a valid name for backup objects list");
        }
        return fullName.substring(0, indexOfExt);
    }

    private Optional<Integer> backupPeriodForTopic(Map<NameAndId, Integer> backupRetentionInDays, String topic, Uuid topicId, boolean weakCheckAllowed) {
        NameAndId weakIdentifier;
        NameAndId topicInfo = new NameAndId(topic, topicId);
        if (backupRetentionInDays.containsKey(topicInfo)) {
            return Optional.of(backupRetentionInDays.get(topicInfo));
        }
        if (weakCheckAllowed && !topicId.equals(Uuid.ZERO_UUID) && backupRetentionInDays.containsKey(weakIdentifier = new NameAndId(topic, Uuid.ZERO_UUID))) {
            return Optional.of(backupRetentionInDays.get(weakIdentifier));
        }
        return Optional.empty();
    }

    private Optional<ReductionInRetention> retentionChangeForTopic(Map<NameAndId, ReductionInRetention> retentionChanges, String topic, Uuid topicId) {
        NameAndId weakIdentifier;
        NameAndId topicInfo = new NameAndId(topic, topicId);
        if (retentionChanges.containsKey(topicInfo)) {
            return Optional.of(retentionChanges.get(topicInfo));
        }
        if (!topicId.equals(Uuid.ZERO_UUID) && retentionChanges.containsKey(weakIdentifier = new NameAndId(topic, Uuid.ZERO_UUID))) {
            return Optional.of(retentionChanges.get(weakIdentifier));
        }
        return Optional.empty();
    }

    private long durationInDays(Long timeA, Long timeB) {
        if (timeA > timeB) {
            throw new IllegalArgumentException("timeA: " + timeA + " must be less or equal to timeB: " + timeB);
        }
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
        Date dateA = new Date(timeA);
        String dateAStr = dateFormat.format(dateA);
        Date dateB = new Date(timeB);
        String dateBStr = dateFormat.format(dateB);
        LocalDate localDateA = LocalDate.parse(dateAStr);
        LocalDate localDateB = LocalDate.parse(dateBStr);
        return ChronoUnit.DAYS.between(localDateA, localDateB);
    }

    public void waitForTierTopicToBeAvailable() throws InterruptedException {
        int numRetries = 0;
        while (!this.isTierTopicAvailable.get().booleanValue() && !this.isShutdownInitiated()) {
            if (numRetries > 20) {
                throw new RuntimeException("Backup object lifecycle manager did not start because TierTopicManager is not ready after multiple retries");
            }
            log.debug("Backup object lifecycle manager will back off as TierTopicManager is not ready");
            LifecycleManager.sleep(1000L);
            ++numRetries;
        }
        log.info("Starting backup object lifecycle manager at brokerId: " + this.config.brokerId);
    }

    private void addToDeletionList(TopicIdPartition topicIdPartition, UUID objectId, Long deletionTime, Integer retentionDays, Map<String, Map<TopicIdPartition, java.util.List<BlobMetadata>>> dateToDeletionLists) {
        String dateKey = StateManager.convertToDateKey(deletionTime);
        dateToDeletionLists.putIfAbsent(dateKey, new HashMap());
        Map<TopicIdPartition, java.util.List<BlobMetadata>> tpIdToBlobs = dateToDeletionLists.get(dateKey);
        tpIdToBlobs.putIfAbsent(topicIdPartition, new ArrayList());
        tpIdToBlobs.get(topicIdPartition).add(new BlobMetadata(objectId.toString(), deletionTime, retentionDays));
    }

    public Map<NameAndId, Long> getKafkaTopicRetentionMs() throws InterruptedException {
        HashMap<NameAndId, Long> topicToConfigs = new HashMap<NameAndId, Long>();
        InternalAdmin admin = this.config.internalAdmin.get();
        ArrayList<String> topicNames = new ArrayList<String>(admin.listAllTopics());
        log.info("Total number of topics in cluster: " + topicNames.size());
        long totalTimeMs = 0L;
        int startIndex = 0;
        int endIndex = Math.min(startIndex + 500, topicNames.size());
        while (startIndex < endIndex) {
            this.canCLMRunElseThrow();
            long startTimeMs = this.time.milliseconds();
            java.util.List<String> subset = topicNames.subList(startIndex, endIndex);
            topicToConfigs.putAll(this.requestConfigsWithRetry(subset));
            startIndex = endIndex;
            endIndex = Math.min(startIndex + 500, topicNames.size());
            if (startIndex < endIndex) {
                this.pause(500L, TimeUnit.MILLISECONDS);
            }
            long timeDiffMs = this.time.milliseconds() - startTimeMs;
            totalTimeMs += timeDiffMs;
            log.debug("Pulled retention configuration for " + subset.size() + " topics in " + timeDiffMs + " millis");
        }
        log.info("Pulled retention configuration for " + topicToConfigs.size() + "/" + topicNames.size() + " topics in " + totalTimeMs + " millis");
        log.debug("Pulled retention configuration for the following topics: " + String.valueOf(topicToConfigs.keySet()));
        return topicToConfigs;
    }

    public Map<NameAndId, Long> requestConfigsWithRetry(java.util.List<String> topics) throws InterruptedException {
        HashMap<NameAndId, Long> topicToConfigs = new HashMap<NameAndId, Long>();
        HashSet<String> remainingTopics = new HashSet<String>(topics);
        InternalAdmin admin = this.config.internalAdmin.get();
        java.util.List<String> configurationKeys = Arrays.asList("retention.ms", "cleanup.policy");
        HashMap<Short, Long> errorTypeToCount = new HashMap<Short, Long>();
        int numTries = 0;
        while (!remainingTopics.isEmpty()) {
            ++numTries;
            ArrayList<String> copy = new ArrayList<String>(remainingTopics);
            Map<NameAndId, List<DescribeConfigsResponseData.DescribeConfigsResult>> configs = admin.topicConfigurations(copy, configurationKeys);
            configs.forEach((topic, results) -> results.foreach(result -> {
                switch (Errors.forCode(result.errorCode())) {
                    case NONE: {
                        HashMap props = new HashMap();
                        result.configs().forEach(config -> props.put(config.name(), config.value()));
                        if (!Topic.isInternal(topic.name()) && props.getOrDefault("cleanup.policy", "delete").equals("delete")) {
                            topicToConfigs.put((NameAndId)topic, Long.parseLong((String)props.get("retention.ms")));
                        }
                        remainingTopics.remove(topic.name());
                        break;
                    }
                    case UNKNOWN_TOPIC_OR_PARTITION: {
                        log.info("Topic deleted " + topic.name() + " till CLM could get its configurations");
                        remainingTopics.remove(topic.name());
                        break;
                    }
                    default: {
                        log.info("Obtained error code " + result.errorCode() + " for topic " + topic.name() + " from InternalAdmin");
                        errorTypeToCount.putIfAbsent(result.errorCode(), 0L);
                        errorTypeToCount.compute(result.errorCode(), (k, v) -> v == null ? 1L : v + 1L);
                    }
                }
                return null;
            }));
            java.util.List<String> partitionsToRemove = LifecycleManager.determineDeletedTopicPartitionsToRemoveFromCLM(copy, configs);
            partitionsToRemove.forEach(topic -> {
                log.warn("Topic: " + topic + " removed from CLM as it has been deleted");
                remainingTopics.remove(topic);
            });
            if (remainingTopics.isEmpty()) continue;
            errorTypeToCount.forEach((e, c) -> log.warn("Received error code: " + e.toString() + " " + c.toString() + " times while fetching configs"));
            if (numTries >= 10) {
                throw new RuntimeException("CLM could not pull configs for all the topics");
            }
            this.pause(2000L, TimeUnit.MILLISECONDS);
        }
        return topicToConfigs;
    }

    public static java.util.List<String> determineDeletedTopicPartitionsToRemoveFromCLM(java.util.List<String> remainingTopicsList, Map<NameAndId, List<DescribeConfigsResponseData.DescribeConfigsResult>> configs) {
        Set topicsFetchedFromInternalAdmin = configs.keySet().stream().map(NameAndId::name).collect(Collectors.toSet());
        java.util.List<String> partitionsToDelete = remainingTopicsList.stream().filter(topic -> !topicsFetchedFromInternalAdmin.contains(topic)).collect(Collectors.toList());
        return partitionsToDelete;
    }

    private void canCLMRunElseThrow() throws InterruptedException {
        if (!this.canCLMRun.get().booleanValue()) {
            throw new RuntimeException("CustomLifecycleManager will exit because backup object lifecycle management config has been disabled");
        }
        if (Thread.interrupted() || this.isShutdownInitiated()) {
            throw new InterruptedException("CustomLifecycleManager thread has been interrupted");
        }
    }

    public Map<NameAndId, Integer> getBackupRetentionInDaysForAllTopics(Map<NameAndId, Long> latestRetentionConfigs) {
        HashMap<NameAndId, Integer> topicToBackupRetentionInDays = new HashMap<NameAndId, Integer>();
        Integer maxBackupInDays = this.config.maxBackupInDays.get();
        maxBackupInDays = maxBackupInDays < 0 ? Integer.MAX_VALUE : maxBackupInDays;
        String configString = this.config.topicRetentionToBackupInDays.get();
        TreeMap<Long, Integer> retentionToBackup = LifecycleManagerConfig.parseRetentionToBackupConfig(configString);
        this.maxBackupForSegmentsFromLiveTopics = Math.min(retentionToBackup.lastEntry().getValue(), maxBackupInDays);
        log.debug("MaxBackupInDays: " + maxBackupInDays + " TopicRetentionToBackup mapping: " + String.valueOf(retentionToBackup));
        for (Map.Entry<NameAndId, Long> entry : latestRetentionConfigs.entrySet()) {
            Long topicRetentionInDays = entry.getValue() == -1L ? -1L : TimeUnit.MILLISECONDS.toDays(entry.getValue());
            Map.Entry<Long, Integer> floorEntry = retentionToBackup.floorEntry(topicRetentionInDays);
            int backupSegmentRetentionInDays = Math.min(floorEntry.getValue(), maxBackupInDays);
            topicToBackupRetentionInDays.put(entry.getKey(), backupSegmentRetentionInDays);
        }
        return topicToBackupRetentionInDays;
    }

    private java.util.List<Long> loadTierOffsets(LifecycleManagerState state) {
        ArrayList<Long> tierOffsets = new ArrayList<Long>();
        if (state != null && state.tierOffsetsLength() == this.config.tierMetadataNumPartitions.shortValue()) {
            for (int i = 0; i < state.tierOffsetsLength(); ++i) {
                tierOffsets.add(state.tierOffsets(i));
            }
        }
        return tierOffsets;
    }

    private TierTopicReader tierTopicReader(LifecycleManagerState state) throws InterruptedException {
        java.util.List<Long> tierOffsets = this.loadTierOffsets(state);
        return this.createTierTopicReader(tierOffsets);
    }

    public TierTopicReader createTierTopicReader(java.util.List<Long> tierOffsets) throws InterruptedException {
        return new TierTopicReader(new TierTopicReaderConfig(this.config.interBrokerClientConfigs, this.config.clusterId, this.config.brokerId, this.config.tierMetadataNumPartitions.shortValue(), this.config.tierMetadataMaxPollMs, tierOffsets, 10000L, value -> this.lifecycleManagerMetrics.consumerLagMetric.update((Long)value), this.canCLMRun, this::isShutdownInitiated, this.time, 10, this::maxLookBackPeriodInDays));
    }

    public void appendVersionsToDeletionList(Map<String, java.util.List<VersionInformation>> blobs, BlobMetadata segmentBlob, Long currTimeMs, java.util.List<TierObjectStore.KeyAndVersion> deletionList) {
        if (blobs == null || blobs.isEmpty()) {
            return;
        }
        for (Map.Entry<String, java.util.List<VersionInformation>> blobWithVersions : blobs.entrySet()) {
            String blobKey = blobWithVersions.getKey();
            for (VersionInformation version : blobWithVersions.getValue()) {
                deletionList.add(new TierObjectStore.KeyAndVersion(blobKey, version.getVersionId()));
                this.updateDeletionCounters(currTimeMs, blobKey, segmentBlob.retentionDays, segmentBlob.timeForDeletionMs);
            }
        }
    }

    public Void handleListObjectException(Throwable ex, BlobMetadata blob, Map<BlobMetadata, Throwable> errors) {
        if (ex instanceof IllegalArgumentException) {
            log.error(ex.getMessage());
        } else if (ex instanceof WrappedInterruptedException) {
            errors.put(blob, ex.getCause());
        } else {
            errors.put(blob, ex);
        }
        return null;
    }

    public static class DeletionCounters {
        long numDataSegmentsDeleted;
        long numDataSegmentsDeletedBeforeBucketPolicy;
        long numObjectsDeleted;
        long numObjectsDeletedBeforeBucketPolicy;
        long spaceSavings100MBEachDay;
        long backupCost100MBEachDay;
        long numSegmentsDeletedInDueTime;
        long netDelayInDeletionOfOneDataBlobInMs;

        public DeletionCounters() {
            this.initializeCounters();
        }

        private void initializeCounters() {
            this.numDataSegmentsDeleted = 0L;
            this.numSegmentsDeletedInDueTime = 0L;
            this.netDelayInDeletionOfOneDataBlobInMs = 0L;
            this.numDataSegmentsDeletedBeforeBucketPolicy = 0L;
            this.numObjectsDeleted = 0L;
            this.numObjectsDeletedBeforeBucketPolicy = 0L;
            this.spaceSavings100MBEachDay = 0L;
            this.backupCost100MBEachDay = 0L;
        }
    }

    public static class BlobMetadata {
        String objectId;
        Long timeForDeletionMs;
        Integer retentionDays;

        public BlobMetadata(String objectId, Long timeForDeletionMs, Integer retentionValue) {
            this.objectId = objectId;
            this.timeForDeletionMs = timeForDeletionMs;
            this.retentionDays = retentionValue;
        }
    }

    public static class ReductionInRetention {
        NameAndId topic;
        int lastNotedRetentionValueInDays;
        int desiredRetentionInDays;
        Long minDeletionTimestamp;
        Long changeTimestamp;
        boolean hasCompleted;

        public ReductionInRetention(NameAndId topic, int lastNotedRetentionValueInDays, int desiredRetentionInDays, Long minDeletionTimestamp, Long changeTimestamp, boolean hasCompleted) {
            this.topic = topic;
            this.lastNotedRetentionValueInDays = lastNotedRetentionValueInDays;
            this.desiredRetentionInDays = desiredRetentionInDays;
            this.minDeletionTimestamp = minDeletionTimestamp;
            this.changeTimestamp = changeTimestamp;
            this.hasCompleted = hasCompleted;
        }

        public String toString() {
            return "Topic=" + this.topic.toString() + ", lastNotedRetentionInDays=" + this.lastNotedRetentionValueInDays + ", desiredRetentionInDays=" + this.desiredRetentionInDays + ", minDeletionTimestamp=" + this.minDeletionTimestamp.toString() + ", changeTimestamp=" + this.changeTimestamp.toString() + ", hasCompleted=" + this.hasCompleted;
        }
    }
}

