/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.store.encryption;

import com.azure.core.exception.HttpResponseException;
import com.azure.core.http.HttpClient;
import com.azure.core.util.HttpClientOptions;
import com.azure.identity.WorkloadIdentityCredential;
import com.azure.identity.WorkloadIdentityCredentialBuilder;
import com.azure.security.keyvault.keys.cryptography.CryptographyClient;
import com.azure.security.keyvault.keys.cryptography.CryptographyClientBuilder;
import com.azure.security.keyvault.keys.cryptography.models.KeyWrapAlgorithm;
import com.azure.security.keyvault.keys.cryptography.models.UnwrapResult;
import com.azure.security.keyvault.keys.cryptography.models.WrapResult;
import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.models.AccessTier;
import com.azure.storage.blob.models.BlobHttpHeaders;
import com.azure.storage.blob.models.BlobRequestConditions;
import io.confluent.kafka.multitenant.ByokMetadata;
import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import kafka.tier.exceptions.TierObjectStoreFatalException;
import kafka.tier.exceptions.TierObjectStoreRetriableException;
import kafka.tier.store.AzureBlockBlobTierObjectStoreConfig;
import kafka.tier.store.encryption.CleartextDataKey;
import kafka.tier.store.encryption.DataEncryptionKeyHolder;
import kafka.tier.store.encryption.EncryptedDataKey;
import kafka.tier.store.encryption.TenantAwareEncryptionKeyManager;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.utils.ByteBufferInputStream;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.multitenant.MultiTenantMetadata;
import org.apache.kafka.server.util.KafkaScheduler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AzureTenantAwareEncryptionManager
extends TenantAwareEncryptionKeyManager {
    private static final Logger log = LoggerFactory.getLogger(AzureTenantAwareEncryptionManager.class);
    private static final KeyWrapAlgorithm ENCRYPTION_ALGORITHM_AZURE = KeyWrapAlgorithm.RSA_OAEP_256;
    private final BlobContainerClient blobContainerClient;
    private final AzureBlockBlobTierObjectStoreConfig config;
    private final AzureCryptographyClientSupplier cryptographyClientSupplier;

    public AzureTenantAwareEncryptionManager(BlobContainerClient blobContainerClient, AzureBlockBlobTierObjectStoreConfig config, KafkaScheduler kafkaScheduler, MultiTenantMetadata multiTenantMetadata, Time time, Metrics metrics) {
        this(blobContainerClient, new DefaultAzureCryptographyClientSupplier(), config, kafkaScheduler, multiTenantMetadata, time, metrics);
    }

    public AzureTenantAwareEncryptionManager(BlobContainerClient blobContainerClient, AzureCryptographyClientSupplier cryptographyClientSupplier, AzureBlockBlobTierObjectStoreConfig config, KafkaScheduler kafkaScheduler, MultiTenantMetadata multiTenantMetadata, Time time, Metrics metrics) {
        super(multiTenantMetadata, config.tenantAwareEncryptionKeyManagerKeyRotationIntervalMs, config.tenantAwareEncryptionKeyManagerMaxTenantKeyCacheSize, config.tenantAwareEncryptionKeyManagerTenantKeyCacheEvictionTimeSec, time, metrics, kafkaScheduler);
        this.blobContainerClient = blobContainerClient;
        this.cryptographyClientSupplier = cryptographyClientSupplier;
        this.config = config;
    }

    public AzureBlockBlobTierObjectStoreConfig getConfig() {
        return this.config;
    }

    @Override
    protected String getDataEncryptionKeyAlgorithm() {
        return "AES";
    }

    @Override
    protected int getDataEncryptionKeySizeBytes() {
        return 256;
    }

    @Override
    public String getKeyManagementClientEncryptionAlgorithm() {
        return ENCRYPTION_ALGORITHM_AZURE.toString();
    }

    @Override
    public void close() {
        super.close();
        try {
            this.cryptographyClientSupplier.close();
        }
        catch (IOException e) {
            log.error("Received IOException while closing HTTP Client during shutdown of CryptographyClientSupplier.", e);
        }
    }

    @Override
    protected Map<String, String> fetchWellKnownPathMetadata(String logicalClusterId) {
        String path = AzureTenantAwareEncryptionManager.getLastActiveKeyPath(this.config.azureBlobPrefix, logicalClusterId);
        log.info("[{}] Downloading previously generated key from path {}", (Object)logicalClusterId, (Object)path);
        try {
            BlobClient blobClient = this.blobContainerClient.getBlobClient(path);
            if (blobClient.exists().booleanValue()) {
                return blobClient.getProperties().getMetadata();
            }
            log.info("[{}] Previously generated key not found at path {}", (Object)logicalClusterId, (Object)path);
            return new HashMap<String, String>();
        }
        catch (Exception e) {
            if (AzureTenantAwareEncryptionManager.isRetryableAzureException(e)) {
                if (this.metrics != null) {
                    this.metrics.recordFetchKeyRetriableError();
                }
                throw new TierObjectStoreRetriableException(String.format("[%s] Failed to fetch well known path metadata from %s", logicalClusterId, path), e);
            }
            if (this.metrics != null) {
                this.metrics.recordFetchKeyFatalError();
            }
            throw new TierObjectStoreFatalException(String.format("[%s] Unknown exception when fetching well known path metadata from %s", logicalClusterId, path), e);
        }
    }

    @Override
    protected void writeWellKnownPathMetadata(String logicalClusterId, Map<String, String> metadata) {
        String path = AzureTenantAwareEncryptionManager.getLastActiveKeyPath(this.config.azureBlobPrefix, logicalClusterId);
        log.info("[{}] Uploading metadata for newly generated key to path {}", (Object)logicalClusterId, (Object)path);
        try {
            BlobClient blobClient = this.blobContainerClient.getBlobClient(path);
            blobClient.getBlockBlobClient().uploadWithResponse(new BufferedInputStream(new ByteBufferInputStream(ByteBuffer.wrap(new byte[0]))), 0L, new BlobHttpHeaders(), metadata, AccessTier.HOT, null, new BlobRequestConditions(), null, null);
        }
        catch (Exception e) {
            if (AzureTenantAwareEncryptionManager.isRetryableAzureException(e)) {
                if (this.metrics != null) {
                    this.metrics.recordWriteKeyRetriableError();
                }
                throw new TierObjectStoreRetriableException(String.format("[%s] Failed to write metadata to well known path at %s", logicalClusterId, path), e);
            }
            if (this.metrics != null) {
                this.metrics.recordWriteKeyFatalError();
            }
            throw new TierObjectStoreFatalException(String.format("[%s] Unknown exception when writing metadata to well known path at %s", logicalClusterId, path), e);
        }
    }

    private static boolean isRetryableAzureException(Exception e) {
        if (e instanceof HttpResponseException) {
            int statusCode = ((HttpResponseException)e).getResponse().getStatusCode();
            return statusCode != 501 && statusCode != 505 && (statusCode >= 500 || statusCode == 408 || statusCode == 429);
        }
        return false;
    }

    @Override
    protected CleartextDataKey decryptKey(String logicalClusterId, EncryptedDataKey encryptedDataKey, ByokMetadata byokMetadata) {
        UnwrapResult decryptResult;
        String clientId = byokMetadata.clientId();
        String keyId = byokMetadata.keyId();
        String tenantId = byokMetadata.tenantId();
        String vaultId = byokMetadata.vaultId();
        byte[] encryptedDataKeyBytes = encryptedDataKey.keyMaterial();
        try {
            CryptographyClient cryptographyClient = this.cryptographyClientSupplier.get(logicalClusterId, clientId, keyId, tenantId, vaultId);
            long decryptStart = this.time.hiResClockMs();
            decryptResult = cryptographyClient.unwrapKey(ENCRYPTION_ALGORITHM_AZURE, encryptedDataKeyBytes);
            if (this.metrics != null) {
                this.metrics.recordDecryptCall(this.time.hiResClockMs() - decryptStart);
            }
        }
        catch (Exception e) {
            if (AzureTenantAwareEncryptionManager.isRetryableAzureException(e)) {
                if (this.metrics != null) {
                    this.metrics.recordDecryptRetriableError();
                }
                throw new TierObjectStoreRetriableException(String.format("[%s] Failed to decrypt key due to a retryable failure in decrypt call to Azure CryptographyClient.", logicalClusterId), e);
            }
            if (this.metrics != null) {
                this.metrics.recordDecryptFatalError();
            }
            throw new TierObjectStoreFatalException(String.format("[%s] Failed to decrypt key due to a non-retryable failure in decrypt call to Azure CryptographyClient.", logicalClusterId), e);
        }
        byte[] decryptedKeyBytes = decryptResult.getKey();
        return new CleartextDataKey(decryptedKeyBytes);
    }

    @Override
    protected DataEncryptionKeyHolder encryptKey(String logicalClusterId, CleartextDataKey dataKey, ByokMetadata byokMetadata) {
        WrapResult encryptResult;
        String clientId = byokMetadata.clientId();
        String keyId = byokMetadata.keyId();
        String tenantId = byokMetadata.tenantId();
        String vaultId = byokMetadata.vaultId();
        try {
            CryptographyClient cryptographyClient = this.cryptographyClientSupplier.get(logicalClusterId, clientId, keyId, tenantId, vaultId);
            long encryptStart = this.time.hiResClockMs();
            encryptResult = cryptographyClient.wrapKey(ENCRYPTION_ALGORITHM_AZURE, dataKey.rawKeyMaterial());
            if (this.metrics != null) {
                this.metrics.recordEncryptCall(this.time.hiResClockMs() - encryptStart);
            }
        }
        catch (Exception e) {
            if (AzureTenantAwareEncryptionManager.isRetryableAzureException(e)) {
                if (this.metrics != null) {
                    this.metrics.recordEncryptRetriableError();
                }
                throw new TierObjectStoreRetriableException(String.format("[%s] Failed to generate encrypted key due to a retryable failure in encrypt call to Azure CryptographyClient.", logicalClusterId), e);
            }
            if (this.metrics != null) {
                this.metrics.recordEncryptFatalError();
            }
            throw new TierObjectStoreFatalException(String.format("[%s] Failed to generate encrypted key due to a non-retryable failure in encrypt call to Azure CryptographyClient.", logicalClusterId), e);
        }
        byte[] encryptedKeyBytes = encryptResult.getEncryptedKey();
        return new DataEncryptionKeyHolder(new EncryptedDataKey(encryptedKeyBytes), dataKey, Optional.of(Instant.ofEpochMilli(this.getTime().milliseconds())));
    }

    public static class DefaultAzureCryptographyClientSupplier
    implements AzureCryptographyClientSupplier {
        private final HttpClient httpClient;
        private static final String TOKEN_ID_FILE_ENV = "AZURE_FEDERATED_TOKEN_FILE";

        public DefaultAzureCryptographyClientSupplier() {
            HttpClientOptions clientOptions = new HttpClientOptions().setMaximumConnectionPoolSize(10);
            this.httpClient = HttpClient.createDefault(clientOptions);
        }

        @Override
        public CryptographyClient get(String logicalClusterId, String clientId, String keyId, String tenantId, String vaultId) {
            String tokenFilePath = System.getenv(TOKEN_ID_FILE_ENV);
            if (tokenFilePath == null) {
                throw new RuntimeException(String.format("[%s] Environment variable %s is not set.", logicalClusterId, TOKEN_ID_FILE_ENV));
            }
            WorkloadIdentityCredential workloadIdentityCredential = ((WorkloadIdentityCredentialBuilder)((WorkloadIdentityCredentialBuilder)new WorkloadIdentityCredentialBuilder().clientId(clientId)).tenantId(tenantId)).tokenFilePath(tokenFilePath).build();
            return new CryptographyClientBuilder().keyIdentifier(keyId).credential(workloadIdentityCredential).httpClient(this.httpClient).buildClient();
        }

        @Override
        public void close() {
        }
    }

    public static interface AzureCryptographyClientSupplier
    extends Closeable {
        public CryptographyClient get(String var1, String var2, String var3, String var4, String var5);
    }
}

