/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.tools.migrations.util;

import com.google.common.annotations.VisibleForTesting;
import io.confluent.ksql.tools.migrations.MigrationConfig;
import io.confluent.ksql.tools.migrations.MigrationException;
import io.confluent.ksql.tools.migrations.util.MigrationFile;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public final class MigrationsDirectoryUtil {
    public static final String MIGRATIONS_DIR = "migrations";
    public static final String MIGRATIONS_CONFIG_FILE = "ksql-migrations.properties";
    private static final Pattern MIGRATION_FILE_MATCHER = Pattern.compile("V([0-9]{6})__(.+)\\.sql");
    private static final Logger LOGGER = LogManager.getLogger(MigrationsDirectoryUtil.class);

    private MigrationsDirectoryUtil() {
    }

    public static String getMigrationsDir(String configFilePath, MigrationConfig config) {
        String migrationsDir = config.getString("ksql.migrations.dir.override");
        if (migrationsDir != null && !migrationsDir.isEmpty()) {
            return migrationsDir;
        }
        return MigrationsDirectoryUtil.getMigrationsDirFromConfigFile(configFilePath);
    }

    @VisibleForTesting
    public static String getMigrationsDirFromConfigFile(String configFilePath) {
        Path parentDir = Paths.get(configFilePath, new String[0]).getParent();
        if (parentDir == null) {
            throw new MigrationException("Could not find parent directory for config file '" + configFilePath + "': no parent dir exists.");
        }
        return parentDir.resolve(MIGRATIONS_DIR).toString();
    }

    public static String getFilePrefixForVersion(String version) {
        return "V" + StringUtils.leftPad((String)version, (int)6, (String)"0");
    }

    public static Optional<MigrationFile> getMigrationForVersion(String version, String migrationsDir) {
        String prefix = MigrationsDirectoryUtil.getFilePrefixForVersion(version);
        File directory = new File(migrationsDir);
        if (!directory.isDirectory()) {
            throw new MigrationException(migrationsDir + " is not a directory.");
        }
        String[] names = directory.list();
        if (names == null) {
            throw new MigrationException("Failed to retrieve files from " + migrationsDir);
        }
        List<MigrationFile> matches = Arrays.stream(names).filter(name -> name.startsWith(prefix)).map(name -> MigrationsDirectoryUtil.getMigrationFromFilename(migrationsDir, name)).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
        MigrationsDirectoryUtil.validateMigrationVersionsUnique(matches);
        return matches.size() > 0 ? Optional.of(matches.get(0)) : Optional.empty();
    }

    public static String getFileContentsForName(String filename) {
        try {
            return new String(Files.readAllBytes(Paths.get(filename, new String[0])), StandardCharsets.UTF_8);
        }
        catch (IOException e) {
            throw new MigrationException(String.format("Failed to read %s: %s", filename, e.getMessage()));
        }
    }

    public static String computeHashForFile(String filename) {
        try {
            byte[] bytes = Files.readAllBytes(Paths.get(filename, new String[0]));
            StringBuilder builder = new StringBuilder();
            for (byte b : MessageDigest.getInstance("MD5").digest(bytes)) {
                builder.append(String.format("%02x", b));
            }
            return builder.toString();
        }
        catch (IOException | NoSuchAlgorithmException e) {
            throw new MigrationException(String.format("Could not compute hash for file '%s': %s", filename, e.getMessage()));
        }
    }

    public static List<Integer> getAllVersions(String migrationsDir) {
        return MigrationsDirectoryUtil.getAllMigrations(migrationsDir).stream().map(MigrationFile::getVersion).collect(Collectors.toList());
    }

    public static List<MigrationFile> getAllMigrations(String migrationsDir) {
        String[] allNames;
        File directory = new File(migrationsDir);
        if (!directory.isDirectory()) {
            throw new MigrationException(migrationsDir + " is not a directory.");
        }
        try {
            allNames = directory.list();
        }
        catch (SecurityException e) {
            throw new MigrationException("Failed to retrieve files from " + migrationsDir + ": " + e.getMessage());
        }
        if (allNames == null) {
            throw new MigrationException("Failed to retrieve files from " + migrationsDir);
        }
        List filenames = Arrays.stream(allNames).sorted().filter(name -> !new File((String)name).isDirectory()).collect(Collectors.toList());
        List<MigrationFile> migrations = filenames.stream().map(name -> MigrationsDirectoryUtil.getMigrationFromFilename(migrationsDir, name)).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
        MigrationsDirectoryUtil.validateMigrationVersionsUnique(migrations);
        return migrations;
    }

    private static Optional<MigrationFile> getMigrationFromFilename(String migrationsDir, String filename) {
        Matcher matcher = MIGRATION_FILE_MATCHER.matcher(filename);
        if (!matcher.find()) {
            LOGGER.warn("Skipping file does not match expected migration file pattern 'V<six digit number>__<name>.sql': {}", (Object)filename);
            return Optional.empty();
        }
        int version = Integer.parseInt(matcher.group(1));
        if (version <= 0) {
            throw new MigrationException("MigrationFile file versions must be positive. Found: " + filename);
        }
        String description = matcher.group(2).replace('_', ' ');
        return Optional.of(new MigrationFile(version, description, Paths.get(migrationsDir, filename).toString()));
    }

    private static void validateMigrationVersionsUnique(List<MigrationFile> migrations) {
        if (migrations.size() == 0) {
            return;
        }
        MigrationFile previous = migrations.get(0);
        for (int i = 1; i < migrations.size(); ++i) {
            if (migrations.get(i).getVersion() == previous.getVersion()) {
                throw new MigrationException(String.format("Found multiple migration files with the same version. Version: %d. Files: '%s', '%s'", migrations.get(i).getVersion(), migrations.get(i).getFilepath(), previous.getFilepath()));
            }
            previous = migrations.get(i);
        }
    }
}

