/*
 * Decompiled with CFR 0.152.
 */
package kafka.server.resource;

import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryNotificationInfo;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.MemoryType;
import java.lang.management.MemoryUsage;
import java.util.Optional;
import javax.management.ListenerNotFoundException;
import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.openmbean.CompositeData;
import kafka.server.BrokerReconfigurable;
import kafka.server.KafkaConfig;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.CumulativeCount;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.collection.Set;
import scala.jdk.javaapi.CollectionConverters;

public class HeapWatcher
implements BrokerReconfigurable,
NotificationFilter,
NotificationListener {
    private static final Logger LOG = LoggerFactory.getLogger(HeapWatcher.class);
    private static final String METRICS_GROUP_NAME = HeapWatcher.class.getSimpleName();
    final MetricName almostOOMMetricName;
    private final Metrics metrics;
    private final Sensor almostOOMSensor;
    private final MemoryPoolMXBean memoryPoolMxBean;
    private long usageThresholdBytes;
    private boolean registered;
    public static Set<String> reconfigurableConfigs = CollectionConverters.asScala(java.util.Set.of("confluent.heap.tenured.notify.bytes"));

    public static Optional<HeapWatcher> maybeCreateHeapWatcher(Metrics metrics, KafkaConfig kafkaConfig) {
        if (!HeapWatcher.isEnabledFromConfig(kafkaConfig)) {
            LOG.debug("HeapWatcher disabled in config");
            return Optional.empty();
        }
        HeapWatcher newHeapWatcher = new HeapWatcher(metrics, kafkaConfig);
        return Optional.of(newHeapWatcher);
    }

    protected HeapWatcher(Metrics metrics, KafkaConfig kafkaConfig) throws IllegalArgumentException, ConfigException {
        LOG.trace("Creating HeapWatcher");
        this.registered = false;
        long threshold = HeapWatcher.usageThersholdBytesFromConfig(kafkaConfig);
        this.memoryPoolMxBean = HeapWatcher.getTenuredPoolMXBean();
        if (this.memoryPoolMxBean == null) {
            LOG.error("Unable to find the MemoryPoolMXBean for the tenured pool, not creating a HeapWatcher");
            throw new IllegalStateException("Unable to find the MemoryPoolMXBean for the tenured pool, not creating a HeapWatcher");
        }
        if (!this.memoryPoolMxBean.isCollectionUsageThresholdSupported()) {
            throw new IllegalArgumentException(String.format("The memory pool supplied (%s) does not support a collection usage threshold", this.memoryPoolMxBean.getName()));
        }
        this.metrics = metrics;
        this.almostOOMSensor = this.metrics.sensor("AlmostOOM");
        this.almostOOMMetricName = metrics.metricName("almost-oom", METRICS_GROUP_NAME, "Number of times an almost-OOM has occurred");
        this.almostOOMSensor.add(this.almostOOMMetricName, new CumulativeCount());
        this.registerWithMemoryMXBean();
        this.setCollectionUsageThreshold(threshold);
    }

    private synchronized void registerWithMemoryMXBean() {
        MemoryMXBean memoryMXBean = ManagementFactory.getMemoryMXBean();
        NotificationEmitter emitter = (NotificationEmitter)((Object)memoryMXBean);
        LOG.debug("Added HeapWatcher as a listener for the tenured heap MemoryMXBean");
        emitter.addNotificationListener(this, this, null);
        this.registered = true;
    }

    synchronized void unregisterWithMemoryMXBean() {
        if (!this.registered) {
            LOG.error("unregisterWithMemoryMXBean() called but HeapWatcher wasn't registered with the MemoryMXBean");
            return;
        }
        MemoryMXBean memoryMXBean = ManagementFactory.getMemoryMXBean();
        NotificationEmitter emitter = (NotificationEmitter)((Object)memoryMXBean);
        try {
            emitter.removeNotificationListener(this, this, null);
            LOG.debug("Removed HeapWatcher as a listener for the tenured heap MemoryMXBean");
            this.registered = false;
        }
        catch (ListenerNotFoundException e) {
            LOG.warn("removeNotificationListener from MemoryMXBean failed. Maybe it was never registered?", (Throwable)e);
        }
    }

    private synchronized void setCollectionUsageThreshold(long threshold) {
        long maxThreshold;
        if (!this.registered) {
            LOG.error("Trying to setCollectionUsageThreshold before registering with the MemoryMXBean. Ignoring");
        }
        if ((maxThreshold = this.memoryPoolMxBean.getUsage().getMax()) > -1L && threshold > maxThreshold) {
            LOG.info("Asked to set collection usage threshold to {}, more than the max available to the pool {}. Disabling notifications insetad", (Object)threshold, (Object)maxThreshold);
            threshold = 0L;
        }
        LOG.info("Setting collection usage threshold to {}", (Object)threshold);
        this.memoryPoolMxBean.setCollectionUsageThreshold(threshold);
        this.usageThresholdBytes = this.memoryPoolMxBean.getCollectionUsageThreshold();
        if (this.usageThresholdBytes != threshold) {
            LOG.error("Asked the MemoryPoolMXBean to set collection threshold to {}, but it was set to {} instead", (Object)this.usageThresholdBytes, (Object)threshold);
        }
    }

    private static long usageThersholdBytesFromConfig(KafkaConfig kafkaConfig) throws ConfigException {
        Long value = kafkaConfig.confluentConfig().heapWatcherTenuredNotifyBytes();
        if (value == null) {
            return 0L;
        }
        if (value < 0L) {
            throw new ConfigException(String.format("Value for %s is %d, a negative number", "confluent.heap.tenured.notify.bytes", value));
        }
        return value;
    }

    private static boolean isEnabledFromConfig(KafkaConfig kafkaConfig) throws ConfigException {
        Boolean value = kafkaConfig.confluentConfig().heapWatcherTenuredEnabled();
        if (value == null) {
            return false;
        }
        return value;
    }

    private static boolean isTenuredPoolName(String name) {
        return name.contains("Old") || name.contains("Tenured");
    }

    static MemoryPoolMXBean getTenuredPoolMXBean() {
        for (MemoryPoolMXBean poolBean : ManagementFactory.getMemoryPoolMXBeans()) {
            if (!poolBean.getType().equals((Object)MemoryType.HEAP) || !HeapWatcher.isTenuredPoolName(poolBean.getName())) continue;
            return poolBean;
        }
        return null;
    }

    private void notifyListeners() {
        this.almostOOMSensor.record();
    }

    @Override
    public boolean isNotificationEnabled(Notification notification) {
        if (!notification.getType().equals("java.management.memory.collection.threshold.exceeded")) {
            LOG.trace("Asked about notification that isn't MEMORY_COLLECTION_THRESHOLD_EXCEEDED: {}", (Object)notification);
            return false;
        }
        MemoryNotificationInfo info = MemoryNotificationInfo.from((CompositeData)notification.getUserData());
        LOG.debug("Filtering MEMORY_COLLECTION_THRESHOLD_EXCEEDED notification with payload: {}", (Object)info);
        return HeapWatcher.isTenuredPoolName(info.getPoolName());
    }

    @Override
    public void handleNotification(Notification notification, Object handback) {
        if (LOG.isDebugEnabled()) {
            MemoryNotificationInfo info = MemoryNotificationInfo.from((CompositeData)notification.getUserData());
            LOG.debug("Handling notification with payload: {}", (Object)info);
        }
        MemoryUsage usage = this.memoryPoolMxBean.getCollectionUsage();
        long usedMemory = usage.getUsed();
        LOG.info("Got notified about the tenured pool, collection usage memory being {} (notification threshold is {})", (Object)usedMemory, (Object)this.usageThresholdBytes);
        if (usedMemory > this.usageThresholdBytes) {
            this.notifyListeners();
        }
    }

    public synchronized void shutdown() {
        this.unregisterWithMemoryMXBean();
        this.metrics.removeSensor(this.almostOOMSensor.name());
    }

    @Override
    public Set<String> reconfigurableConfigs() {
        return reconfigurableConfigs;
    }

    @Override
    public void validateReconfiguration(KafkaConfig newConfig) {
        HeapWatcher.usageThersholdBytesFromConfig(newConfig);
    }

    @Override
    public void reconfigure(KafkaConfig oldConfig, KafkaConfig newConfig) {
        Long newThreshold = HeapWatcher.usageThersholdBytesFromConfig(newConfig);
        if (newThreshold != this.usageThresholdBytes) {
            this.setCollectionUsageThreshold(newThreshold);
        }
    }
}

