/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.databalancing.metric;

import com.google.protobuf.Message;
import io.confluent.kafka.databalancing.RebalancerConfig;
import io.confluent.kafka.databalancing.Utils;
import io.confluent.kafka.databalancing.metric.LogSizeMetricReader;
import io.confluent.kafka.databalancing.metric.Metrics;
import io.confluent.kafka.databalancing.topology.Broker;
import io.confluent.metrics.record.ConfluentMetric;
import io.confluent.serializers.ProtoSerde;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndTimestamp;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.serialization.Deserializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MetricsCollector {
    public static final Logger LOG = LoggerFactory.getLogger(MetricsCollector.class);
    public static final int CONSUMER_REPLAY_PERIOD_MS = 30000;
    public static final int METRIC_TOPIC_METADATA_POLL_RETRY_MS = 200;
    private final String metricsTopic;
    private final int collectionTimeoutMs;
    private final Properties consumerProps;

    public MetricsCollector(String metricsTopic, int collectionTimeoutMs, Properties consumerProps) {
        this.metricsTopic = metricsTopic;
        this.collectionTimeoutMs = collectionTimeoutMs;
        this.consumerProps = consumerProps;
    }

    public MetricsCollector(RebalancerConfig config) {
        this(config.getString("confluent.rebalancer.metrics.topic"), config.getInt("confluent.rebalancer.metrics.collection.timeout.ms"), config.consumerProps());
    }

    public Metrics collectMetrics(Set<TopicPartition> partitions, String clusterId, Set<Broker> brokers) throws TimeoutException {
        if (clusterId == null) {
            throw new IllegalArgumentException("Did not receive a cluster id to filter on. Please ensure that you are using Kafka >= 0.10.1, and that you have some brokers running in the cluster.");
        }
        HashMap<TopicPartition, Long> logSizes = new HashMap<TopicPartition, Long>();
        HashSet<TopicPartition> partitionsToCollect = new HashSet<TopicPartition>(partitions);
        HashMap<Broker, List<ConfluentMetric.VolumeMetrics>> volumeMetrics = new HashMap<Broker, List<ConfluentMetric.VolumeMetrics>>();
        HashSet<Broker> brokersToCollect = new HashSet<Broker>(brokers);
        this.consumerProps.put("enable.auto.commit", "false");
        if (!this.consumerProps.containsKey("group.id")) {
            this.consumerProps.put("group.id", "confluent-rebalancer-metrics-consumer");
        }
        if (!this.consumerProps.containsKey("metadata.max.age.ms")) {
            this.consumerProps.put("metadata.max.age.ms", Integer.toString(this.collectionTimeoutMs / 4));
        }
        if (!this.consumerProps.containsKey("request.timeout.ms")) {
            this.consumerProps.put("request.timeout.ms", "20000");
        }
        try (KafkaConsumer consumer = new KafkaConsumer(this.consumerProps, (Deserializer)new ByteArrayDeserializer(), (Deserializer)new ByteArrayDeserializer());){
            ProtoSerde serdes = new ProtoSerde((Message)ConfluentMetric.MetricsMessage.getDefaultInstance());
            LogSizeMetricReader logSizeMetricReader = new LogSizeMetricReader();
            long startTime = System.currentTimeMillis();
            Collection<TopicPartition> topicPartitions = MetricsCollector.partitionsForTopic((KafkaConsumer<byte[], byte[]>)consumer, this.metricsTopic, this.collectionTimeoutMs);
            consumer.assign(topicPartitions);
            long time = System.currentTimeMillis() - 30000L;
            MetricsCollector.seekToOffsetsByTime((KafkaConsumer<byte[], byte[]>)consumer, topicPartitions, time);
            LOG.info("Going to wait for at most {} ms to collect metrics for {} partitions and {} brokers.", new Object[]{this.collectionTimeoutMs, partitionsToCollect.size(), brokersToCollect.size()});
            LOG.debug("Collecting metrics for the following partitions: {}", partitions);
            LOG.debug("Collecting metrics for the following brokers: {}", brokers);
            while (!partitionsToCollect.isEmpty() || !brokersToCollect.isEmpty()) {
                if ((long)this.collectionTimeoutMs < System.currentTimeMillis() - startTime) {
                    if (partitionsToCollect.isEmpty()) {
                        break;
                    }
                    throw new TimeoutException(String.format("Failed to collect metrics for %d out of %d partitions within %d ms. Partitions with missing metrics: %s", partitionsToCollect.size(), partitions.size(), this.collectionTimeoutMs, partitionsToCollect));
                }
                ConsumerRecords records = consumer.poll(Duration.ofMillis(100L));
                for (ConsumerRecord record : records) {
                    ConfluentMetric.MetricsMessage message = (ConfluentMetric.MetricsMessage)serdes.deserialize((byte[])record.value());
                    if (!message.getClusterId().equals(clusterId)) continue;
                    Map<TopicPartition, Long> currentLogSizes = logSizeMetricReader.metricValues(message);
                    for (TopicPartition topicPartition : currentLogSizes.keySet()) {
                        if (!partitionsToCollect.remove(topicPartition) && !logSizes.containsKey(topicPartition)) continue;
                        logSizes.put(topicPartition, currentLogSizes.get(topicPartition));
                    }
                    Broker broker = new Broker(message.getBrokerId());
                    List volumes = message.getSystemMetrics().getVolumesList();
                    if (volumes.isEmpty() || !brokersToCollect.remove(broker) && !volumeMetrics.containsKey(broker)) continue;
                    volumeMetrics.put(broker, volumes);
                }
            }
        }
        Metrics metrics = new Metrics(logSizes, volumeMetrics);
        LOG.debug("Collected metrics: {}", (Object)metrics);
        return metrics;
    }

    private static Collection<TopicPartition> partitionsForTopic(KafkaConsumer<byte[], byte[]> consumer, String topic, int collectionTimeoutMs) throws TimeoutException {
        long startTime = System.currentTimeMillis();
        List partitions = consumer.partitionsFor(topic);
        while (partitions == null || partitions.isEmpty()) {
            try {
                Thread.sleep(200L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            if ((long)collectionTimeoutMs < System.currentTimeMillis() - startTime) {
                throw new TimeoutException("No partition information received for " + topic + " in " + collectionTimeoutMs + " ms. Aborting.");
            }
            partitions = consumer.partitionsFor(topic);
        }
        ArrayList<TopicPartition> topicPartitions = new ArrayList<TopicPartition>();
        for (PartitionInfo partitionInfo : partitions) {
            topicPartitions.add(new TopicPartition(topic, partitionInfo.partition()));
        }
        return topicPartitions;
    }

    private static void seekToOffsetsByTime(KafkaConsumer<byte[], byte[]> consumer, Collection<TopicPartition> topicPartitions, long time) {
        HashMap<TopicPartition, Long> partitionToTimestamp = new HashMap<TopicPartition, Long>();
        for (TopicPartition topicPartition : topicPartitions) {
            partitionToTimestamp.put(topicPartition, time);
        }
        Map offsets = consumer.offsetsForTimes(partitionToTimestamp);
        HashSet<TopicPartition> partitionsWithNoOffset = new HashSet<TopicPartition>();
        for (Map.Entry topicAndOffset : offsets.entrySet()) {
            TopicPartition topicPartition = (TopicPartition)topicAndOffset.getKey();
            OffsetAndTimestamp offsetAndTimestamp = (OffsetAndTimestamp)topicAndOffset.getValue();
            if (offsetAndTimestamp == null) {
                partitionsWithNoOffset.add(topicPartition);
                consumer.seekToEnd(Collections.singletonList(topicPartition));
                continue;
            }
            long offset = offsetAndTimestamp.offset();
            consumer.seek(topicPartition, offset);
        }
        if (!partitionsWithNoOffset.isEmpty()) {
            LOG.debug("Seeking to the last offset for partitions with no metrics message containing a timestamp greater than or equal to {}. The partitions are: {}.", (Object)new Date(time), (Object)Utils.mkString(partitionsWithNoOffset, ","));
        }
    }
}

