/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.databalancing.throttle;

import io.confluent.kafka.databalancing.AlterTopicConfigsEntry;
import io.confluent.kafka.databalancing.RebalancerAdmin;
import io.confluent.kafka.databalancing.exception.ValidationException;
import io.confluent.kafka.databalancing.throttle.Throttle;
import io.confluent.kafka.databalancing.topology.Broker;
import io.confluent.kafka.databalancing.topology.ClusterReassignment;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Throttler
implements Throttle {
    private static final Logger logger = LoggerFactory.getLogger(Throttler.class);
    static final String LEADER_RATE_PROP = "leader.replication.throttled.rate";
    static final String FOLLOWER_RATE_PROP = "follower.replication.throttled.rate";
    static final String LEADER_REPLICAS_PROP = "leader.replication.throttled.replicas";
    static final String FOLLOWER_REPLICAS_PROP = "follower.replication.throttled.replicas";
    private final RebalancerAdmin admin;

    public Throttler(RebalancerAdmin rebalancerAdmin) {
        this.admin = rebalancerAdmin;
    }

    @Override
    public void engage(long quota, Set<Integer> liveBrokers, ClusterReassignment reassignment) {
        if (this.admin.reassignPartitionsInProgress()) {
            Collection brokers = reassignment.brokers().stream().filter(broker -> liveBrokers.contains(broker.id())).collect(Collectors.toList());
            this.limit(quota, brokers);
        } else {
            this.disengage();
            this.limitAndThrottle(quota, liveBrokers, reassignment);
        }
    }

    void limitAndThrottle(long quota, Set<Integer> liveBrokers, ClusterReassignment reassignment) {
        Collection brokers = reassignment.brokers().stream().filter(broker -> liveBrokers.contains(broker.id())).collect(Collectors.toList());
        this.limit(quota, brokers);
        this.updateThrottledReplicas(reassignment, this::replace);
    }

    @Override
    public boolean disengage() {
        return this.removeReplicas() | this.removeRates();
    }

    @Override
    public void limit(long quota, Collection<Broker> brokers) {
        List<Integer> brokerIds = brokers.stream().map(Broker::id).collect(Collectors.toList());
        String quotaValue = String.valueOf(quota);
        Map<Integer, Map<String, Optional<String>>> configs = this.admin.fetchBrokerConfigs(brokerIds);
        configs.values().removeIf(properties -> !Throttler.needsUpdate(properties, LEADER_RATE_PROP, quotaValue) && !Throttler.needsUpdate(properties, FOLLOWER_RATE_PROP, quotaValue));
        if (configs.isEmpty()) {
            return;
        }
        HashMap<String, String> alterations = new HashMap<String, String>(2);
        alterations.put(LEADER_RATE_PROP, quotaValue);
        alterations.put(FOLLOWER_RATE_PROP, quotaValue);
        this.admin.alterBrokerConfigs(configs, alterations);
        for (Integer brokerId : configs.keySet()) {
            logger.info("Updated broker config for broker [{}] to set [{}={}, {}={}]", new Object[]{brokerId, LEADER_RATE_PROP, quotaValue, FOLLOWER_RATE_PROP, quotaValue});
        }
    }

    @Override
    public void throttleReplicas(ClusterReassignment reassignment) {
        this.updateThrottledReplicas(reassignment, this::union);
    }

    @Override
    public void unthrottleReplicas(ClusterReassignment completed) {
        this.updateThrottledReplicas(completed, this::diff);
    }

    private void updateThrottledReplicas(ClusterReassignment reassignment, BiFunction<Map<TopicPartition, SortedSet<Integer>>, Map<TopicPartition, SortedSet<Integer>>, SortedMap<TopicPartition, SortedSet<Integer>>> join) {
        Map<String, Map<String, Optional<String>>> configs = this.admin.fetchTopicConfigs(reassignment.topics());
        TreeMap<String, Map<String, Optional<String>>> sortedConfigs = new TreeMap<String, Map<String, Optional<String>>>(configs);
        ArrayList<AlterTopicConfigsEntry> alterEntries = new ArrayList<AlterTopicConfigsEntry>();
        for (Map.Entry<String, Map<String, Optional<String>>> entry : sortedConfigs.entrySet()) {
            String topic = entry.getKey();
            Map<String, Optional<String>> properties = entry.getValue();
            String currentThrottledLeadersString = properties.getOrDefault(LEADER_REPLICAS_PROP, Optional.empty()).orElse(null);
            String updatedThrottledLeadersString = this.updatedThrottledReplicasConfig(topic, currentThrottledLeadersString, reassignment.moveSources(topic), join);
            String currentThrottledFollowersString = properties.getOrDefault(FOLLOWER_REPLICAS_PROP, Optional.empty()).orElse(null);
            String updatedThrottledFollowersString = this.updatedThrottledReplicasConfig(topic, currentThrottledFollowersString, reassignment.moveDestinations(topic), join);
            HashMap<String, String> alterations = new HashMap<String, String>(2);
            if (!Objects.equals(currentThrottledLeadersString, updatedThrottledLeadersString)) {
                alterations.put(LEADER_REPLICAS_PROP, updatedThrottledLeadersString.isEmpty() ? null : updatedThrottledLeadersString);
            }
            if (!Objects.equals(currentThrottledFollowersString, updatedThrottledFollowersString)) {
                alterations.put(FOLLOWER_REPLICAS_PROP, updatedThrottledFollowersString.isEmpty() ? null : updatedThrottledFollowersString);
            }
            if (alterations.isEmpty()) continue;
            alterEntries.add(new AlterTopicConfigsEntry(topic, properties, alterations));
        }
        if (!alterEntries.isEmpty()) {
            this.admin.alterTopicConfigs(alterEntries);
        }
    }

    private String updatedThrottledReplicasConfig(String topic, String currentThrottledReplicasString, Map<TopicPartition, SortedSet<Integer>> replicasToThrottle, BiFunction<Map<TopicPartition, SortedSet<Integer>>, Map<TopicPartition, SortedSet<Integer>>, SortedMap<TopicPartition, SortedSet<Integer>>> join) {
        Map<TopicPartition, SortedSet<Integer>> currentThrottledReplicas = this.parseThrottledReplicasConfig(topic, currentThrottledReplicasString);
        SortedMap<TopicPartition, SortedSet<Integer>> updatedThrottledReplicas = join.apply(currentThrottledReplicas, replicasToThrottle);
        return this.format(updatedThrottledReplicas);
    }

    private SortedMap<TopicPartition, SortedSet<Integer>> diff(Map<TopicPartition, SortedSet<Integer>> currentThrottledReplicas, Map<TopicPartition, SortedSet<Integer>> completeThrottledReplicas) {
        TreeMap<TopicPartition, SortedSet<Integer>> sortedMoves = new TreeMap<TopicPartition, SortedSet<Integer>>(Throttler.topicPartitionComparator());
        sortedMoves.putAll(currentThrottledReplicas);
        for (Map.Entry<TopicPartition, SortedSet<Integer>> newReplicaEntry : completeThrottledReplicas.entrySet()) {
            TopicPartition topicPartition = newReplicaEntry.getKey();
            SortedSet replicas = sortedMoves.computeIfAbsent(topicPartition, tp -> new TreeSet());
            replicas.removeAll((Collection)newReplicaEntry.getValue());
        }
        return sortedMoves;
    }

    private SortedMap<TopicPartition, SortedSet<Integer>> union(Map<TopicPartition, SortedSet<Integer>> currentThrottledReplicas, Map<TopicPartition, SortedSet<Integer>> newThrottledReplicas) {
        TreeMap<TopicPartition, SortedSet<Integer>> sortedMoves = new TreeMap<TopicPartition, SortedSet<Integer>>(Throttler.topicPartitionComparator());
        sortedMoves.putAll(currentThrottledReplicas);
        for (Map.Entry<TopicPartition, SortedSet<Integer>> newReplicaEntry : newThrottledReplicas.entrySet()) {
            TopicPartition topicPartition = newReplicaEntry.getKey();
            SortedSet replicas = sortedMoves.computeIfAbsent(topicPartition, tp -> new TreeSet());
            replicas.addAll((Collection)newReplicaEntry.getValue());
        }
        return sortedMoves;
    }

    private SortedMap<TopicPartition, SortedSet<Integer>> replace(Map<TopicPartition, SortedSet<Integer>> currentThrottledReplicas, Map<TopicPartition, SortedSet<Integer>> newThrottledReplicas) {
        TreeMap<TopicPartition, SortedSet<Integer>> sortedMoves = new TreeMap<TopicPartition, SortedSet<Integer>>(Throttler.topicPartitionComparator());
        sortedMoves.putAll(newThrottledReplicas);
        return sortedMoves;
    }

    private boolean removeReplicas() {
        Map<String, Map<String, Optional<String>>> configs = this.admin.fetchTopicConfigs(this.admin.getAllTopicsInCluster(true));
        ArrayList<AlterTopicConfigsEntry> alterEntries = new ArrayList<AlterTopicConfigsEntry>();
        for (Map.Entry<String, Map<String, Optional<String>>> entry : configs.entrySet()) {
            Map<String, Optional<String>> properties = entry.getValue();
            if (!Throttler.needsRemoval(properties, LEADER_REPLICAS_PROP) && !Throttler.needsRemoval(properties, FOLLOWER_REPLICAS_PROP)) continue;
            HashMap<String, String> alterations = new HashMap<String, String>(2);
            alterations.put(LEADER_REPLICAS_PROP, null);
            alterations.put(FOLLOWER_REPLICAS_PROP, null);
            alterEntries.add(new AlterTopicConfigsEntry(entry.getKey(), properties, alterations));
        }
        if (alterEntries.isEmpty()) {
            return false;
        }
        this.admin.alterTopicConfigs(alterEntries);
        return true;
    }

    private boolean removeRates() {
        Set<Integer> brokerIds = this.admin.getAllBrokersInCluster();
        Map<Integer, Map<String, Optional<String>>> configs = this.admin.fetchBrokerConfigs(brokerIds);
        configs.values().removeIf(properties -> !Throttler.needsRemoval(properties, LEADER_RATE_PROP) && !Throttler.needsRemoval(properties, FOLLOWER_RATE_PROP));
        if (configs.isEmpty()) {
            return false;
        }
        HashMap<String, String> alterations = new HashMap<String, String>(2);
        alterations.put(LEADER_RATE_PROP, null);
        alterations.put(FOLLOWER_RATE_PROP, null);
        this.admin.alterBrokerConfigs(configs, alterations);
        for (Integer brokerId : configs.keySet()) {
            logger.info("Updated broker config for broker [{}] to remove [{}, {}]", new Object[]{brokerId, LEADER_RATE_PROP, FOLLOWER_RATE_PROP});
        }
        return true;
    }

    private String format(SortedMap<TopicPartition, SortedSet<Integer>> moves) {
        StringBuilder formatted = new StringBuilder();
        for (Map.Entry<TopicPartition, SortedSet<Integer>> entry : moves.entrySet()) {
            TopicPartition tp = entry.getKey();
            Iterator iterator = entry.getValue().iterator();
            while (iterator.hasNext()) {
                int brokerId = (Integer)iterator.next();
                formatted.append(String.format("%s:%s,", tp.partition(), brokerId));
            }
        }
        if (formatted.length() > 0) {
            formatted.deleteCharAt(formatted.length() - 1);
        }
        return formatted.toString();
    }

    private Map<TopicPartition, SortedSet<Integer>> parseThrottledReplicasConfig(String topic, String throttledReplicasConfigValue) {
        if (throttledReplicasConfigValue == null || throttledReplicasConfigValue.isEmpty()) {
            return Collections.emptyMap();
        }
        HashMap<TopicPartition, SortedSet<Integer>> res = new HashMap<TopicPartition, SortedSet<Integer>>();
        for (String partitionAndReplica : throttledReplicasConfigValue.split(",")) {
            String[] partitionAndReplicaParts = partitionAndReplica.split(":");
            if (partitionAndReplicaParts.length != 2) {
                throw new ValidationException("Illegal throttled replicas found for topic " + topic + ": `" + throttledReplicasConfigValue + "`");
            }
            Integer partitionId = Integer.parseInt(partitionAndReplicaParts[0]);
            Integer brokerId = Integer.parseInt(partitionAndReplicaParts[1]);
            SortedSet replicas = res.computeIfAbsent(new TopicPartition(topic, partitionId.intValue()), tp -> new TreeSet());
            replicas.add(brokerId);
        }
        return res;
    }

    private static boolean needsUpdate(Map<String, Optional<String>> properties, String key, String value) {
        Optional<String> config = properties.get(key);
        return config == null || config.map(configValue -> !Objects.equals(value, configValue)).orElse(true) != false;
    }

    private static boolean needsRemoval(Map<String, Optional<String>> properties, String key) {
        Optional<String> config = properties.get(key);
        return config != null && config.map(configValue -> !configValue.isEmpty()).orElse(true) != false;
    }

    private static Comparator<TopicPartition> topicPartitionComparator() {
        return new Comparator<TopicPartition>(){

            @Override
            public int compare(TopicPartition tp1, TopicPartition tp2) {
                if (tp1 == null && tp2 == null) {
                    return 0;
                }
                if (tp2 == null) {
                    return -1;
                }
                if (tp1 == null) {
                    return 1;
                }
                int topicComparison = tp1.topic().compareTo(tp2.topic());
                return topicComparison == 0 ? Integer.compare(tp1.partition(), tp2.partition()) : topicComparison;
            }
        };
    }
}

