/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.secretregistry.storage;

import io.confluent.kafka.secretregistry.exceptions.SecretRegistryException;
import io.confluent.kafka.secretregistry.rest.SecretRegistryConfig;
import io.confluent.kafka.secretregistry.storage.KafkaStoreReaderThread;
import io.confluent.kafka.secretregistry.storage.Store;
import io.confluent.kafka.secretregistry.storage.StoreUpdateHandler;
import io.confluent.kafka.secretregistry.storage.exceptions.SerializationException;
import io.confluent.kafka.secretregistry.storage.exceptions.StoreException;
import io.confluent.kafka.secretregistry.storage.exceptions.StoreInitializationException;
import io.confluent.kafka.secretregistry.storage.exceptions.StoreTimeoutException;
import io.confluent.kafka.secretregistry.storage.serialization.Serializer;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.Config;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.admin.TopicDescription;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.KafkaFuture;
import org.apache.kafka.common.TopicPartitionInfo;
import org.apache.kafka.common.config.ConfigResource;
import org.apache.kafka.common.errors.TopicExistsException;
import org.apache.kafka.common.serialization.ByteArraySerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaStore<K, V>
implements Store<K, V> {
    private static final Logger log = LoggerFactory.getLogger(KafkaStore.class);
    private final String topic;
    private final int desiredReplicationFactor;
    private final String groupId;
    private final StoreUpdateHandler<K, V> storeUpdateHandler;
    private final Serializer<K, V> serializer;
    private final Store<K, V> localStore;
    private final AtomicBoolean initialized = new AtomicBoolean(false);
    private final int initTimeout;
    private final int timeout;
    private final String bootstrapBrokers;
    private KafkaProducer<byte[], byte[]> producer;
    private KafkaStoreReaderThread<K, V> kafkaTopicReader;
    private final K noopKey;
    private volatile long lastWrittenOffset = -1L;
    private final SecretRegistryConfig config;

    public KafkaStore(SecretRegistryConfig config, String groupId, StoreUpdateHandler<K, V> storeUpdateHandler, Serializer<K, V> serializer, Store<K, V> localStore, K noopKey) throws SecretRegistryException {
        this.topic = config.getString("kafkastore.topic");
        this.desiredReplicationFactor = config.getInt("kafkastore.topic.replication.factor");
        this.groupId = groupId;
        this.initTimeout = config.getInt("kafkastore.init.timeout.ms");
        this.timeout = config.getInt("kafkastore.timeout.ms");
        this.storeUpdateHandler = storeUpdateHandler;
        this.serializer = serializer;
        this.localStore = localStore;
        this.noopKey = noopKey;
        this.config = config;
        this.bootstrapBrokers = config.bootstrapBrokers();
        log.info("Initializing KafkaStore with broker endpoints: " + this.bootstrapBrokers);
    }

    @Override
    public void init() throws StoreInitializationException {
        if (this.initialized.get()) {
            throw new StoreInitializationException("Illegal state while initializing store. Store was already initialized");
        }
        this.localStore.init();
        this.createOrVerifySecretTopic();
        Properties props = new Properties();
        KafkaStore.addSecretRegistryConfigsToClientProperties(this.config, props);
        props.put("bootstrap.servers", this.bootstrapBrokers);
        props.put("acks", "-1");
        props.put("key.serializer", ByteArraySerializer.class);
        props.put("value.serializer", ByteArraySerializer.class);
        props.put("retries", (Object)0);
        this.producer = new KafkaProducer(props);
        this.kafkaTopicReader = new KafkaStoreReaderThread<K, V>(this.bootstrapBrokers, this.topic, this.groupId, this.storeUpdateHandler, this.serializer, this.localStore, (Producer<byte[], byte[]>)this.producer, this.noopKey, this.config);
        this.kafkaTopicReader.readToEnd();
        this.kafkaTopicReader.start();
        boolean isInitialized = this.initialized.compareAndSet(false, true);
        if (!isInitialized) {
            throw new StoreInitializationException("Illegal state while initializing store. Store was already initialized");
        }
    }

    public static void addSecretRegistryConfigsToClientProperties(SecretRegistryConfig config, Properties props) {
        props.putAll((Map<?, ?>)config.originalsWithPrefix("kafkastore."));
    }

    private void createOrVerifySecretTopic() throws StoreInitializationException {
        Properties props = new Properties();
        KafkaStore.addSecretRegistryConfigsToClientProperties(this.config, props);
        props.put("bootstrap.servers", this.bootstrapBrokers);
        try (AdminClient admin = AdminClient.create((Properties)props);){
            Set allTopics = (Set)admin.listTopics().names().get((long)this.initTimeout, TimeUnit.MILLISECONDS);
            if (allTopics.contains(this.topic)) {
                this.verifySecretTopic(admin);
            } else {
                this.createSecretTopic(admin);
            }
        }
        catch (TimeoutException e) {
            throw new StoreInitializationException("Timed out trying to create or validate secret topic configuration", e);
        }
        catch (InterruptedException | ExecutionException e) {
            throw new StoreInitializationException("Failed trying to create or validate secret topic configuration", e);
        }
    }

    private void createSecretTopic(AdminClient admin) throws StoreInitializationException, InterruptedException, ExecutionException, TimeoutException {
        log.info("Creating secrets topic {}", (Object)this.topic);
        int numLiveBrokers = ((Collection)admin.describeCluster().nodes().get((long)this.initTimeout, TimeUnit.MILLISECONDS)).size();
        if (numLiveBrokers <= 0) {
            throw new StoreInitializationException("No live Kafka brokers");
        }
        int secretTopicReplicationFactor = Math.min(numLiveBrokers, this.desiredReplicationFactor);
        if (secretTopicReplicationFactor < this.desiredReplicationFactor) {
            log.warn("Creating the secret topic " + this.topic + " using a replication factor of " + secretTopicReplicationFactor + ", which is less than the desired one of " + this.desiredReplicationFactor + ". If this is a production environment, it's crucial to add more brokers and increase the replication factor of the topic.");
        }
        NewTopic secretTopicRequest = new NewTopic(this.topic, 1, (short)secretTopicReplicationFactor);
        HashMap<String, String> topicConfigs = new HashMap<String, String>(this.config.originalsWithPrefix("kafkastore.topic.config."));
        topicConfigs.put("cleanup.policy", "compact");
        secretTopicRequest.configs(topicConfigs);
        try {
            admin.createTopics(Collections.singleton(secretTopicRequest)).all().get((long)this.initTimeout, TimeUnit.MILLISECONDS);
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof TopicExistsException) {
                this.verifySecretTopic(admin);
            }
            throw e;
        }
    }

    private void verifySecretTopic(AdminClient admin) throws StoreInitializationException, InterruptedException, ExecutionException, TimeoutException {
        ConfigResource topicResource;
        Map configs;
        Config topicConfigs;
        String retentionPolicy;
        log.info("Validating secrets topic {}", (Object)this.topic);
        Set<String> topics = Collections.singleton(this.topic);
        Map topicFutureMap = admin.describeTopics(topics).topicNameValues();
        HashMap<String, TopicDescription> topicDescription = new HashMap<String, TopicDescription>();
        for (Map.Entry entry : topicFutureMap.entrySet()) {
            topicDescription.put((String)entry.getKey(), (TopicDescription)((KafkaFuture)entry.getValue()).get((long)this.initTimeout, TimeUnit.MILLISECONDS));
        }
        TopicDescription description = (TopicDescription)topicDescription.get(this.topic);
        int numPartitions = description.partitions().size();
        if (numPartitions != 1) {
            throw new StoreInitializationException("The secret topic " + this.topic + " should have only 1 partition but has " + numPartitions);
        }
        if (((TopicPartitionInfo)description.partitions().get(0)).replicas().size() < this.desiredReplicationFactor) {
            log.warn("The replication factor of the secret topic " + this.topic + " is less than the desired one of " + this.desiredReplicationFactor + ". If this is a production environment, it's crucial to add more brokers and increase the replication factor of the topic.");
        }
        if ((retentionPolicy = (topicConfigs = (Config)(configs = (Map)admin.describeConfigs(Collections.singleton(topicResource = new ConfigResource(ConfigResource.Type.TOPIC, this.topic))).all().get((long)this.initTimeout, TimeUnit.MILLISECONDS)).get(topicResource)).get("cleanup.policy").value()) == null || !"compact".equals(retentionPolicy)) {
            log.error("The retention policy of the secret topic " + this.topic + " is incorrect. You must configure the topic to 'compact' cleanup policy to avoid Kafka deleting your secrets after a week. Refer to Kafka documentation for more details on cleanup policies");
            throw new StoreInitializationException("The retention policy of the secret topic " + this.topic + " is incorrect. Expected cleanup.policy to be 'compact' but it is " + retentionPolicy);
        }
    }

    public void waitUntilKafkaReaderReachesLastOffset(int timeoutMs) throws StoreException {
        long offsetOfLastMessage = this.getLatestOffset(timeoutMs);
        log.info("Wait to catch up until the offset of the last message at " + offsetOfLastMessage);
        this.kafkaTopicReader.waitUntilOffset(offsetOfLastMessage, timeoutMs, TimeUnit.MILLISECONDS);
        log.debug("Reached offset at " + offsetOfLastMessage);
    }

    public void markLastWrittenOffsetInvalid() {
        this.lastWrittenOffset = -1L;
    }

    @Override
    public V get(K key) throws StoreException {
        this.assertInitialized();
        return this.localStore.get(key);
    }

    @Override
    public void put(K key, V value) throws StoreTimeoutException, StoreException {
        this.assertInitialized();
        if (key == null) {
            throw new StoreException("Key should not be null");
        }
        ProducerRecord producerRecord = null;
        try {
            producerRecord = new ProducerRecord(this.topic, Integer.valueOf(0), (Object)this.serializer.serializeKey(key), value == null ? null : this.serializer.serializeValue(value));
        }
        catch (SerializationException e) {
            throw new StoreException("Error serializing secret while creating the Kafka produce record", e);
        }
        boolean knownSuccessfulWrite = false;
        try {
            log.trace("Sending record to KafkaStore topic: " + String.valueOf(producerRecord));
            Future ack = this.producer.send(producerRecord);
            RecordMetadata recordMetadata = (RecordMetadata)ack.get(this.timeout, TimeUnit.MILLISECONDS);
            log.trace("Waiting for the local store to catch up to offset " + recordMetadata.offset());
            this.lastWrittenOffset = recordMetadata.offset();
            this.kafkaTopicReader.waitUntilOffset(this.lastWrittenOffset, this.timeout, TimeUnit.MILLISECONDS);
            knownSuccessfulWrite = true;
        }
        catch (InterruptedException e) {
            throw new StoreException("Put operation interrupted while waiting for an ack from Kafka", e);
        }
        catch (ExecutionException e) {
            throw new StoreException("Put operation failed while waiting for an ack from Kafka", e);
        }
        catch (TimeoutException e) {
            throw new StoreTimeoutException("Put operation timed out while waiting for an ack from Kafka", e);
        }
        catch (KafkaException ke) {
            throw new StoreException("Put operation to Kafka failed", ke);
        }
        finally {
            if (!knownSuccessfulWrite) {
                this.lastWrittenOffset = -1L;
            }
        }
    }

    @Override
    public Iterator<V> getAll(K key1, K key2) throws StoreException {
        this.assertInitialized();
        return this.localStore.getAll(key1, key2);
    }

    @Override
    public void putAll(Map<K, V> entries) throws StoreException {
        this.assertInitialized();
        for (Map.Entry<K, V> entry : entries.entrySet()) {
            this.put(entry.getKey(), entry.getValue());
        }
    }

    @Override
    public void delete(K key) throws StoreException {
        this.assertInitialized();
        this.put(key, null);
    }

    @Override
    public Iterator<K> getAllKeys() throws StoreException {
        this.assertInitialized();
        return this.localStore.getAllKeys();
    }

    @Override
    public void close() {
        if (this.kafkaTopicReader != null) {
            this.kafkaTopicReader.shutdown();
            log.debug("Kafka store reader thread shut down");
        }
        if (this.producer != null) {
            this.producer.close();
            log.debug("Kafka store producer shut down");
        }
        this.localStore.close();
        log.debug("Kafka store shut down complete");
    }

    KafkaStoreReaderThread<K, V> getKafkaStoreReaderThread() {
        return this.kafkaTopicReader;
    }

    private void assertInitialized() throws StoreException {
        if (!this.initialized.get()) {
            throw new StoreException("Illegal state. Store not initialized yet");
        }
    }

    private long getLatestOffset(int timeoutMs) throws StoreException {
        ProducerRecord producerRecord = null;
        if (this.lastWrittenOffset >= 0L) {
            return this.lastWrittenOffset;
        }
        try {
            producerRecord = new ProducerRecord(this.topic, Integer.valueOf(0), (Object)this.serializer.serializeKey(this.noopKey), null);
        }
        catch (SerializationException e) {
            throw new StoreException("Failed to serialize noop key.", e);
        }
        try {
            log.trace("Sending Noop record to KafkaStore to find last offset.");
            Future ack = this.producer.send(producerRecord);
            RecordMetadata metadata = (RecordMetadata)ack.get(timeoutMs, TimeUnit.MILLISECONDS);
            this.lastWrittenOffset = metadata.offset();
            log.trace("Noop record's offset is " + this.lastWrittenOffset);
            return this.lastWrittenOffset;
        }
        catch (Exception e) {
            throw new StoreException("Failed to write Noop record to kafka store.", e);
        }
    }
}

