/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.detector.tenantstriping;

import com.linkedin.kafka.cruisecontrol.common.CellResource;
import com.linkedin.kafka.cruisecontrol.config.KafkaCruiseControlConfig;
import com.linkedin.kafka.cruisecontrol.model.CellResourceLoad;
import com.linkedin.kafka.cruisecontrol.model.CellResourceUsage;
import com.linkedin.kafka.cruisecontrol.model.TenantResource;
import com.linkedin.kafka.cruisecontrol.model.TenantStripingInfo;
import com.linkedin.kafka.cruisecontrol.model.view.CellTenantView;
import io.confluent.databalancer.metrics.TenantStripingMetrics;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CellSelector {
    private static final Logger LOG = LoggerFactory.getLogger(CellSelector.class);
    private final long maxReplicasPerBroker;
    private final double cellLoadUpperBound;
    private List<CellTenantView> readyCells;
    private final int stripingRateLimit;
    private int stripesPlacedForCurrentRound;
    private Map<Integer, CellResourceUsage> cellResourceLoadStats;
    private static final Map<TenantResource, CellResource> cellResourceByTenantResource = CellSelector.getTenantResourceToCellResourceMap();
    private final Map<String, List<Integer>> tenantToCellAssignments;
    private final TenantStripingMetrics tenantStripingMetrics;

    public CellSelector(KafkaCruiseControlConfig kccConfig, TenantStripingMetrics tenantStripingMetrics) {
        this.maxReplicasPerBroker = kccConfig.getLong("max.replicas");
        this.cellLoadUpperBound = kccConfig.getDouble("cell.load.upper.bound");
        this.stripingRateLimit = kccConfig.getInt("tenant.striping.rate.limit");
        this.stripesPlacedForCurrentRound = 0;
        this.tenantToCellAssignments = new HashMap<String, List<Integer>>();
        this.tenantStripingMetrics = tenantStripingMetrics;
    }

    public void updateReadyCells(List<CellTenantView> readyCells) {
        this.readyCells = readyCells;
    }

    public void updateCellResourceLoadStats(Map<Integer, CellResourceUsage> cellResourceLoadStats) {
        this.cellResourceLoadStats = cellResourceLoadStats;
    }

    public void updateTenantStripes(String tenantId, List<Integer> selectedCells) {
        this.tenantToCellAssignments.put(tenantId, selectedCells);
    }

    public Map<String, List<Integer>> getTenantToCellAssignments() {
        return this.tenantToCellAssignments;
    }

    public List<Integer> selectCells(int maxDesiredStripeFactor, TenantStripingInfo tenantStripingInfo) {
        Optional<CellTenantView> selectedCell;
        ArrayList<Integer> stripeDestinationCells = new ArrayList<Integer>();
        int existingStripes = tenantStripingInfo.existingStripes().size();
        int desiredNumberOfStripes = Math.min(this.stripingRateLimit - this.stripesPlacedForCurrentRound, maxDesiredStripeFactor - existingStripes);
        int feasibleStripes = existingStripes + Math.min(desiredNumberOfStripes, this.readyCells.size());
        LOG.debug("For tenant {}, feasible stripe factor is {}, max desired stripe factor: {}, desired number of stripes for the current round: {}, ready cells: {}", new Object[]{tenantStripingInfo.tenantId(), feasibleStripes, maxDesiredStripeFactor, desiredNumberOfStripes, this.readyCells.size()});
        if (!this.isTenantStripingInfoValid(tenantStripingInfo)) {
            LOG.error("Skipping tenant striping for tenant {}. Tenant striping data not valid.", (Object)tenantStripingInfo.tenantId());
            return stripeDestinationCells;
        }
        HashSet<CellTenantView> eligibleCells = new HashSet<CellTenantView>(this.getEligibleDestinationCells(this.readyCells, tenantStripingInfo));
        int stripesToBePlaced = feasibleStripes - existingStripes;
        if (eligibleCells.isEmpty() && stripesToBePlaced > 0) {
            this.tenantStripingMetrics.recordInsufficientCellsAvailable(tenantStripingInfo.tenantId(), desiredNumberOfStripes);
            LOG.debug("No eligible cells found for tenant {}. Existing stripes: {}, stripes needed: {}", new Object[]{tenantStripingInfo.tenantId(), existingStripes, stripesToBePlaced});
            return stripeDestinationCells;
        }
        int stripesPlaced = 0;
        for (int stripes = stripesToBePlaced; stripes > 0 && (selectedCell = this.selectCellForPlacement(eligibleCells, tenantStripingInfo, feasibleStripes)).isPresent(); --stripes) {
            stripeDestinationCells.add(selectedCell.get().id());
            eligibleCells.remove(selectedCell.get());
            ++stripesPlaced;
        }
        this.incrementStripeCount(stripesPlaced);
        int stripesNeededDelta = desiredNumberOfStripes - stripeDestinationCells.size();
        if (stripesNeededDelta > 0) {
            this.tenantStripingMetrics.recordInsufficientCellsAvailable(tenantStripingInfo.tenantId(), stripesNeededDelta);
            LOG.debug("Insufficient cells available for tenant {}. Stripes still needed: {}", (Object)tenantStripingInfo.tenantId(), (Object)stripesNeededDelta);
        } else {
            this.tenantStripingMetrics.clearInsufficientCellsAvailableMetric(tenantStripingInfo.tenantId());
        }
        return stripeDestinationCells;
    }

    private Optional<CellTenantView> selectCellForPlacement(Set<CellTenantView> candidateCells, TenantStripingInfo tenantStripingInfo, int numberOfStripes) {
        double minDestinationCellLoad = Double.MAX_VALUE;
        Optional<CellTenantView> destinationCell = Optional.empty();
        for (CellTenantView candidateCell : candidateCells) {
            CellResourceUsage cellResourceLoads;
            CellTenantView clonedCellTenantView = new CellTenantView(candidateCell);
            double destinationCellLoad = this.calculateLoadWithTenantStripe(clonedCellTenantView, cellResourceLoads = this.cellResourceLoadStats.get(candidateCell.id()), tenantStripingInfo, numberOfStripes);
            if (!(destinationCellLoad < minDestinationCellLoad)) continue;
            LOG.debug("CellSelector found a possible destination cell: {} for tenant: {}. After the move, the load of the destination cell will be {}.", new Object[]{tenantStripingInfo.tenantId(), candidateCell.id(), destinationCellLoad});
            minDestinationCellLoad = destinationCellLoad;
            destinationCell = Optional.of(candidateCell);
        }
        if (destinationCell.isPresent()) {
            LOG.info("CellSelector found a destination cell for tenant {}: cell {}.", (Object)tenantStripingInfo.tenantId(), (Object)((CellTenantView)destinationCell.get()).id());
            if (minDestinationCellLoad > this.cellLoadUpperBound) {
                LOG.info("CellSelector found a destination cell for tenant {}: cell {}. However, the load of the destination cell is above the upper bound {}.", new Object[]{tenantStripingInfo.tenantId(), destinationCell.get().id(), this.cellLoadUpperBound});
            }
            return destinationCell;
        }
        LOG.info("CellSelector didn't find a destination cell for tenant {}.", (Object)tenantStripingInfo.tenantId());
        return Optional.empty();
    }

    double calculateLoadWithTenantStripe(CellTenantView cellTenantView, CellResourceUsage loadByCellResource, TenantStripingInfo tenantStripingInfo, int numberOfStripes) {
        if (tenantStripingInfo.tenantId() == null || tenantStripingInfo.tenantId().isEmpty()) {
            throw new IllegalArgumentException("Tenant ID cannot be null or empty");
        }
        double cellLoad = Math.min(1.0, loadByCellResource.resourceToLoadMap().values().stream().map(CellResourceLoad::ratio).max(Double::compare).orElse(0.0));
        for (TenantResource tenantResource : TenantResource.cachedValues()) {
            Double resourceUsage = tenantStripingInfo.spikedResourceUsages().containsResource(tenantResource) ? tenantStripingInfo.spikedResourceUsages().resourceValue(tenantResource) : tenantStripingInfo.latestResourceUsages().resourceValue(tenantResource);
            Optional<CellResource> cellResourceOpt = this.getCellResource(tenantResource);
            if (cellResourceOpt.isEmpty()) continue;
            CellResource cellResource = cellResourceOpt.get();
            CellResourceLoad cellResourceLoad = loadByCellResource.load(cellResource);
            double perStripeUsage = resourceUsage / (double)numberOfStripes;
            double eligibleDestinationCapacity = cellTenantView.getEligibleCapacity(cellResource, this.maxReplicasPerBroker);
            double cellResourceUsage = cellResourceLoad.ratio() * eligibleDestinationCapacity;
            CellResourceLoad estimatedCellLoadWithTenantStripe = cellTenantView.loadForResource(cellResource, cellResourceUsage + perStripeUsage, this.maxReplicasPerBroker);
            cellLoad = Math.max(cellLoad, estimatedCellLoadWithTenantStripe.ratio());
        }
        return cellLoad;
    }

    List<CellTenantView> getEligibleDestinationCells(List<CellTenantView> availableCells, TenantStripingInfo tenantStripingInfo) {
        return availableCells.stream().filter(cell -> !tenantStripingInfo.existingStripes().contains((Object)cell.id())).filter(cell -> this.isCellResourceLoadInfoValid(cell.id())).filter(cell -> !cell.detectOverload(this.cellLoadUpperBound, this.maxReplicasPerBroker).isOverloaded()).collect(Collectors.toList());
    }

    private boolean isTenantStripingInfoValid(TenantStripingInfo tenantStripingInfo) {
        for (TenantResource tenantResource : TenantResource.cachedValues()) {
            boolean isTenantResourceUsageValid = tenantStripingInfo.spikedResourceUsages().containsResource(tenantResource) || tenantStripingInfo.latestResourceUsages().containsResource(tenantResource);
            if (isTenantResourceUsageValid) continue;
            LOG.error("Missing resource usage data for tenant {} and resource {}. ", (Object)tenantStripingInfo.tenantId(), (Object)tenantResource);
            return false;
        }
        return true;
    }

    private boolean isCellResourceLoadInfoValid(int cellId) {
        CellResourceUsage loadByCellResource = this.cellResourceLoadStats.get(cellId);
        if (loadByCellResource == null) {
            LOG.error("Cell resource load data not found for cell ID: {}", (Object)cellId);
            return false;
        }
        for (CellResource cellResource : CellResource.cachedValues()) {
            CellResourceLoad cellResourceLoad = loadByCellResource.load(cellResource);
            if (cellResourceLoad != null) continue;
            LOG.error("CellResourceLoad for resource {} not found in cell {}", (Object)cellResource, (Object)cellId);
            return false;
        }
        return true;
    }

    Map<Integer, CellResourceUsage> getUpdatedCellResourceLoadStats(List<CellTenantView> availableCells, List<Integer> selectedCells, TenantStripingInfo tenantStripingInfo) {
        Map<Integer, CellResourceUsage> updatedCellResourceLoadStats = this.deepCopyCellResourceLoadStats(this.cellResourceLoadStats);
        HashSet<Integer> selectedCellIds = new HashSet<Integer>(selectedCells);
        for (CellTenantView cellTenantView : availableCells) {
            int cellId = cellTenantView.id();
            CellResourceUsage loadByCellResource = updatedCellResourceLoadStats.get(cellId);
            if (!selectedCellIds.contains(cellId) && !tenantStripingInfo.existingStripes().contains((Object)cellId)) continue;
            for (TenantResource tenantResource : TenantResource.cachedValues()) {
                Double tenantResourceUsage = tenantStripingInfo.spikedResourceUsages().containsResource(tenantResource) ? tenantStripingInfo.spikedResourceUsages().resourceValue(tenantResource) : tenantStripingInfo.latestResourceUsages().resourceValue(tenantResource);
                Optional<CellResource> cellResourceOpt = this.getCellResource(tenantResource);
                if (cellResourceOpt.isEmpty()) {
                    LOG.debug("Cell resource not found for corresponding tenant resource {}", (Object)tenantResource);
                    continue;
                }
                CellResource cellResource = cellResourceOpt.get();
                CellResourceLoad cellResourceLoad = loadByCellResource.load(cellResource);
                double eligibleCapacity = cellTenantView.getEligibleCapacity(cellResource, this.maxReplicasPerBroker);
                double cellResourceUsage = cellResourceLoad.ratio() * eligibleCapacity;
                CellResourceLoad updatedCellResourceLoad = this.getUpdatedCellResourceLoad(cellTenantView, cellResource, cellResourceUsage, tenantResourceUsage, selectedCellIds, tenantStripingInfo, cellId);
                loadByCellResource.addOrUpdateLoad(updatedCellResourceLoad);
            }
        }
        return updatedCellResourceLoadStats;
    }

    private CellResourceLoad getUpdatedCellResourceLoad(CellTenantView cellTenantView, CellResource cellResource, double cellResourceUsage, double tenantResourceUsage, Set<Integer> selectedCellIds, TenantStripingInfo tenantStripingInfo, int cellId) {
        CellResourceLoad updatedCellResourceLoad;
        int existingStripeFactor = tenantStripingInfo.existingStripes().size();
        int updatedStripeFactor = existingStripeFactor + selectedCellIds.size();
        double updatedPerStripeUsage = tenantResourceUsage / (double)updatedStripeFactor;
        if (selectedCellIds.contains(cellId)) {
            updatedCellResourceLoad = cellTenantView.loadForResource(cellResource, cellResourceUsage + updatedPerStripeUsage, this.maxReplicasPerBroker);
        } else {
            double existingTenantStripeUsage = cellTenantView.tenants().stream().filter(tenant -> tenant.tenantId().equals(tenantStripingInfo.tenantId())).findAny().orElseThrow(() -> new IllegalStateException(String.format("Tenant %s not found in cell %d", tenantStripingInfo.tenantId(), cellId))).getResourceUsage(cellId, cellResource.resource());
            updatedCellResourceLoad = cellTenantView.loadForResource(cellResource, cellResourceUsage - existingTenantStripeUsage + updatedPerStripeUsage, this.maxReplicasPerBroker);
        }
        return updatedCellResourceLoad;
    }

    void incrementStripeCount(int newStripes) {
        this.stripesPlacedForCurrentRound += newStripes;
    }

    boolean isRateLimitReached() {
        if (this.stripesPlacedForCurrentRound > this.stripingRateLimit) {
            throw new RuntimeException("Number of stripes placed in current round are greater than rate limit");
        }
        return this.stripesPlacedForCurrentRound == this.stripingRateLimit;
    }

    public void clearCurrentRoundState() {
        this.stripesPlacedForCurrentRound = 0;
        this.readyCells.clear();
        this.cellResourceLoadStats.clear();
        this.tenantToCellAssignments.clear();
    }

    private static Map<TenantResource, CellResource> getTenantResourceToCellResourceMap() {
        HashMap<TenantResource, CellResource> cellResourceByTenantResource = new HashMap<TenantResource, CellResource>();
        for (TenantResource tenantResource : TenantResource.cachedValues()) {
            Optional<CellResource> cellResourceOpt = CellResource.fromTenantResource(tenantResource);
            cellResourceOpt.ifPresent(cellResource -> cellResourceByTenantResource.put(tenantResource, (CellResource)((Object)cellResource)));
        }
        return cellResourceByTenantResource;
    }

    private Optional<CellResource> getCellResource(TenantResource tenantResource) {
        if (cellResourceByTenantResource.containsKey((Object)tenantResource)) {
            return Optional.of(cellResourceByTenantResource.get((Object)tenantResource));
        }
        return Optional.empty();
    }

    private Map<Integer, CellResourceUsage> deepCopyCellResourceLoadStats(Map<Integer, CellResourceUsage> cellResourceLoadStats) {
        HashMap<Integer, CellResourceUsage> copiedCellResourceLoadStats = new HashMap<Integer, CellResourceUsage>();
        for (Map.Entry<Integer, CellResourceUsage> entry : cellResourceLoadStats.entrySet()) {
            int cellId = entry.getKey();
            CellResourceUsage existingUsage = entry.getValue();
            CellResourceUsage clonedUsage = new CellResourceUsage();
            for (CellResource resource : CellResource.cachedValues()) {
                CellResourceLoad load = existingUsage.load(resource);
                if (load == null) continue;
                clonedUsage.addOrUpdateLoad(new CellResourceLoad(load));
            }
            copiedCellResourceLoadStats.put(cellId, clonedUsage);
        }
        return copiedCellResourceLoadStats;
    }

    public void close() {
    }
}

