/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.model;

import com.linkedin.cruisecontrol.monitor.sampling.aggregator.AggregatedMetricValues;
import com.linkedin.kafka.cruisecontrol.analyzer.AnalyzerUtils;
import com.linkedin.kafka.cruisecontrol.analyzer.BalancingConstraint;
import com.linkedin.kafka.cruisecontrol.common.CellResource;
import com.linkedin.kafka.cruisecontrol.common.Resource;
import com.linkedin.kafka.cruisecontrol.config.BrokerCapacityInfo;
import com.linkedin.kafka.cruisecontrol.config.KafkaCruiseControlConfig;
import com.linkedin.kafka.cruisecontrol.exception.OptimizationFailureException;
import com.linkedin.kafka.cruisecontrol.model.Broker;
import com.linkedin.kafka.cruisecontrol.model.Capacity;
import com.linkedin.kafka.cruisecontrol.model.Cell;
import com.linkedin.kafka.cruisecontrol.model.ClusterModelCheckpoint;
import com.linkedin.kafka.cruisecontrol.model.ClusterModelHelper;
import com.linkedin.kafka.cruisecontrol.model.ClusterModelStats;
import com.linkedin.kafka.cruisecontrol.model.Host;
import com.linkedin.kafka.cruisecontrol.model.LeadershipLoadDeltaCalculator;
import com.linkedin.kafka.cruisecontrol.model.Load;
import com.linkedin.kafka.cruisecontrol.model.Partition;
import com.linkedin.kafka.cruisecontrol.model.Rack;
import com.linkedin.kafka.cruisecontrol.model.Replica;
import com.linkedin.kafka.cruisecontrol.model.ReplicaId;
import com.linkedin.kafka.cruisecontrol.model.ReplicaRelocationContext;
import com.linkedin.kafka.cruisecontrol.model.ResourceStats;
import com.linkedin.kafka.cruisecontrol.model.Tenant;
import com.linkedin.kafka.cruisecontrol.model.TopicImbalanceScore;
import com.linkedin.kafka.cruisecontrol.model.TopicImbalanceScoreType;
import com.linkedin.kafka.cruisecontrol.model.Utilization;
import com.linkedin.kafka.cruisecontrol.model.internal.AbstractEntityWithCapacity;
import com.linkedin.kafka.cruisecontrol.monitor.BrokerStats;
import com.linkedin.kafka.cruisecontrol.monitor.ModelGeneration;
import io.confluent.kafka.multitenant.TenantUtils;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.Stack;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.message.DescribeCellsResponseData;
import org.apache.kafka.common.message.DescribeTenantsResponseData;
import org.apache.kafka.metadata.TopicPlacement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClusterModel
extends AbstractEntityWithCapacity
implements Serializable,
ResourceStats {
    private static final Logger LOG = LoggerFactory.getLogger(ClusterModel.class);
    private static final long serialVersionUID = -6840253566423285966L;
    private static final Broker GENESIS_BROKER = Broker.createGenesisBroker();
    public static final double TOPIC_IMBALANCE_SCORE_LEADERS_WEIGHT = 0.5;
    public static final double TOPIC_IMBALANCE_SCORE_FOLLOWERS_WEIGHT = 0.5;
    private final ModelGeneration generation;
    private final Map<String, Rack> racksById;
    private final Map<Integer, Rack> brokerIdToRack;
    private final Map<Integer, Cell> cellIdToCell;
    private final Cell deadBrokersCell;
    private final Map<String, Tenant> tenantsById;
    private final Map<TopicPartition, Partition> partitionsByTopicPartition;
    private final Set<Replica> selfHealingEligibleReplicas;
    private final SortedSet<Broker> newBrokers;
    private final SortedSet<Broker> brokersWithBadDisks;
    private final Set<Broker> aliveBrokers;
    private final SortedSet<Broker> deadBrokers;
    private final Set<Broker> ignoredBrokers;
    private final SortedSet<Broker> brokers;
    private final Set<TopicPartition> reassigningPartitions;
    private final double monitoredPartitionsRatio;
    private final Set<String> topicNames;
    private Optional<Set<String>> topics;
    private final Map<Integer, Load> potentialLeadershipLoadByBrokerId;
    private final Map<Integer, Utilization> potentialLeadershipUtilizationByBrokerId;
    private int unknownHostId;
    private final Map<Integer, String> capacityEstimationInfoByBrokerId;
    private Optional<Map<String, TopicPlacement>> topicPlacements;
    private final Map<String, TopicImbalanceScore> topicImbalanceScores;
    private final double topicImbalanceScoreBalancedThreshold;
    private Set<Integer> activeBrokerReplicaExclusions = new HashSet<Integer>();
    private boolean isCellEnabled;
    private boolean replicasAreInExpectedCell;
    private final Stack<ReplicaAction> actionsHistoryItem;
    private long lastCheckpointId;

    public ClusterModel(ModelGeneration generation, double monitoredPartitionsRatio) {
        this(generation, monitoredPartitionsRatio, KafkaCruiseControlConfig.DEFAULT_TOPIC_BALANCING_SLIGHTLY_IMBALANCED_TOPIC_IMBALANCE_SCORE_THRESHOLD);
    }

    public ClusterModel(ModelGeneration generation, double monitoredPartitionsRatio, double topicImbalanceScoreBalancedThreshold) {
        this.generation = generation;
        this.racksById = new HashMap<String, Rack>();
        this.brokerIdToRack = new HashMap<Integer, Rack>();
        this.cellIdToCell = new HashMap<Integer, Cell>();
        this.deadBrokersCell = Cell.createDeadCell();
        this.tenantsById = new HashMap<String, Tenant>();
        this.partitionsByTopicPartition = new HashMap<TopicPartition, Partition>();
        this.selfHealingEligibleReplicas = new HashSet<Replica>();
        this.newBrokers = new TreeSet<Broker>();
        this.brokersWithBadDisks = new TreeSet<Broker>();
        this.aliveBrokers = new HashSet<Broker>();
        this.brokers = new TreeSet<Broker>();
        this.deadBrokers = new TreeSet<Broker>();
        this.ignoredBrokers = new HashSet<Broker>();
        this.load = new Load();
        this.utilization = Utilization.from(this.load);
        this.capacity = Capacity.create();
        this.topicNames = new HashSet<String>();
        this.potentialLeadershipLoadByBrokerId = new HashMap<Integer, Load>();
        this.potentialLeadershipUtilizationByBrokerId = new HashMap<Integer, Utilization>();
        this.monitoredPartitionsRatio = monitoredPartitionsRatio;
        this.unknownHostId = 0;
        this.capacityEstimationInfoByBrokerId = new HashMap<Integer, String>();
        this.topicPlacements = Optional.empty();
        this.topicImbalanceScores = new HashMap<String, TopicImbalanceScore>();
        this.reassigningPartitions = new HashSet<TopicPartition>();
        this.topicImbalanceScoreBalancedThreshold = topicImbalanceScoreBalancedThreshold;
        this.topics = Optional.empty();
        this.actionsHistoryItem = new Stack();
        this.lastCheckpointId = 0L;
    }

    public ModelGeneration generation() {
        return this.generation;
    }

    public double monitoredPartitionsRatio() {
        return this.monitoredPartitionsRatio;
    }

    public ClusterModelStats getClusterStats(BalancingConstraint balancingConstraint) {
        return new ClusterModelStats().populate(this, balancingConstraint);
    }

    public Set<String> aliveRackIds() {
        return this.racksById.values().stream().filter(Rack::isRackAlive).map(Rack::id).collect(Collectors.toSet());
    }

    public Rack rack(String rackId) {
        return this.racksById.get(rackId);
    }

    public Cell cell(int cellId) {
        return Cell.isDeadCellId(cellId) ? this.deadBrokersCell : this.cellIdToCell.get(cellId);
    }

    public Optional<Tenant> tenant(String tenantId) {
        return Optional.ofNullable(this.tenantsById.get(tenantId));
    }

    public Collection<Cell> cells() {
        return Collections.unmodifiableCollection(this.cellIdToCell.values());
    }

    public void isCellEnabled(boolean isCellEnabled) {
        this.isCellEnabled = isCellEnabled;
    }

    public boolean isCellEnabled() {
        return this.isCellEnabled;
    }

    public void replicasAreInExpectedCell(boolean replicasAreInExpectedCell) {
        this.replicasAreInExpectedCell = replicasAreInExpectedCell;
    }

    public boolean replicasAreInExpectedCell() {
        return this.replicasAreInExpectedCell;
    }

    public boolean skipCellBalancing() {
        return !this.isCellEnabled || this.cellIdToCell.size() <= 1;
    }

    public Map<TopicPartition, List<ReplicaId>> getReplicaDistribution() {
        HashMap<TopicPartition, List<ReplicaId>> replicaDistribution = new HashMap<TopicPartition, List<ReplicaId>>(this.partitionsByTopicPartition.size());
        for (Map.Entry<TopicPartition, Partition> entry : this.partitionsByTopicPartition.entrySet()) {
            TopicPartition tp = entry.getKey();
            Partition partition = entry.getValue();
            List replicaIds = partition.replicas().stream().map(r -> new ReplicaId(r.broker().id())).collect(Collectors.toList());
            replicaDistribution.put(tp, replicaIds);
        }
        return replicaDistribution;
    }

    public Map<TopicPartition, ReplicaId> getLeaderDistribution() {
        HashMap<TopicPartition, ReplicaId> leaders = new HashMap<TopicPartition, ReplicaId>(this.partitionsByTopicPartition.size());
        for (Map.Entry<TopicPartition, Partition> entry : this.partitionsByTopicPartition.entrySet()) {
            Replica leaderReplica = entry.getValue().leader();
            leaders.put(entry.getKey(), new ReplicaId(leaderReplica.broker().id()));
        }
        return leaders;
    }

    public Map<TopicPartition, List<ReplicaId>> getObserverDistribution() {
        HashMap<TopicPartition, List<ReplicaId>> observerDistribution = new HashMap<TopicPartition, List<ReplicaId>>(this.partitionsByTopicPartition.size());
        for (Map.Entry<TopicPartition, Partition> entry : this.partitionsByTopicPartition.entrySet()) {
            TopicPartition tp = entry.getKey();
            Partition partition = entry.getValue();
            List replicaIds = partition.replicas().stream().filter(Replica::isObserver).map(r -> new ReplicaId(r.broker().id())).collect(Collectors.toList());
            observerDistribution.put(tp, replicaIds);
        }
        return observerDistribution;
    }

    public Set<Replica> selfHealingEligibleReplicas() {
        return this.selfHealingEligibleReplicas;
    }

    public Load potentialLeadershipLoadFor(Integer brokerId) {
        return this.potentialLeadershipLoadByBrokerId.get(brokerId);
    }

    public Utilization potentialLeadershipUtilizationFor(Integer brokerId) {
        return this.potentialLeadershipUtilizationByBrokerId.get(brokerId);
    }

    public Partition partition(TopicPartition tp) {
        return this.partitionsByTopicPartition.get(tp);
    }

    public List<Partition> partitionsByTenantId(String tenantId) {
        Predicate<Partition> predicate = partition -> TenantUtils.extractTenantId((String)partition.topicPartition().topic()).map(t -> t.equals(tenantId)).orElse(false);
        return this.partitionsByTopicPartition.values().stream().filter(predicate).collect(Collectors.toList());
    }

    Map<TopicPartition, Partition> partitionsByTopicPartition() {
        return this.partitionsByTopicPartition;
    }

    public SortedMap<String, List<Partition>> getPartitionsByTopic() {
        TreeMap<String, List<Partition>> partitionsByTopic = new TreeMap<String, List<Partition>>();
        for (String string : this.topics()) {
            partitionsByTopic.put(string, new ArrayList());
        }
        for (Map.Entry entry : this.partitionsByTopicPartition.entrySet()) {
            ((List)partitionsByTopic.get(((TopicPartition)entry.getKey()).topic())).add((Partition)entry.getValue());
        }
        return partitionsByTopic;
    }

    public Set<Replica> leaderReplicas() {
        return this.partitionsByTopicPartition.values().stream().map(Partition::leader).collect(Collectors.toSet());
    }

    public Set<Replica> leaderReplicasForTopic(String topic) {
        return this.brokers.stream().flatMap(broker -> broker.leaderReplicasOfTopicInBroker(topic).stream()).collect(Collectors.toSet());
    }

    public Set<Replica> followerReplicasForTopic(String topic) {
        return this.brokers.stream().flatMap(broker -> broker.followerReplicasOfTopicInBroker(topic).stream()).collect(Collectors.toSet());
    }

    private void updateTopicReplicaDistributionBasedImbalanceScore(String topic) {
        SortedSet<Broker> eligibleDestinationBrokers = this.eligibleDestinationBrokers();
        int numEligibleDestinationBrokers = eligibleDestinationBrokers.size();
        if (numEligibleDestinationBrokers == 0) {
            LOG.warn("Trying to calculate replica distribution based topic imbalance score while there are no eligible destination brokers so calculation is skipped.");
            return;
        }
        if (this.isCellEnabled) {
            this.updateTopicReplicaDistributionBasedImbalanceScoreAtCellLevel(topic);
        } else {
            this.updateTopicReplicaDistributionBasedImbalanceScoreAtClusterLevel(topic);
        }
    }

    private void updateTopicReplicaDistributionBasedImbalanceScoreAtClusterLevel(String topic) {
        List<Integer> leaderDistributionForTopic = this.replicaDistributionForTopic(topic, true);
        List<Integer> followerDistributionForTopic = this.replicaDistributionForTopic(topic, false);
        TopicImbalanceScore topicImbalanceScore = this.topicImbalanceScores.computeIfAbsent(topic, t -> new TopicImbalanceScore());
        topicImbalanceScore.setImbalanceScore(this.calculateTopicReplicaDistributionBasedImbalanceScore(leaderDistributionForTopic, followerDistributionForTopic), TopicImbalanceScoreType.REPLICA_DISTRIBUTION_BASED);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Topic {}'s replica distribution based imbalance score: {}, it has leader distribution as: {} and follower distribution as: {}", new Object[]{topic, topicImbalanceScore.getImbalanceScore(TopicImbalanceScoreType.REPLICA_DISTRIBUTION_BASED), leaderDistributionForTopic, followerDistributionForTopic});
        }
    }

    private void updateTopicReplicaDistributionBasedImbalanceScoreAtCellLevel(String topic) {
        for (int cellId : this.cellIdToCell.keySet()) {
            this.updateTopicReplicaDistributionBasedImbalanceScoreForCell(topic, cellId);
        }
        TopicImbalanceScore topicImbalanceScore = this.topicImbalanceScores.computeIfAbsent(topic, t -> new TopicImbalanceScore());
        topicImbalanceScore.setImbalanceScore(topicImbalanceScore.getMaxPerCellImbalanceScore(TopicImbalanceScoreType.REPLICA_DISTRIBUTION_BASED), TopicImbalanceScoreType.REPLICA_DISTRIBUTION_BASED);
        LOG.debug("For topic {}, overall replica distribution based imbalance score: {}", (Object)topic, (Object)topicImbalanceScore.getImbalanceScore(TopicImbalanceScoreType.REPLICA_DISTRIBUTION_BASED));
    }

    private void updateTopicReplicaDistributionBasedImbalanceScoreForCell(String topic, int cellId) {
        List<Integer> leaderDistributionForTopic = this.replicaDistributionForTopic(topic, true, cellId);
        List<Integer> followerDistributionForTopic = this.replicaDistributionForTopic(topic, false, cellId);
        TopicImbalanceScore topicImbalanceScore = this.topicImbalanceScores.computeIfAbsent(topic, t -> new TopicImbalanceScore());
        topicImbalanceScore.setImbalanceScore(this.calculateTopicReplicaDistributionBasedImbalanceScore(leaderDistributionForTopic, followerDistributionForTopic), TopicImbalanceScoreType.REPLICA_DISTRIBUTION_BASED, cellId);
        if (LOG.isDebugEnabled()) {
            LOG.debug("For cell {}, Topic {}'s replica distribution based imbalance score: {}, it has leader distribution as: {} and follower distribution as: {}", new Object[]{cellId, topic, topicImbalanceScore.getImbalanceScore(TopicImbalanceScoreType.REPLICA_DISTRIBUTION_BASED, cellId), leaderDistributionForTopic, followerDistributionForTopic});
        }
    }

    private void updateTopicPartitionDistributionBasedImbalanceScore(String topic) {
        TopicImbalanceScore topicImbalanceScore = this.topicImbalanceScores.computeIfAbsent(topic, t -> new TopicImbalanceScore());
        Optional<Tenant> tenant = this.tenantOfTopic(topic);
        if (!tenant.isPresent()) {
            topicImbalanceScore.setImbalanceScore(0.0, TopicImbalanceScoreType.PARTITION_DISTRIBUTION_BASED);
            return;
        }
        ArrayList<Integer> partitionDistributionForTopic = new ArrayList<Integer>();
        for (Cell cell : this.cells()) {
            int numOfPartitions = cell.numTopicLeaderReplicas(topic);
            if (numOfPartitions != 0 && !tenant.get().cellIds().contains(cell.id())) {
                topicImbalanceScore.setImbalanceScore((Double)Double.MAX_VALUE, TopicImbalanceScoreType.PARTITION_DISTRIBUTION_BASED);
                return;
            }
            if (!tenant.get().cellIds().contains(cell.id())) continue;
            partitionDistributionForTopic.add(numOfPartitions);
        }
        topicImbalanceScore.setImbalanceScore(ClusterModelHelper.calculateNormalizedStdDeviationOfRatio(partitionDistributionForTopic), TopicImbalanceScoreType.PARTITION_DISTRIBUTION_BASED);
    }

    public List<Integer> replicaDistributionForTopic(String topic, Boolean isLeader) {
        int limitNumberOfBrokers = isLeader != false ? this.numLeaderReplicasForTopicOnEligibleDestinationBrokers(topic) : this.numFollowerReplicasForTopicOnEligibleDestinationBrokers(topic);
        return this.eligibleDestinationBrokers().stream().mapToInt(broker -> isLeader != false ? broker.numLeaderReplicasOfTopicInBroker(topic) : broker.numFollowerReplicasOfTopicInBroker(topic)).boxed().sorted(Comparator.reverseOrder()).limit(limitNumberOfBrokers).collect(Collectors.toList());
    }

    public List<Integer> replicaDistributionForTopic(String topic, Boolean isLeader, int cellId) {
        int limitNumberOfBrokers = isLeader != false ? this.numLeaderReplicasForTopicOnEligibleDestinationBrokers(topic, cellId) : this.numFollowerReplicasForTopicOnEligibleDestinationBrokers(topic, cellId);
        return this.eligibleDestinationBrokers(cellId).stream().mapToInt(broker -> isLeader != false ? broker.numLeaderReplicasOfTopicInBroker(topic) : broker.numFollowerReplicasOfTopicInBroker(topic)).boxed().sorted(Comparator.reverseOrder()).limit(limitNumberOfBrokers).collect(Collectors.toList());
    }

    public Double replicaDistributionImbalanceScore(List<Integer> replicaDistributionForTopic) {
        return ClusterModelHelper.calculateNormalizedStdDeviationOfRatio(replicaDistributionForTopic);
    }

    public Double calculateTopicReplicaDistributionBasedImbalanceScore(List<Integer> leaderDistributionForTopic, List<Integer> followerDistributionForTopic) {
        double leaderImbalanceScore = this.replicaDistributionImbalanceScore(leaderDistributionForTopic);
        double followerImbalanceScore = this.replicaDistributionImbalanceScore(followerDistributionForTopic);
        return 0.5 * leaderImbalanceScore + 0.5 * followerImbalanceScore;
    }

    public Double topicImbalanceScore(String topic, TopicImbalanceScoreType imbalanceScoreType) {
        TopicImbalanceScore imbalanceScore = this.topicImbalanceScores.computeIfAbsent(topic, t -> new TopicImbalanceScore());
        return imbalanceScore.getImbalanceScore(imbalanceScoreType);
    }

    public Double topicImbalanceScore(String topic, TopicImbalanceScoreType imbalanceScoreType, int cellId) {
        TopicImbalanceScore imbalanceScore = this.topicImbalanceScores.computeIfAbsent(topic, t -> new TopicImbalanceScore());
        return imbalanceScore.getImbalanceScore(imbalanceScoreType, cellId);
    }

    public boolean isTopicBalanced(String topic) {
        return this.topicImbalanceScore(topic, TopicImbalanceScoreType.REPLICA_DISTRIBUTION_BASED) <= this.topicImbalanceScoreBalancedThreshold;
    }

    public Set<String> balancedTopics() {
        return this.topics().stream().filter(this::isTopicBalanced).collect(Collectors.toSet());
    }

    private void updateBrokerLists(Broker broker) {
        switch (broker.strategy()) {
            case DEAD: {
                this.aliveBrokers.remove(broker);
                this.deadBrokers.add(broker);
                this.brokersWithBadDisks.remove(broker);
                this.ignoredBrokers.remove(broker);
                break;
            }
            case NEW: {
                this.newBrokers.add(broker);
            }
            case ALIVE: {
                this.aliveBrokers.add(broker);
                this.deadBrokers.remove(broker);
                this.brokersWithBadDisks.remove(broker);
                this.ignoredBrokers.remove(broker);
                break;
            }
            case IGNORE: {
                this.deadBrokers.remove(broker);
                this.brokersWithBadDisks.remove(broker);
                this.ignoredBrokers.add(broker);
                this.aliveBrokers.add(broker);
                break;
            }
            case BAD_DISKS: {
                this.aliveBrokers.add(broker);
                this.deadBrokers.remove(broker);
                this.brokersWithBadDisks.add(broker);
                break;
            }
            default: {
                throw new IllegalArgumentException("Illegal broker strategy " + String.valueOf((Object)broker.strategy()) + " is provided.");
            }
        }
        for (String topic : this.topics()) {
            this.updateTopicReplicaDistributionBasedImbalanceScore(topic);
            this.updateTopicPartitionDistributionBasedImbalanceScore(topic);
        }
    }

    public void relocateReplica(TopicPartition tp, int sourceBrokerId, int destinationBrokerId, ReplicaRelocationContext replicaRelocationContext) {
        this.relocateReplica(tp, sourceBrokerId, destinationBrokerId, replicaRelocationContext, true);
    }

    private void relocateReplica(TopicPartition tp, int sourceBrokerId, int destinationBrokerId, ReplicaRelocationContext replicaRelocationContext, boolean recordMovementInTheUndoLog) {
        Replica replica = this.removeReplica(sourceBrokerId, tp);
        if (replica == null) {
            throw new IllegalArgumentException("Replica is not in the cluster.");
        }
        Broker destinationBroker = this.broker(destinationBrokerId);
        replica.setBroker(destinationBroker);
        if (LOG.isDebugEnabled()) {
            LOG.debug("While relocating Replica: {} to broker: {}, setting its load to: {}", new Object[]{replica, destinationBrokerId, replicaRelocationContext.getLoadForBroker(destinationBrokerId)});
        }
        replica.load().resetLoadTo(replicaRelocationContext.getLoadForBroker(destinationBrokerId));
        replica.broker().rack().addReplica(replica);
        replica.broker().cell().addReplica(replica);
        this.load.addLoad(replica.load());
        this.utilization.addLoad(replica.broker().strategy(), replica.load());
        this.tenantOfTopic(replica.topicPartition().topic()).ifPresent(tenant -> tenant.addReplicaWithGivenLoadAndUtilization(destinationBroker.cell().id(), destinationBroker.strategy(), replica.load()));
        Load leaderLoad = this.partition(tp).leader().load();
        this.potentialLeadershipLoadByBrokerId.get(destinationBrokerId).addLoad(leaderLoad);
        this.potentialLeadershipUtilizationByBrokerId.get(destinationBrokerId).addLoad(destinationBroker.strategy(), leaderLoad);
        this.updateTopicReplicaDistributionBasedImbalanceScore(tp.topic());
        this.updateTopicPartitionDistributionBasedImbalanceScore(tp.topic());
        if (recordMovementInTheUndoLog) {
            this.actionsHistoryItem.push(new ReplicaAction(tp, sourceBrokerId, destinationBrokerId, ReplicaAction.ReplicaActionType.RELOCATION, this.lastCheckpointId));
        }
    }

    public boolean relocateLeadership(TopicPartition tp, int sourceBrokerId, int destinationBrokerId) {
        return this.relocateLeadership(tp, sourceBrokerId, destinationBrokerId, true);
    }

    private boolean relocateLeadership(TopicPartition tp, int sourceBrokerId, int destinationBrokerId, boolean recordMovementInTheUndoLog) {
        Replica sourceReplica = this.partitionsByTopicPartition.get(tp).replica(sourceBrokerId);
        if (!sourceReplica.isLeader()) {
            return false;
        }
        Replica destinationReplica = this.partitionsByTopicPartition.get(tp).replica(destinationBrokerId);
        if (destinationReplica.isLeader()) {
            throw new IllegalArgumentException("Cannot relocate leadership of partition " + String.valueOf(tp) + "from broker " + sourceBrokerId + " to broker " + destinationBrokerId + " because the destination replica is a leader.");
        }
        Broker sourceBroker = this.broker(sourceBrokerId);
        Broker destinationBroker = this.broker(destinationBrokerId);
        AggregatedMetricValues leadershipLoadDelta = new LeadershipLoadDeltaCalculator(tp, sourceBroker, destinationBroker).calculateLeadershipLoadDelta();
        Rack sourceRack = sourceBroker.rack();
        sourceRack.makeFollower(sourceBrokerId, tp, leadershipLoadDelta);
        Cell sourceCell = sourceBroker.cell();
        sourceCell.makeFollower(sourceBroker.strategy(), leadershipLoadDelta);
        this.utilization.subtractLoad(sourceBroker.strategy(), leadershipLoadDelta);
        Rack destinationRack = destinationBroker.rack();
        destinationRack.makeLeader(destinationBrokerId, tp, leadershipLoadDelta);
        Cell destinationCell = destinationBroker.cell();
        destinationCell.makeLeader(destinationBroker.strategy(), leadershipLoadDelta);
        this.utilization.addLoad(destinationBroker.strategy(), leadershipLoadDelta);
        this.tenantOfTopic(sourceReplica.topicPartition().topic()).ifPresent(tenant -> tenant.transferUtilizationOfMovedLeadership(sourceBroker, destinationBroker, leadershipLoadDelta));
        Partition partition = this.partitionsByTopicPartition.get(tp);
        partition.relocateLeadership(destinationReplica);
        this.updateTopicReplicaDistributionBasedImbalanceScore(tp.topic());
        if (recordMovementInTheUndoLog) {
            this.actionsHistoryItem.push(new ReplicaAction(tp, sourceBrokerId, destinationBrokerId, ReplicaAction.ReplicaActionType.LEADERSHIP_CHANGE, this.lastCheckpointId));
        }
        return true;
    }

    public boolean changeObservership(TopicPartition tp, int replicaId) {
        return this.changeObservership(tp, replicaId, true);
    }

    private boolean changeObservership(TopicPartition tp, int replicaId, boolean recordMovementInTheUndoLog) {
        Replica replica = this.partitionsByTopicPartition.get(tp).replica(replicaId);
        boolean toBeObserver = !replica.isObserver();
        replica.setObservership(toBeObserver);
        if (recordMovementInTheUndoLog) {
            this.actionsHistoryItem.push(new ReplicaAction(tp, replicaId, replicaId, ReplicaAction.ReplicaActionType.OBSERVER_SWAP, this.lastCheckpointId));
        }
        return toBeObserver;
    }

    public Set<Broker> aliveBrokers() {
        return this.aliveBrokers;
    }

    public SortedSet<Broker> eligibleSourceBrokers() {
        TreeSet<Broker> allBrokers = new TreeSet<Broker>(this.brokers);
        allBrokers.removeAll(this.ignoredBrokers);
        return allBrokers;
    }

    public SortedSet<Broker> eligibleSourceBrokers(int cellId) {
        SortedSet allBrokers = this.brokers.stream().filter(broker -> broker.cell().id() == cellId).collect(Collectors.toCollection(TreeSet::new));
        allBrokers.removeAll(this.ignoredBrokers);
        return allBrokers;
    }

    public SortedSet<Broker> eligibleDestinationBrokers() {
        TreeSet<Broker> allBrokers = new TreeSet<Broker>(this.brokers);
        allBrokers.removeAll(this.ignoredBrokers);
        allBrokers.removeAll(this.deadBrokers);
        return allBrokers;
    }

    public SortedSet<Broker> eligibleDestinationBrokers(int cellId) {
        SortedSet allBrokers = this.brokers.stream().filter(broker -> broker.cell().id() == cellId).collect(Collectors.toCollection(TreeSet::new));
        allBrokers.removeAll(this.ignoredBrokers);
        allBrokers.removeAll(this.deadBrokers);
        return allBrokers;
    }

    public SortedSet<Broker> eligibleSourceAndDestinationBrokers() {
        SortedSet<Broker> result = this.eligibleDestinationBrokers();
        result.retainAll(this.eligibleSourceBrokers());
        return result;
    }

    public SortedSet<Broker> eligibleSourceOrDestinationBrokers() {
        SortedSet<Broker> result = this.eligibleSourceBrokers();
        result.addAll(this.eligibleDestinationBrokers());
        return result;
    }

    public SortedSet<Broker> deadBrokers() {
        return new TreeSet<Broker>(this.deadBrokers);
    }

    public Set<Broker> ignoredBrokers() {
        return Collections.unmodifiableSet(this.ignoredBrokers);
    }

    public SortedSet<Broker> brokenBrokers() {
        TreeSet<Broker> brokenBrokers = new TreeSet<Broker>(this.deadBrokers);
        brokenBrokers.addAll(this.brokersWithBadDisks());
        return Collections.unmodifiableSortedSet(brokenBrokers);
    }

    public Map<Integer, String> capacityEstimationInfoByBrokerId() {
        return Collections.unmodifiableMap(this.capacityEstimationInfoByBrokerId);
    }

    public SortedSet<Broker> newBrokers() {
        return this.newBrokers;
    }

    public SortedSet<Broker> brokersWithBadDisks() {
        return Collections.unmodifiableSortedSet(this.brokersWithBadDisks);
    }

    public Set<Broker> brokersHavingOfflineReplicasOnBadDisks() {
        HashSet<Broker> brokersWithOfflineReplicasOnBadDisks = new HashSet<Broker>();
        for (Broker brokerWithBadDisks : this.brokersWithBadDisks) {
            if (brokerWithBadDisks.currentOfflineReplicas().isEmpty()) continue;
            brokersWithOfflineReplicasOnBadDisks.add(brokerWithBadDisks);
        }
        return brokersWithOfflineReplicasOnBadDisks;
    }

    public Set<Broker> aliveBrokersMatchingAttributes(Map<String, String> attributes) {
        return this.aliveBrokers().stream().filter(broker -> broker.attributes().equals(attributes)).collect(Collectors.toSet());
    }

    public boolean clusterHasEligibleDestinationBrokers() {
        return this.racksById.values().stream().anyMatch(Rack::isEligibleDestination);
    }

    public Replica removeReplica(int brokerId, TopicPartition tp) {
        for (Rack rack : this.racksById.values()) {
            Replica removedReplica = rack.removeReplica(brokerId, tp);
            if (removedReplica == null) continue;
            if (removedReplica.broker().id() != brokerId) {
                throw new IllegalStateException(String.format("Removed replica %s is not on the broker %d", removedReplica, brokerId));
            }
            this.load.subtractLoad(removedReplica.load());
            this.utilization.subtractLoad(removedReplica.broker().strategy(), removedReplica.load());
            this.tenantOfTopic(removedReplica.topicPartition().topic()).ifPresent(tenant -> tenant.removeReplicaWithGivenLoadAndUtilization(removedReplica.broker().cell().id(), removedReplica.broker().strategy(), removedReplica.load()));
            removedReplica.broker().cell().removeReplica(removedReplica);
            Load leaderLoad = this.partition(tp).leader().load();
            this.potentialLeadershipLoadByBrokerId.get(brokerId).subtractLoad(leaderLoad);
            this.potentialLeadershipUtilizationByBrokerId.get(brokerId).subtractLoad(removedReplica.broker().strategy(), leaderLoad);
            this.updateTopicReplicaDistributionBasedImbalanceScore(tp.topic());
            this.updateTopicPartitionDistributionBasedImbalanceScore(tp.topic());
            return removedReplica;
        }
        return null;
    }

    public SortedSet<Broker> allBrokers() {
        return new TreeSet<Broker>(this.brokers);
    }

    public SortedSet<Broker> allBrokersWithStateOtherThan(Broker.Strategy stateToSkip) {
        return this.brokers.stream().filter(broker -> !broker.strategy().equals((Object)stateToSkip)).collect(Collectors.toCollection(TreeSet::new));
    }

    public Set<Broker> brokersOfStatesMatchingAttributes(Collection<Broker> brokerPool, EnumSet<Broker.Strategy> states, Map<String, String> attributes) {
        return brokerPool.stream().filter(broker -> states.contains((Object)broker.strategy()) && broker.attributes().equals(attributes)).collect(Collectors.toSet());
    }

    public Set<Broker> brokersNotOfStatesMatchingAttributes(Collection<Broker> brokerPool, EnumSet<Broker.Strategy> states, Map<String, String> attributes) {
        return brokerPool.stream().filter(broker -> !states.contains((Object)broker.strategy()) && broker.attributes().equals(attributes)).collect(Collectors.toSet());
    }

    public Broker broker(int brokerId) {
        Rack rack = this.brokerIdToRack.get(brokerId);
        return rack == null ? null : rack.broker(brokerId);
    }

    public void trackSortedReplicas(String sortName, Function<Replica, Boolean> selectionFunc, Function<Replica, Integer> priorityFunc, Function<Replica, Double> scoreFunc) {
        this.trackSortedReplicas(this.brokers, sortName, selectionFunc, priorityFunc, scoreFunc);
    }

    public void trackSortedReplicas(Collection<Broker> brokersToTrack, String sortName, Function<Replica, Boolean> selectionFunc, Function<Replica, Integer> priorityFunc, Function<Replica, Double> scoreFunc) {
        brokersToTrack.forEach(b -> b.trackSortedReplicas(sortName, selectionFunc, priorityFunc, scoreFunc));
    }

    public void untrackSortedReplicas(String sortName) {
        this.brokers.forEach(b -> b.untrackSortedReplicas(sortName));
    }

    public int numTopicReplicas(String topic) {
        int numTopicReplicas = 0;
        for (Rack rack : this.racksById.values()) {
            numTopicReplicas += rack.numTopicReplicas(topic);
        }
        return numTopicReplicas;
    }

    public int numLeaderReplicas() {
        return this.partitionsByTopicPartition.size();
    }

    public int numLeaderReplicasForTopicOnEligibleDestinationBrokers(String topic) {
        return this.eligibleDestinationBrokers().stream().mapToInt(b -> b.numLeaderReplicasOfTopicInBroker(topic)).sum();
    }

    public int numLeaderReplicasForTopicOnEligibleDestinationBrokers(String topic, int cellId) {
        return this.eligibleDestinationBrokers(cellId).stream().mapToInt(b -> b.numLeaderReplicasOfTopicInBroker(topic)).sum();
    }

    public int numFollowerReplicasForTopicOnEligibleDestinationBrokers(String topic) {
        return this.eligibleDestinationBrokers().stream().mapToInt(b -> b.numFollowerReplicasOfTopicInBroker(topic)).sum();
    }

    public int numFollowerReplicasForTopicOnEligibleDestinationBrokers(String topic, int cellId) {
        return this.eligibleDestinationBrokers(cellId).stream().mapToInt(b -> b.numFollowerReplicasOfTopicInBroker(topic)).sum();
    }

    public int numLeaderReplicasOnEligibleSourceBrokers() {
        return this.eligibleSourceBrokers().stream().mapToInt(Broker::numLeaderReplicas).sum();
    }

    public int numReplicas() {
        return this.partitionsByTopicPartition.values().stream().mapToInt(p -> p.replicas().size()).sum();
    }

    public int numReplicasOnEligibleSourceBrokers() {
        return this.eligibleSourceBrokers().stream().mapToInt(broker -> broker.replicas().size()).sum();
    }

    public Set<String> topics() {
        if (this.topics.isEmpty()) {
            this.topics = Optional.of(Collections.unmodifiableSet(this.topicNames));
        }
        return this.topics.get();
    }

    public Map<Integer, Cell> cellsById() {
        return Collections.unmodifiableMap(this.cellIdToCell);
    }

    public Map<String, Tenant> tenantsById() {
        return Collections.unmodifiableMap(this.tenantsById);
    }

    public void setReplicaLoad(String rackId, int brokerId, TopicPartition tp, AggregatedMetricValues metricValues, List<Long> windows) {
        Broker broker = this.broker(brokerId);
        if (!broker.replica(tp).load().isEmpty()) {
            throw new IllegalStateException(String.format("The load for %s on broker %d, rack %s already has metric values.", tp, brokerId, rackId));
        }
        Rack rack = this.rack(rackId);
        rack.setReplicaLoad(brokerId, tp, metricValues, windows);
        broker.cell().replicaLoad(broker, metricValues, windows);
        this.tenantOfTopic(tp.topic()).ifPresent(tenant -> tenant.addReplicaWithGivenLoadAndUtilization(broker.cell().id(), broker.strategy(), metricValues, windows));
        this.load.addMetricValues(metricValues, windows);
        this.utilization.addMetricValues(broker.strategy(), metricValues, windows);
        Replica leader = this.partition(tp).leader();
        if (leader != null && leader.broker().id() == brokerId) {
            for (Replica replica : this.partition(tp).replicas()) {
                int replicaBrokerId = replica.broker().id();
                this.potentialLeadershipLoadByBrokerId.get(replicaBrokerId).addMetricValues(metricValues, windows);
                this.potentialLeadershipUtilizationByBrokerId.get(replicaBrokerId).addMetricValues(replica.broker().strategy(), metricValues, windows);
            }
        }
    }

    public void handleDeadBroker(String rackId, int brokerId, BrokerCapacityInfo brokerCapacityInfo) {
        this.createRackIfAbsent(rackId);
        if (this.broker(brokerId) == null) {
            this.createBroker(rackId, this.deadBrokersCell.id(), String.format("UNKNOWN_HOST-%d", this.unknownHostId++), brokerId, brokerCapacityInfo, Broker.Strategy.DEAD);
        }
    }

    public Replica createReplica(String rackId, int brokerId, TopicPartition tp, int index, boolean isLeader) {
        return this.createReplica(rackId, brokerId, tp, index, isLeader, !this.broker(brokerId).isAlive(), false, false);
    }

    public Replica createReplica(String rackId, int brokerId, TopicPartition tp, int index, boolean isLeader, boolean isOffline, boolean isFuture, boolean isObserver) {
        Replica replica;
        Broker broker = this.broker(brokerId);
        if (!isFuture) {
            replica = new Replica(tp, broker, isLeader, isOffline, isObserver);
        } else {
            replica = new Replica(tp, GENESIS_BROKER, false);
            replica.setBroker(broker);
        }
        this.rack(rackId).addReplica(replica);
        if (!this.partitionsByTopicPartition.containsKey(tp)) {
            this.partitionsByTopicPartition.put(tp, new Partition(tp));
            this.topicNames.add(tp.topic());
        }
        if (broker.currentOfflineReplicas().contains(replica)) {
            this.selfHealingEligibleReplicas.add(replica);
            this.partitionsByTopicPartition.get(replica.topicPartition()).addBadDiskBroker(broker);
        }
        Partition partition = this.partitionsByTopicPartition.get(tp);
        if (replica.isLeader()) {
            partition.addLeader(replica, index);
            this.updateTopicReplicaDistributionBasedImbalanceScore(tp.topic());
            this.updateTopicPartitionDistributionBasedImbalanceScore(tp.topic());
            return replica;
        }
        partition.addFollower(replica, index);
        Replica leaderReplica = this.partition(tp).leader();
        if (leaderReplica != null) {
            Load leaderLoad = leaderReplica.load();
            this.potentialLeadershipLoadByBrokerId.get(brokerId).addMetricValues(leaderLoad.loadByWindows(), leaderLoad.windows());
            this.potentialLeadershipUtilizationByBrokerId.get(brokerId).addMetricValues(broker.strategy(), leaderLoad.loadByWindows(), leaderLoad.windows());
        }
        this.topicNames.add(tp.topic());
        this.updateTopicReplicaDistributionBasedImbalanceScore(tp.topic());
        this.updateTopicPartitionDistributionBasedImbalanceScore(tp.topic());
        return replica;
    }

    public void deleteReplica(TopicPartition topicPartition, int brokerId) {
        int currentReplicaCount = this.partitionsByTopicPartition.get(topicPartition).replicas().size();
        if (currentReplicaCount < 2) {
            throw new IllegalStateException(String.format("Unable to delete replica for topic partition %s since it only has %d replicas.", topicPartition, currentReplicaCount));
        }
        this.removeReplica(brokerId, topicPartition);
        Partition partition = this.partitionsByTopicPartition.get(topicPartition);
        partition.deleteReplica(brokerId);
        this.topicNames.add(topicPartition.topic());
        this.updateTopicReplicaDistributionBasedImbalanceScore(topicPartition.topic());
        this.updateTopicPartitionDistributionBasedImbalanceScore(topicPartition.topic());
    }

    public Broker createBroker(String rackId, int cellId, String host, int brokerId, BrokerCapacityInfo brokerCapacityInfo, Broker.Strategy strategy) {
        this.potentialLeadershipLoadByBrokerId.putIfAbsent(brokerId, new Load());
        this.potentialLeadershipUtilizationByBrokerId.putIfAbsent(brokerId, Utilization.from(this.potentialLeadershipLoadByBrokerId.get(brokerId)));
        Rack rack = this.rack(rackId);
        this.brokerIdToRack.put(brokerId, rack);
        Cell cell = this.cell(cellId);
        if (brokerCapacityInfo.isEstimated()) {
            this.capacityEstimationInfoByBrokerId.put(brokerId, brokerCapacityInfo.estimationInfo());
        }
        Broker broker = rack.createBroker(brokerId, host, cell, brokerCapacityInfo, strategy);
        this.aliveBrokers.add(broker);
        this.brokers.add(broker);
        if (broker.isAlive()) {
            this.capacity.addCapacity(Capacity.from(broker, brokerCapacityInfo));
        }
        this.updateBrokerLists(broker);
        return broker;
    }

    public Map<Integer, Rack> getBrokerIdToRack() {
        return Collections.unmodifiableMap(this.brokerIdToRack);
    }

    public Broker createBroker(String rackId, String host, int brokerId, BrokerCapacityInfo brokerCapacityInfo, Broker.Strategy strategy) {
        return this.createBroker(rackId, -1, host, brokerId, brokerCapacityInfo, strategy);
    }

    public Rack createRackIfAbsent(String rackId) {
        if (!this.racksById.containsKey(rackId)) {
            Rack rack = new Rack(rackId);
            this.racksById.put(rackId, rack);
        }
        return this.racksById.get(rackId);
    }

    public Cell createCellIfAbsent(Optional<Integer> cellId) {
        int cellIdToUse = cellId.orElse(-1);
        if (!this.cellIdToCell.containsKey(cellIdToUse)) {
            Cell cell = new Cell(cellIdToUse);
            this.cellIdToCell.put(cellIdToUse, cell);
        }
        return this.cellIdToCell.get(cellIdToUse);
    }

    public Cell createCell(DescribeCellsResponseData.Cell cellResponseData) {
        if (cellResponseData == null) {
            LOG.info("Create a cell with default cell id and unknown state since no cell metadata is available.");
            return this.createCellIfAbsent(Optional.empty());
        }
        int cellIdToUse = cellResponseData.cellId();
        if (!this.cellIdToCell.containsKey(cellIdToUse)) {
            Cell cell = new Cell(cellResponseData);
            this.cellIdToCell.put(cellIdToUse, cell);
        }
        return this.cellIdToCell.get(cellIdToUse);
    }

    public void createTenant(DescribeTenantsResponseData.TenantDescription tenantDescription) {
        Tenant tenant = new Tenant(tenantDescription);
        this.tenantsById.put(tenantDescription.tenantId(), tenant);
        List<Integer> cellIds = tenant.cellIds();
        for (Integer cellId : cellIds) {
            Cell cell = this.cellIdToCell.get(cellId);
            if (cell == null) {
                LOG.error("Unexpected error: cell {} doesn't exist while tenant {} is assigned to it, current cells: {}", new Object[]{cellId, tenant.tenantId(), this.cellIdToCell});
                continue;
            }
            cell.addTenant(tenant);
        }
    }

    public List<Tenant> tenantsSortedByCellResource(CellResource cellResource) {
        ArrayList<Tenant> sortedTenants = new ArrayList<Tenant>(this.tenantsById.values());
        sortedTenants.sort(cellResource.tenantResourceUtilComparator());
        return sortedTenants;
    }

    public void addCell(Cell cell) {
        if (!this.cellIdToCell.containsKey(cell.id())) {
            this.cellIdToCell.put(cell.id(), cell);
        }
    }

    public boolean createOrDeleteReplicas(Map<Short, Set<String>> topicsByReplicationFactor, Map<String, List<Integer>> brokersByRack) throws OptimizationFailureException {
        boolean replicasAltered = false;
        SortedMap<String, List<Partition>> partitionsByTopic = this.getPartitionsByTopic();
        for (Map.Entry<Short, Set<String>> entry : topicsByReplicationFactor.entrySet()) {
            short replicationFactor = entry.getKey();
            Set<String> topics = entry.getValue();
            for (String topic : topics) {
                for (Partition partition : (List)partitionsByTopic.get(topic)) {
                    if (partition.replicas().size() == replicationFactor) continue;
                    if (partition.replicas().size() < replicationFactor) {
                        this.addReplicasUpToReplicationFactor(partition, replicationFactor, brokersByRack);
                    } else {
                        Set newReplicas = partition.replicas().stream().filter(r -> r.broker().strategy() == Broker.Strategy.IGNORE).map(r -> r.broker().id()).collect(Collectors.toSet());
                        newReplicas.add(partition.leader().broker().id());
                        if (newReplicas.size() > replicationFactor) {
                            throw new OptimizationFailureException(String.format("Required RF for partition %s is %d, but we currently have %d replicas which can't be removed.", partition.topicPartition().toString(), replicationFactor, newReplicas.size()));
                        }
                        ArrayList<Replica> replicas = new ArrayList<Replica>(partition.replicas());
                        for (Replica replica : replicas) {
                            int brokerId = replica.broker().id();
                            if (newReplicas.contains(brokerId)) continue;
                            if (newReplicas.size() < replicationFactor) {
                                newReplicas.add(brokerId);
                                continue;
                            }
                            this.deleteReplica(partition.topicPartition(), brokerId);
                        }
                    }
                    replicasAltered = true;
                    this.updateTopicReplicaDistributionBasedImbalanceScore(topic);
                    this.updateTopicPartitionDistributionBasedImbalanceScore(topic);
                }
            }
        }
        return replicasAltered;
    }

    public boolean updateReplicationFactor(Map<Short, Set<String>> topicsByReplicationFactor, Set<Integer> brokersExcludedForReplicaMovement) throws OptimizationFailureException {
        Map<Integer, String> includedBrokerIdToRack = this.brokerIdToRack.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> ((Rack)entry.getValue()).id()));
        includedBrokerIdToRack.keySet().removeAll(brokersExcludedForReplicaMovement);
        Map<String, List<Integer>> rackToBrokerId = includedBrokerIdToRack.entrySet().stream().collect(Collectors.groupingBy(Map.Entry::getValue, Collectors.mapping(Map.Entry::getKey, Collectors.toList())));
        return this.createOrDeleteReplicas(topicsByReplicationFactor, rackToBrokerId);
    }

    public List<Broker> brokersUnderThreshold(Collection<Broker> brokers, Resource resource, double utilizationThreshold) {
        CapacityLimitProvider provider = broker -> ClusterModelHelper.resourceStatsFor(broker, resource).capacity().totalCapacityFor(resource) * utilizationThreshold;
        return this.brokersUnderCapacityLimit(brokers, resource, provider);
    }

    public List<Broker> brokersUnderCapacityLimit(Collection<Broker> brokers, Resource resource, CapacityLimitProvider capacityLimitProvider) {
        return this.brokersUnderCapacityLimit(brokers, Collections.singletonList(resource), capacityLimitProvider);
    }

    public List<Broker> brokersUnderCapacityLimit(Collection<Broker> brokers, List<Resource> resources, CapacityLimitProvider capacityLimitProvider) {
        ArrayList<Broker> brokersUnderThreshold = new ArrayList<Broker>();
        for (Broker broker : brokers) {
            double capacityLimit = capacityLimitProvider.capacityLimit(broker);
            double utilization = resources.stream().mapToDouble(r -> ClusterModelHelper.resourceStatsFor(broker, r).load().expectedUtilizationFor((Resource)((Object)r))).sum();
            if (utilization >= capacityLimit) continue;
            brokersUnderThreshold.add(broker);
        }
        return brokersUnderThreshold;
    }

    public List<Broker> brokersOverThreshold(Collection<Broker> originalBrokers, Resource resource, double utilizationThreshold) {
        ArrayList<Broker> brokersOverThreshold = new ArrayList<Broker>();
        for (Broker broker : originalBrokers) {
            ResourceStats resourceStats = ClusterModelHelper.resourceStatsFor(broker, resource);
            double capacityLimit = resourceStats.capacity().totalCapacityFor(resource) * utilizationThreshold;
            double utilization = resourceStats.load().expectedUtilizationFor(resource);
            if (utilization <= capacityLimit) continue;
            brokersOverThreshold.add(broker);
        }
        return brokersOverThreshold;
    }

    public void sanityCheck() {
        HashMap<String, Integer> errorMsgAndNumWindows = new HashMap<String, Integer>();
        int expectedNumWindows = this.load.numWindows();
        for (Map.Entry<Integer, Load> entry : this.potentialLeadershipLoadByBrokerId.entrySet()) {
            int n = entry.getKey();
            Load load = entry.getValue();
            if (load.numWindows() == expectedNumWindows || this.broker(n).replicas().size() == 0) continue;
            errorMsgAndNumWindows.put(String.format("Leadership(%d)", n), load.numWindows());
        }
        for (Rack rack : this.racksById.values()) {
            if (rack.load().numWindows() != expectedNumWindows && rack.replicas().size() != 0) {
                errorMsgAndNumWindows.put(String.format("Rack(%s)", rack.id()), rack.load().numWindows());
            }
            for (Host host : rack.hosts()) {
                if (host.load().numWindows() != expectedNumWindows && host.replicas().size() != 0) {
                    errorMsgAndNumWindows.put(String.format("Host(%s)", host.name()), host.load().numWindows());
                }
                for (Broker broker : rack.brokers()) {
                    if (broker.load().numWindows() != expectedNumWindows && broker.replicas().size() != 0) {
                        errorMsgAndNumWindows.put(String.format("Broker(%d)", broker.id()), broker.load().numWindows());
                    }
                    for (Replica replica : broker.replicas()) {
                        if (replica.load().numWindows() == expectedNumWindows) continue;
                        errorMsgAndNumWindows.put(String.format("Replica(%s-%d)", replica.topicPartition(), broker.id()), replica.load().numWindows());
                    }
                }
            }
        }
        StringBuilder exceptionMsg = new StringBuilder();
        for (Map.Entry entry : errorMsgAndNumWindows.entrySet()) {
            exceptionMsg.append(String.format("[%s: %d]%n", entry.getKey(), entry.getValue()));
        }
        if (exceptionMsg.length() > 0) {
            throw new IllegalArgumentException(String.format("Loads must have all have %d windows. Following loads violate this constraint with specified number of windows: %s", expectedNumWindows, exceptionMsg));
        }
        String string = "Inconsistent load distribution.";
        for (Broker broker : this.allBrokers()) {
            for (Resource resource : Resource.cachedValues()) {
                double sumOfReplicaUtilization = 0.0;
                for (Replica replica : broker.replicas()) {
                    sumOfReplicaUtilization += replica.load().expectedUtilizationFor(resource);
                }
                double brokerUtilization = broker.load().expectedUtilizationFor(resource);
                if (AnalyzerUtils.compare(sumOfReplicaUtilization, brokerUtilization, resource) == 0) continue;
                throw new IllegalArgumentException(String.format("%s Broker utilization for %s is different from the total replica utilization in the broker with id: %d. Sum of the replica utilization: %f, broker utilization: %f", new Object[]{string, resource, broker.id(), sumOfReplicaUtilization, brokerUtilization}));
            }
        }
        HashMap<Resource, Double> hashMap = new HashMap<Resource, Double>(Resource.cachedValues().size());
        for (Rack rack : this.racksById.values()) {
            HashMap<Resource, Double> sumOfHostUtilizationByResource = new HashMap<Resource, Double>(Resource.cachedValues().size());
            for (Host host : rack.hosts()) {
                for (Resource resource : Resource.cachedValues()) {
                    double sumOfBrokerUtilization = 0.0;
                    for (Broker broker : host.brokers()) {
                        sumOfBrokerUtilization += broker.load().expectedUtilizationFor(resource);
                    }
                    double hostUtilization = host.load().expectedUtilizationFor(resource);
                    if (AnalyzerUtils.compare(sumOfBrokerUtilization, hostUtilization, resource) != 0) {
                        throw new IllegalArgumentException(String.format("%s Host utilization for %s is different from the total broker utilization in the host : %s. Sum of the broker utilization: %f, host utilization: %f", new Object[]{string, resource, host.name(), sumOfBrokerUtilization, hostUtilization}));
                    }
                    sumOfHostUtilizationByResource.compute(resource, (k, v) -> (v == null ? 0.0 : v) + hostUtilization);
                }
            }
            for (Map.Entry entry : sumOfHostUtilizationByResource.entrySet()) {
                Resource resource = (Resource)((Object)entry.getKey());
                double sumOfHostsUtil = (Double)entry.getValue();
                double rackUtilization = rack.load().expectedUtilizationFor(resource);
                if (AnalyzerUtils.compare(rackUtilization, sumOfHostsUtil, resource) != 0) {
                    throw new IllegalArgumentException(String.format("%s Rack utilization for %s is different from the total host utilization in rack : %s. Sum of the host utilization: %f, rack utilization: %f", new Object[]{string, resource, rack.id(), sumOfHostsUtil, rackUtilization}));
                }
                hashMap.compute(resource, (k, v) -> (v == null ? 0.0 : v) + sumOfHostsUtil);
            }
        }
        for (Map.Entry entry : hashMap.entrySet()) {
            Resource resource;
            resource = (Resource)((Object)entry.getKey());
            double sumOfRackUtil = (Double)entry.getValue();
            double clusterUtilization = this.load.expectedUtilizationFor(resource);
            if (AnalyzerUtils.compare(this.load.expectedUtilizationFor(resource), sumOfRackUtil, resource) == 0) continue;
            throw new IllegalArgumentException(String.format("%s Cluster utilization for %s is different from the total rack utilization in the cluster. Sum of the rack utilization: %f, cluster utilization: %f", new Object[]{string, resource, sumOfRackUtil, clusterUtilization}));
        }
        for (Broker broker : this.allBrokers()) {
            double sumOfLeaderOfReplicaUtilization = 0.0;
            for (Replica replica : broker.replicas()) {
                sumOfLeaderOfReplicaUtilization += this.partition(replica.topicPartition()).leader().load().expectedUtilizationFor(Resource.NW_OUT);
            }
            double d = this.potentialLeadershipLoadByBrokerId.get(broker.id()).expectedUtilizationFor(Resource.NW_OUT);
            if (AnalyzerUtils.compare(sumOfLeaderOfReplicaUtilization, d, Resource.NW_OUT) != 0) {
                throw new IllegalArgumentException(String.format("%s Leadership utilization for %s is different from the total utilization leader of replicas in the broker with id: %d. Expected: %f Received: %f", new Object[]{string, Resource.NW_OUT, broker.id(), sumOfLeaderOfReplicaUtilization, d}));
            }
            for (Resource resource : Resource.cachedValues()) {
                double cachedLoad;
                double leaderSum;
                if (resource == Resource.CPU || AnalyzerUtils.compare(leaderSum = broker.leaderReplicas().stream().mapToDouble(r -> r.load().expectedUtilizationFor(resource)).sum(), cachedLoad = broker.leadershipLoadForNwResources().expectedUtilizationFor(resource), resource) == 0) continue;
                throw new IllegalArgumentException(String.format("%s Leadership load for resource %s is %f but recomputed sum is %f", new Object[]{string, resource, cachedLoad, leaderSum}));
            }
        }
    }

    public BrokerStats brokerStats() {
        BrokerStats brokerStats = new BrokerStats();
        this.allBrokers().forEach(broker -> {
            double leaderBytesInRate = broker.leadershipLoadForNwResources().expectedUtilizationFor(Resource.NW_IN);
            double cpuUsagePercent = broker.load().expectedUtilizationFor(Resource.CPU) / broker.capacity().totalCapacityFor(Resource.CPU);
            brokerStats.addSingleBrokerStats(broker.host().name(), broker.id(), broker.strategy(), broker.replicas().isEmpty() ? 0.0 : broker.load().expectedUtilizationFor(Resource.DISK), cpuUsagePercent, leaderBytesInRate, broker.load().expectedUtilizationFor(Resource.NW_IN) - leaderBytesInRate, broker.load().expectedUtilizationFor(Resource.NW_OUT), this.potentialLeadershipLoadFor(broker.id()).expectedUtilizationFor(Resource.NW_OUT), broker.replicas().size(), broker.leaderReplicas().size(), this.capacityEstimationInfoByBrokerId.get(broker.id()) != null, broker.capacity().totalCapacityFor(Resource.DISK));
        });
        return brokerStats;
    }

    public void writeTo(OutputStream out) throws IOException {
        String cluster = String.format("<Cluster>%n", new Object[0]);
        out.write(cluster.getBytes(StandardCharsets.UTF_8));
        for (Rack rack : this.racksById.values()) {
            rack.writeTo(out);
        }
        out.write("</Cluster>".getBytes(StandardCharsets.UTF_8));
    }

    public Optional<TopicPlacement> getTopicPlacement(String topic) {
        return this.topicPlacements.filter(stringTopicPlacementMap -> stringTopicPlacementMap.containsKey(topic)).map(stringTopicPlacementMap -> (TopicPlacement)stringTopicPlacementMap.get(topic));
    }

    public void setTopicPlacements(Map<String, TopicPlacement> topicPlacements) {
        this.topicPlacements = Optional.ofNullable(topicPlacements);
    }

    public void setReplicaExclusions(Set<Integer> brokerIds) {
        this.activeBrokerReplicaExclusions = Collections.unmodifiableSet(new HashSet<Integer>(brokerIds));
    }

    public Set<Integer> excludedBrokers() {
        return this.activeBrokerReplicaExclusions;
    }

    public void addReassigningPartition(TopicPartition tp) {
        this.reassigningPartitions.add(tp);
    }

    public boolean hasReassigningPartitions() {
        return !this.reassigningPartitions.isEmpty();
    }

    public Set<TopicPartition> reassigningPartitions() {
        return new HashSet<TopicPartition>(this.reassigningPartitions);
    }

    public String toString() {
        return String.format("ClusterModel[brokerCount=%d,partitionCount=%d,aliveBrokerCount=%d]", this.brokers.size(), this.partitionsByTopicPartition.size(), this.aliveBrokers.size());
    }

    private void addReplicasUpToReplicationFactor(Partition partition, Short desiredReplicationFactor, Map<String, List<Integer>> brokersByRack) {
        Integer brokerId;
        Cell partitionCell = partition.leader().broker().cell();
        brokersByRack = brokersByRack.entrySet().stream().map(entry -> new AbstractMap.SimpleEntry((String)entry.getKey(), ((List)entry.getValue()).stream().filter(partitionCell::containsBroker).collect(Collectors.toList()))).filter(entry -> !((List)entry.getValue()).isEmpty()).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        ArrayList<String> racks = new ArrayList<String>(brokersByRack.keySet());
        int[] cursors = new int[racks.size()];
        int rackCursor = 0;
        ArrayList<Integer> newReplicas = new ArrayList<Integer>();
        HashSet<String> currentOccupiedRack = new HashSet<String>();
        TopicPartition tp = partition.topicPartition();
        for (Replica replica : partition.replicas()) {
            brokerId = replica.broker().id();
            newReplicas.add(brokerId);
            currentOccupiedRack.add(this.brokerIdToRack.get(brokerId).id());
        }
        while (newReplicas.size() < desiredReplicationFactor) {
            String rack = (String)racks.get(rackCursor);
            if (!currentOccupiedRack.contains(rack) || currentOccupiedRack.size() == racks.size()) {
                int cursor = cursors[rackCursor];
                brokerId = brokersByRack.get(rack).get(cursor);
                if (!newReplicas.contains(brokerId)) {
                    newReplicas.add(brokersByRack.get(rack).get(cursor));
                    Load newReplicaLoad = partition.estimatedNewReplicaLoad();
                    this.createReplica(rack, brokerId, tp, partition.replicas().size(), false, false, true, false);
                    this.setReplicaLoad(rack, brokerId, tp, newReplicaLoad.loadByWindows(), newReplicaLoad.windows());
                    currentOccupiedRack.add(rack);
                }
                cursors[rackCursor] = (cursor + 1) % brokersByRack.get(rack).size();
            }
            rackCursor = (rackCursor + 1) % racks.size();
        }
    }

    public Optional<Tenant> tenantOfTopic(String topic) {
        Optional tenantIdOpt = TenantUtils.extractTenantId((String)topic);
        return tenantIdOpt.map(this.tenantsById::get);
    }

    public double expectedUtilizationInEligibleSourceBrokersFor(Resource resource) {
        return this.utilization.eligibleSourceUtilization().map(load -> load.expectedUtilizationFor(resource)).orElse(0.0);
    }

    public double eligibleDestinationCapacityFor(Resource resource) {
        return this.capacity().eligibleDestinationCapacityFor(resource);
    }

    public ClusterModelCheckpoint checkpoint() {
        return new ClusterModelCheckpoint(++this.lastCheckpointId);
    }

    public void rollbackReplicaActionsUntil(ClusterModelCheckpoint checkpoint) {
        block5: while (!this.actionsHistoryItem.isEmpty()) {
            ReplicaAction replicaAction = this.actionsHistoryItem.peek();
            if (replicaAction.checkpointId < checkpoint.checkpointId()) break;
            replicaAction = this.actionsHistoryItem.pop();
            switch (replicaAction.actionType.ordinal()) {
                case 0: {
                    this.relocateReplica(replicaAction.topicPartition, replicaAction.destinationBrokerId, replicaAction.sourceBrokerId, ReplicaRelocationContext.forReplicaMovementAction(this.broker(replicaAction.destinationBrokerId).replica(replicaAction.topicPartition), replicaAction.sourceBrokerId), false);
                    continue block5;
                }
                case 1: {
                    this.relocateLeadership(replicaAction.topicPartition, replicaAction.destinationBrokerId, replicaAction.sourceBrokerId, false);
                    continue block5;
                }
                case 2: {
                    this.changeObservership(replicaAction.topicPartition, replicaAction.sourceBrokerId, false);
                    continue block5;
                }
            }
            throw new IllegalStateException("Unrecognized replica action type " + String.valueOf((Object)replicaAction.actionType));
        }
    }

    public List<ReplicaAction> actionsHistoryItem() {
        return Collections.unmodifiableList(this.actionsHistoryItem);
    }

    public static final class ReplicaAction {
        private final TopicPartition topicPartition;
        private final int sourceBrokerId;
        private final int destinationBrokerId;
        private final ReplicaActionType actionType;
        private final long checkpointId;

        public ReplicaAction(TopicPartition topicPartition, int sourceBrokerId, int destinationBrokerId, ReplicaActionType actionType, long checkpointId) {
            this.topicPartition = topicPartition;
            this.sourceBrokerId = sourceBrokerId;
            this.destinationBrokerId = destinationBrokerId;
            this.actionType = actionType;
            this.checkpointId = checkpointId;
        }

        public TopicPartition getTopicPartition() {
            return this.topicPartition;
        }

        public int getSourceBrokerId() {
            return this.sourceBrokerId;
        }

        public int getDestinationBrokerId() {
            return this.destinationBrokerId;
        }

        public ReplicaActionType getActionType() {
            return this.actionType;
        }

        public long getCheckpointId() {
            return this.checkpointId;
        }

        public static enum ReplicaActionType {
            RELOCATION,
            LEADERSHIP_CHANGE,
            OBSERVER_SWAP;

        }
    }

    @FunctionalInterface
    public static interface CapacityLimitProvider {
        public double capacityLimit(Broker var1);
    }

    @FunctionalInterface
    public static interface ThresholdProvider {
        public double threshold(Broker var1);
    }

    public static class NonExistentBrokerException
    extends Exception {
        public NonExistentBrokerException(String msg) {
            super(msg);
        }
    }
}

