/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.server;

import com.linkedin.kafka.cruisecontrol.client.BlockingSendClient;
import com.linkedin.kafka.cruisecontrol.client.ConnectionException;
import com.linkedin.kafka.cruisecontrol.common.AdminClientResult;
import com.linkedin.kafka.cruisecontrol.common.KafkaCluster;
import com.linkedin.kafka.cruisecontrol.common.RetryUtils;
import com.linkedin.kafka.cruisecontrol.common.SbkAdminUtils;
import com.linkedin.kafka.cruisecontrol.config.KafkaCruiseControlConfig;
import io.confluent.databalancer.utils.OperationRetryer;
import io.confluent.databalancer.utils.RetryableResult;
import java.io.IOException;
import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.errors.ApiException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.InitiateShutdownRequest;
import org.apache.kafka.common.requests.InitiateShutdownResponse;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.network.BrokerEndPoint;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BrokerShutdownManager {
    private static final Logger LOG = LoggerFactory.getLogger(BrokerShutdownManager.class);
    private static final int SHUTDOWN_WAIT_RETRY_DELAY_MS = 100;
    private final BlockingSendClient.Builder blockingSendClientBuilder;
    private final Time time;
    private final SbkAdminUtils adminUtils;
    private final int apiTimeoutMs;
    private final long shutdownWaitMs;
    private final int maxAttempts;
    private final long retryDelayMs;

    public BrokerShutdownManager(SbkAdminUtils adminUtils, KafkaCruiseControlConfig config, BlockingSendClient.Builder blockingSendClientBuilder, Time time) {
        this.adminUtils = adminUtils;
        this.apiTimeoutMs = config.getInt("default.api.timeout.ms");
        this.shutdownWaitMs = config.getLong("broker.removal.shutdown.timeout.ms");
        this.blockingSendClientBuilder = blockingSendClientBuilder;
        this.time = time;
        this.maxAttempts = config.getInt("broker.removal.subtasks.max.attempts");
        this.retryDelayMs = config.getLong("broker.removal.subtasks.retry.interval.ms");
    }

    public Map<Integer, Boolean> maybeShutdownBrokers(Map<Integer, Optional<Long>> brokersToShutdownAndEpochs) throws Exception {
        List brokerIdsWithNoEpochs;
        HashMap<Integer, Boolean> requestSentByBrokerId = new HashMap<Integer, Boolean>();
        Set<Integer> brokerIdsToShutdown = brokersToShutdownAndEpochs.keySet();
        List<Node> aliveBrokerNodesToShutDown = this.fetchAliveNodes(brokerIdsToShutdown);
        Set<Integer> aliveBrokerIdsToShutDown = aliveBrokerNodesToShutDown.stream().map(Node::id).collect(Collectors.toSet());
        if (aliveBrokerIdsToShutDown.isEmpty()) {
            LOG.info("Skipping shutdown of all brokers {} because they're not part of the cluster.", brokerIdsToShutdown);
            return brokerIdsToShutdown.stream().collect(Collectors.toMap(k -> k, v -> false));
        }
        if (aliveBrokerIdsToShutDown.size() != brokerIdsToShutdown.size()) {
            Set deadBrokers = brokerIdsToShutdown.stream().filter(id -> !aliveBrokerIdsToShutDown.contains(id)).collect(Collectors.toSet());
            LOG.info("Skipping shutdown of {}/{} brokers (ids {}) since they're not part of the cluster", new Object[]{deadBrokers.size(), brokerIdsToShutdown.size(), deadBrokers});
            for (Integer deadBroker : deadBrokers) {
                requestSentByBrokerId.put(deadBroker, false);
            }
        }
        if (!(brokerIdsWithNoEpochs = aliveBrokerIdsToShutDown.stream().filter(id -> !((Optional)brokersToShutdownAndEpochs.get(id)).isPresent()).collect(Collectors.toList())).isEmpty()) {
            String errMsg = String.format("Cannot shut down brokers %s because no broker epochs were given for them.", brokerIdsWithNoEpochs);
            LOG.error(errMsg);
            throw new IllegalArgumentException(errMsg);
        }
        for (Node nodeToShutdown : aliveBrokerNodesToShutDown) {
            int brokerId = nodeToShutdown.id();
            try {
                BlockingSendClient shutdownClient = this.blockingSendClientBuilder.build(new BrokerEndPoint(brokerId, nodeToShutdown.host(), nodeToShutdown.port()));
                try {
                    Long brokerEpoch = brokersToShutdownAndEpochs.get(brokerId).get();
                    LOG.debug("Initiating broker shutdown for broker {}", (Object)brokerId);
                    this.initiateBrokerShutdown(shutdownClient, brokerId, brokerEpoch);
                    requestSentByBrokerId.put(brokerId, true);
                }
                finally {
                    if (shutdownClient == null) continue;
                    shutdownClient.close();
                }
            }
            catch (ExecutionException ee) {
                LOG.info("Caught exception while trying to initiate shutdown for broker {}:", (Object)brokerId, (Object)ee);
                if (ee.getCause() instanceof ConnectionException) {
                    LOG.warn("Unable to connect to broker {} for shutdown, is it still alive?", (Object)brokerId);
                    if (this.fetchAliveNodes(Collections.singleton(brokerId)).isEmpty()) {
                        LOG.info("Broker {} appears to have left the cluster, proceeding with shutdown of other nodes", (Object)brokerId);
                        requestSentByBrokerId.put(brokerId, false);
                        continue;
                    }
                    LOG.warn("Broker {} is still in cluster but shutdown attempt failed.", (Object)brokerId);
                    throw ee;
                }
                throw ee;
            }
        }
        LOG.info("Successfully initiated shutdown for brokers {}. Waiting for them to leave the cluster...", aliveBrokerIdsToShutDown);
        this.awaitBrokersShutdown(this.shutdownWaitMs, aliveBrokerIdsToShutDown);
        return requestSentByBrokerId;
    }

    void initiateBrokerShutdown(BlockingSendClient shutdownClient, int brokerId, long brokerEpoch) throws ExecutionException, ApiException, TimeoutException {
        String brokerStr = String.format("broker %d (epoch %d)", brokerId, brokerEpoch);
        try {
            RetryUtils.executeWithRetry(() -> {
                try {
                    InitiateShutdownResponse response = shutdownClient.sendShutdownRequest(new InitiateShutdownRequest.Builder(brokerEpoch));
                    if (response != null && response.data().errorCode() != Errors.NONE.code()) {
                        throw Errors.forCode((short)response.data().errorCode()).exception();
                    }
                    return response;
                }
                catch (ConnectionException e) {
                    throw e;
                }
                catch (IOException e) {
                    LOG.info("Caught IOException (message: {}) while trying to shutdown {}. Assuming broker shut down before it could respond.", (Object)e.getMessage(), (Object)brokerStr);
                    return null;
                }
            }, ex -> ex instanceof ConnectionException || ex instanceof ApiException, this.maxAttempts, this.retryDelayMs);
            LOG.info("Shutdown request for {} was successful", (Object)brokerStr);
        }
        catch (ApiException e) {
            throw e;
        }
        catch (ConnectionException e) {
            throw new ExecutionException(String.format("Failed to connect to %s while trying to send shutdown request.", brokerStr), e);
        }
        catch (Exception e) {
            throw new ExecutionException(String.format("Unexpected exception occurred while trying to send shutdown request for %s", brokerStr), e);
        }
    }

    void awaitBrokersShutdown(long shutdownWaitMs, Set<Integer> allBrokerIdsToAwait) throws InterruptedException, TimeoutException {
        long startTimeMs = this.time.milliseconds();
        HashSet<Integer> leftoverAliveBrokerIdsToShutdown = new HashSet<Integer>(allBrokerIdsToAwait);
        OperationRetryer retryer = new OperationRetryer(this.time, Duration.ofMillis(shutdownWaitMs), Duration.ofMillis(100L), String.format("brokers %s shutdown", allBrokerIdsToAwait));
        retryer.runWithRetries(() -> {
            long nowMs = this.time.milliseconds();
            long elapsedTimeMs = nowMs - startTimeMs;
            AdminClientResult<KafkaCluster> clusterResult = this.adminUtils.describeCluster(this.apiTimeoutMs);
            if (clusterResult.hasException()) {
                LOG.warn("Failed to describe the cluster while awaiting shutdown for brokers {}. Retrying in {}ms", new Object[]{allBrokerIdsToAwait, 100, clusterResult.exception()});
                return RetryableResult.Incomplete.instance();
            }
            if (this.shutdownsCompleted(allBrokerIdsToAwait, leftoverAliveBrokerIdsToShutdown, clusterResult.result(), elapsedTimeMs)) {
                return RetryableResult.Success.of(true);
            }
            return RetryableResult.Incomplete.instance();
        });
    }

    private boolean shutdownsCompleted(Set<Integer> allBrokerIdsToAwait, Set<Integer> leftoverAliveBrokerIdsToShutdown, KafkaCluster latestClusterMetadata, long elapsedTimeMs) {
        Set<Integer> latestAliveBrokerIds = latestClusterMetadata.nodes().stream().map(Node::id).filter(allBrokerIdsToAwait::contains).collect(Collectors.toSet());
        this.maybeLogShutdownDetected(allBrokerIdsToAwait, leftoverAliveBrokerIdsToShutdown, latestAliveBrokerIds, elapsedTimeMs);
        if (latestAliveBrokerIds.isEmpty()) {
            LOG.info("All brokers {} were successfully shut down after {}ms", allBrokerIdsToAwait, (Object)elapsedTimeMs);
            return true;
        }
        LOG.debug("Brokers {} are still part of the cluster ({}ms after the shutdown initiation)", latestAliveBrokerIds, (Object)elapsedTimeMs);
        return false;
    }

    private void maybeLogShutdownDetected(Set<Integer> allBrokerIdsToShutdown, Set<Integer> leftoverAliveBrokerIdsToShutdown, Set<Integer> latestAliveBrokerIds, long elapsedTimeMs) {
        for (Integer brokerToRemove : allBrokerIdsToShutdown) {
            if (!leftoverAliveBrokerIdsToShutdown.contains(brokerToRemove)) continue;
            if (!latestAliveBrokerIds.contains(brokerToRemove)) {
                LOG.info("Broker {} has left the cluster successfully ({}ms after shutdown initiation)", (Object)brokerToRemove, (Object)elapsedTimeMs);
                leftoverAliveBrokerIdsToShutdown.remove(brokerToRemove);
                continue;
            }
            LOG.debug("Broker {} is still part of the cluster ({}ms after the shutdown initiation)", (Object)brokerToRemove, (Object)elapsedTimeMs);
        }
    }

    private List<Node> fetchAliveNodes(Set<Integer> brokerIds) throws ExecutionException, InterruptedException {
        AdminClientResult<KafkaCluster> clusterResult = this.adminUtils.describeCluster(this.apiTimeoutMs);
        if (clusterResult.hasException()) {
            throw new ExecutionException("Failed to describe the cluster", clusterResult.exception());
        }
        KafkaCluster cluster = clusterResult.result();
        return cluster.nodes().stream().filter(node -> brokerIds.contains(node.id())).collect(Collectors.toList());
    }

    public void unregisterBrokers(Set<Integer> brokerIdsToUnregister) throws Exception {
        LOG.info("Attempting to unregister brokers {}...", brokerIdsToUnregister);
        this.adminUtils.unregisterBrokers(brokerIdsToUnregister);
    }
}

