/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.execution.util;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Streams;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.confluent.ksql.execution.expression.tree.Cast;
import io.confluent.ksql.execution.expression.tree.CreateArrayExpression;
import io.confluent.ksql.execution.expression.tree.CreateMapExpression;
import io.confluent.ksql.execution.expression.tree.CreateStructExpression;
import io.confluent.ksql.execution.expression.tree.Expression;
import io.confluent.ksql.execution.expression.tree.Literal;
import io.confluent.ksql.execution.expression.tree.NullLiteral;
import io.confluent.ksql.execution.expression.tree.StringLiteral;
import io.confluent.ksql.execution.expression.tree.Type;
import io.confluent.ksql.execution.util.ExpressionTypeManager;
import io.confluent.ksql.execution.util.Literals;
import io.confluent.ksql.schema.ksql.DefaultSqlValueCoercer;
import io.confluent.ksql.schema.ksql.types.SqlArray;
import io.confluent.ksql.schema.ksql.types.SqlBaseType;
import io.confluent.ksql.schema.ksql.types.SqlMap;
import io.confluent.ksql.schema.ksql.types.SqlStruct;
import io.confluent.ksql.schema.ksql.types.SqlType;
import io.confluent.ksql.schema.ksql.types.SqlTypes;
import io.confluent.ksql.util.DecimalUtil;
import io.confluent.ksql.util.KsqlException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public final class CoercionUtil {
    private static final DefaultSqlValueCoercer LITERAL_COERCER = DefaultSqlValueCoercer.LAX;
    private static final DefaultSqlValueCoercer EXPRESSION_COERCER = DefaultSqlValueCoercer.STRICT;
    private static final BigInteger INT_MAX = BigInteger.valueOf(Integer.MAX_VALUE);
    private static final BigInteger INT_MIN = BigInteger.valueOf(Integer.MIN_VALUE);
    private static final BigInteger BIGINT_MAX = BigInteger.valueOf(Long.MAX_VALUE);
    private static final BigInteger BIGINT_MIN = BigInteger.valueOf(Long.MIN_VALUE);

    private CoercionUtil() {
    }

    static Result coerceUserList(Collection<Expression> expressions, ExpressionTypeManager typeManager) {
        return CoercionUtil.coerceUserList(expressions, typeManager, Collections.emptyMap());
    }

    public static Result coerceUserList(Collection<Expression> expressions, ExpressionTypeManager typeManager, Map<String, SqlType> lambdaTypeMapping) {
        return new UserListCoercer(typeManager, lambdaTypeMapping).coerce(expressions);
    }

    public static final class Result {
        private final Optional<SqlType> commonType;
        private final ImmutableList<Expression> expressions;

        public Result(Optional<SqlType> commonType, List<Expression> expressions) {
            this.commonType = Objects.requireNonNull(commonType, "commonType");
            this.expressions = ImmutableList.copyOf((Collection)Objects.requireNonNull(expressions, "expressions"));
        }

        public Optional<SqlType> commonType() {
            return this.commonType;
        }

        @SuppressFBWarnings(value={"EI_EXPOSE_REP"}, justification="expressions is ImmutableList")
        public List<Expression> expressions() {
            return this.expressions;
        }
    }

    private static final class UserListCoercer {
        private final ExpressionTypeManager typeManager;
        private final Map<String, SqlType> lambdaTypeMapping;

        UserListCoercer(ExpressionTypeManager typeManager, Map<String, SqlType> lambdaTypeMapping) {
            this.typeManager = Objects.requireNonNull(typeManager, "typeManager");
            this.lambdaTypeMapping = Objects.requireNonNull(lambdaTypeMapping, "lambdaTypeMapping");
        }

        Result coerce(Collection<Expression> expressions) {
            List<TypedExpression> typedExpressions = this.typedExpressions(expressions);
            Optional<SqlType> commonType = this.resolveCommonType(typedExpressions);
            if (!commonType.isPresent()) {
                return new Result(commonType, (List<Expression>)ImmutableList.copyOf(expressions));
            }
            List<Expression> converted = UserListCoercer.convertToCommonType(typedExpressions, commonType.get());
            return new Result(commonType, converted);
        }

        private List<TypedExpression> typedExpressions(Collection<Expression> expressions) {
            return expressions.stream().map(e -> new TypedExpression(this.typeManager.getExpressionSqlType((Expression)((Object)e), this.lambdaTypeMapping), (Expression)((Object)e))).collect(Collectors.toList());
        }

        private Optional<SqlType> resolveCommonType(List<TypedExpression> expressions) {
            Optional<SqlType> result = Optional.empty();
            for (TypedExpression e : expressions) {
                result = this.resolveCommonType(e, result);
            }
            return result;
        }

        private Optional<SqlType> resolveCommonType(TypedExpression exp, Optional<SqlType> commonType) {
            if (!commonType.isPresent()) {
                return exp.type();
            }
            if (!exp.type().isPresent()) {
                return commonType;
            }
            SqlType targetType = commonType.get();
            if (exp.expression() instanceof CreateArrayExpression) {
                return this.resolveCommonArrayType(exp, targetType);
            }
            if (exp.expression() instanceof CreateMapExpression) {
                return this.resolveCommonMapType(exp, targetType);
            }
            if (exp.expression() instanceof CreateStructExpression) {
                return this.resolveCommonStructType(exp, targetType);
            }
            if (exp.expression() instanceof StringLiteral) {
                String value = ((StringLiteral)exp.expression()).getValue();
                if (targetType.baseType().isNumber()) {
                    return UserListCoercer.resolveCommonNumericTypeFromStringLiteral(value, targetType);
                }
                if (targetType.baseType() == SqlBaseType.BOOLEAN) {
                    UserListCoercer.validateStringCanBeCoercedToBoolean(value);
                    return commonType;
                }
            }
            return UserListCoercer.resolveCommonSimpleType(exp, targetType);
        }

        private static void validateStringCanBeCoercedToBoolean(String value) {
            LITERAL_COERCER.coerce((Object)value, (SqlType)SqlTypes.BOOLEAN).orElseThrow(() -> UserListCoercer.invalidSyntaxException(value, (SqlType)SqlTypes.BOOLEAN));
        }

        private Optional<SqlType> resolveCommonArrayType(TypedExpression exp, SqlType targetType) {
            SqlType sourceType = exp.type().orElse(null);
            if (targetType.baseType() != SqlBaseType.ARRAY) {
                throw UserListCoercer.coercionFailureException((Object)exp.expression(), sourceType, targetType, Optional.empty());
            }
            try {
                SqlArray targetArray = (SqlArray)targetType;
                CreateArrayExpression sourceArray = (CreateArrayExpression)exp.expression();
                return this.resolveStructuredCommonType(targetArray.getItemType(), (Collection<Expression>)sourceArray.getValues()).map(SqlTypes::array);
            }
            catch (InvalidCoercionException e) {
                throw UserListCoercer.coercionFailureException((Object)exp.expression(), sourceType, targetType, Optional.of(e));
            }
        }

        private Optional<SqlType> resolveCommonMapType(TypedExpression exp, SqlType targetType) {
            SqlType sourceType = exp.type().orElse(null);
            if (targetType.baseType() != SqlBaseType.MAP) {
                throw UserListCoercer.coercionFailureException((Object)exp.expression(), sourceType, targetType, Optional.empty());
            }
            try {
                SqlMap targetMap = (SqlMap)targetType;
                CreateMapExpression sourceMap = (CreateMapExpression)exp.expression();
                Optional<SqlType> keyType = this.resolveStructuredCommonType(targetMap.getKeyType(), (Collection<Expression>)sourceMap.getMap().keySet());
                Optional<SqlType> valueType = this.resolveStructuredCommonType(targetMap.getValueType(), (Collection<Expression>)sourceMap.getMap().values());
                return keyType.isPresent() && valueType.isPresent() ? Optional.of(SqlTypes.map((SqlType)keyType.get(), (SqlType)valueType.get())) : Optional.empty();
            }
            catch (InvalidCoercionException e) {
                throw UserListCoercer.coercionFailureException((Object)exp.expression(), sourceType, targetType, Optional.of(e));
            }
        }

        private Optional<SqlType> resolveCommonStructType(TypedExpression exp, SqlType targetType) {
            SqlType sourceType = exp.type().orElse(null);
            if (targetType.baseType() != SqlBaseType.STRUCT) {
                throw UserListCoercer.coercionFailureException((Object)exp.expression(), sourceType, targetType, Optional.empty());
            }
            try {
                SqlStruct targetStruct = (SqlStruct)targetType;
                CreateStructExpression sourceStruct = (CreateStructExpression)exp.expression();
                List fieldNames = Streams.concat((Stream[])new Stream[]{sourceStruct.getFields().stream().map(CreateStructExpression.Field::getName), targetStruct.fields().stream().map(SqlStruct.Field::name)}).distinct().collect(Collectors.toList());
                SqlStruct.Builder builder = SqlTypes.struct();
                for (String fieldName : fieldNames) {
                    Optional<Expression> sourceFieldValue = sourceStruct.getFields().stream().filter(f -> f.getName().equals(fieldName)).findFirst().map(CreateStructExpression.Field::getValue);
                    Optional<SqlType> sourceFieldType = sourceFieldValue.map(sourceExpression -> this.typeManager.getExpressionSqlType((Expression)((Object)sourceExpression), this.lambdaTypeMapping));
                    Optional<SqlType> targetFieldType = targetStruct.field(fieldName).map(SqlStruct.Field::type);
                    SqlType fieldType = !targetFieldType.isPresent() ? sourceFieldType.orElseThrow(IllegalStateException::new) : (!sourceFieldType.isPresent() ? targetFieldType.orElseThrow(IllegalStateException::new) : this.resolveCommonType(new TypedExpression(sourceFieldType.get(), sourceFieldValue.get()), targetFieldType).orElseThrow(IllegalStateException::new));
                    builder.field(fieldName, fieldType);
                }
                return Optional.of(builder.build());
            }
            catch (InvalidCoercionException e) {
                throw UserListCoercer.coercionFailureException((Object)exp.expression(), sourceType, targetType, Optional.of(e));
            }
        }

        private Optional<SqlType> resolveStructuredCommonType(SqlType targetType, Collection<Expression> expressions) {
            List<TypedExpression> typedElements = this.typedExpressions(expressions);
            if (typedElements.isEmpty()) {
                return Optional.of(targetType);
            }
            ImmutableList typedExpressions = ImmutableList.builder().add((Object)new TypedExpression(targetType, new NullLiteral())).addAll(typedElements).build();
            return this.resolveCommonType((List<TypedExpression>)typedExpressions);
        }

        private static Optional<SqlType> resolveCommonNumericTypeFromStringLiteral(String value, SqlType targetType) {
            Preconditions.checkArgument((boolean)targetType.baseType().isNumber());
            try {
                SqlType sourceType = UserListCoercer.getStringNumericType(value);
                if (sourceType.baseType() == SqlBaseType.DOUBLE || targetType.baseType() == SqlBaseType.DOUBLE) {
                    return Optional.of(SqlTypes.DOUBLE);
                }
                if (sourceType.baseType() == SqlBaseType.DECIMAL || targetType.baseType() == SqlBaseType.DECIMAL) {
                    return Optional.of(DecimalUtil.widen((SqlType)sourceType, (SqlType)targetType));
                }
                return Optional.of(sourceType.baseType().canImplicitlyCast(targetType.baseType()) ? targetType : sourceType);
            }
            catch (NumberFormatException e) {
                throw UserListCoercer.invalidSyntaxException(value, targetType);
            }
        }

        private static SqlType getStringNumericType(String value) {
            boolean containsDpOrScientific;
            BigDecimal result = new BigDecimal(value.trim());
            boolean bl = containsDpOrScientific = value.contains(".") || value.contains("e") || value.contains("E");
            if (!containsDpOrScientific && result.scale() == 0) {
                BigInteger bi = result.toBigIntegerExact();
                if (0 < bi.compareTo(INT_MIN) && bi.compareTo(INT_MAX) < 0) {
                    return SqlTypes.INTEGER;
                }
                if (0 < bi.compareTo(BIGINT_MIN) && bi.compareTo(BIGINT_MAX) < 0) {
                    return SqlTypes.BIGINT;
                }
            }
            return DecimalUtil.fromValue((BigDecimal)result);
        }

        private static Optional<SqlType> resolveCommonSimpleType(TypedExpression exp, SqlType targetType) {
            SqlType sourceType = exp.type().orElseThrow(IllegalStateException::new);
            DefaultSqlValueCoercer coercer = exp.expression() instanceof Literal ? LITERAL_COERCER : EXPRESSION_COERCER;
            Optional common0 = coercer.canCoerce(sourceType, targetType);
            if (common0.isPresent()) {
                return common0;
            }
            Optional common1 = coercer.canCoerce(targetType, sourceType);
            if (common1.isPresent()) {
                return common1;
            }
            throw UserListCoercer.coercionFailureException((Object)exp.expression(), sourceType, targetType, Optional.empty());
        }

        private static List<Expression> convertToCommonType(List<TypedExpression> typedExpressions, SqlType commonType) {
            return typedExpressions.stream().map(e -> UserListCoercer.convertToCommonType(e, commonType)).collect(Collectors.toList());
        }

        private static Expression convertToCommonType(TypedExpression exp, SqlType commonType) {
            if (exp.type().map(type -> type.equals(commonType)).orElse(true).booleanValue()) {
                return exp.expression();
            }
            if (exp.expression() instanceof Literal) {
                return UserListCoercer.convertLiteralToCommonType(exp, commonType);
            }
            return new Cast(exp.expression(), new Type(commonType));
        }

        private static Expression convertLiteralToCommonType(TypedExpression exp, SqlType commonType) {
            Literal literal = (Literal)exp.expression();
            Object coerced = LITERAL_COERCER.coerce(literal.getValue(), commonType).orElseThrow(IllegalStateException::new).orElseThrow(IllegalStateException::new);
            return Literals.getFactory(commonType.baseType()).apply(coerced);
        }

        private static KsqlException invalidSyntaxException(Object value, SqlType targetType) {
            return new KsqlException("invalid input syntax for type " + String.valueOf(targetType.baseType()) + ": \"" + String.valueOf(value) + "\".");
        }

        private static InvalidCoercionException coercionFailureException(Object value, SqlType sourceType, SqlType targetType, Optional<InvalidCoercionException> cause) {
            String msg = "operator does not exist: " + String.valueOf(targetType) + " = " + String.valueOf(sourceType) + " (" + String.valueOf(value) + ")" + System.lineSeparator() + "Hint: You might need to add explicit type casts.";
            return cause.map(e -> new InvalidCoercionException(msg, (Throwable)((Object)e))).orElseGet(() -> new InvalidCoercionException(msg));
        }

        private static final class TypedExpression {
            private final Optional<SqlType> type;
            private final Expression expression;

            TypedExpression(SqlType type, Expression expression) {
                this.type = Optional.ofNullable(type);
                this.expression = Objects.requireNonNull(expression, "expression");
            }

            public Optional<SqlType> type() {
                return this.type;
            }

            public Expression expression() {
                return this.expression;
            }
        }

        private static final class InvalidCoercionException
        extends KsqlException {
            InvalidCoercionException(String message) {
                super(message);
            }

            InvalidCoercionException(String message, Throwable cause) {
                super(message, cause);
            }
        }
    }
}

