/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.server.plugins.auth;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.confluent.kafka.server.plugins.auth.MultiTenantSaslSecrets;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Paths;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import kafka.server.BrokerSession;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.network.PublicCredential;
import org.apache.kafka.common.utils.Crc32C;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MultiTenantSaslSecretsLoader {
    private static final Logger log = LoggerFactory.getLogger(MultiTenantSaslSecretsLoader.class);
    private final int maxCachedInfos;
    private final LinkedHashMap<String, SecretsFileInfo> files;
    private final ObjectMapper objectMapper;
    private String mostRecentlyLoaded;

    public MultiTenantSaslSecretsLoader(int maxCachedInfos) {
        this.maxCachedInfos = maxCachedInfos;
        this.files = new LinkedHashMap();
        this.objectMapper = new ObjectMapper();
        this.mostRecentlyLoaded = null;
    }

    public MultiTenantSaslSecrets load(String path, long now, long refreshMs) {
        try {
            return this.doLoad(path, now, refreshMs);
        }
        catch (FileNotFoundException e) {
            throw new ConfigException("Config file not found: " + path, (Object)e);
        }
        catch (IOException e) {
            throw new ConfigException("Error reading secret file: " + path, (Object)e);
        }
    }

    private synchronized MultiTenantSaslSecrets doLoad(String path, long now, long refreshMs) throws IOException {
        SecretsFileInfo info = this.files.get(path);
        if (info == null) {
            log.debug("Load API keys from {}", (Object)path);
            return this.loadSecretsFileInfo((String)path, (long)now).secrets;
        }
        this.files.remove(path);
        if (now - info.lastCheckMs < refreshMs) {
            this.files.put(path, info);
            return info.secrets;
        }
        long lastModifiedTimeMs = Files.getLastModifiedTime(Paths.get(path, new String[0]), new LinkOption[0]).toMillis();
        if (lastModifiedTimeMs != info.lastModifiedTimeMs) {
            return this.loadSecretsFileInfo((String)path, (long)now, (long)lastModifiedTimeMs, (MultiTenantSaslSecrets)info.secrets).secrets;
        }
        byte[] buf = Files.readAllBytes(Paths.get(path, new String[0]));
        long crc = Crc32C.compute((byte[])buf, (int)0, (int)buf.length);
        if (crc == info.crc && (long)buf.length == info.fileLength) {
            info.lastCheckMs = now;
            this.files.put(path, info);
            return info.secrets;
        }
        return this.loadSecretsFileInfo((String)path, (long)now, (long)lastModifiedTimeMs, (byte[])buf, (long)crc, (MultiTenantSaslSecrets)info.secrets).secrets;
    }

    private synchronized SecretsFileInfo loadSecretsFileInfo(String path, long now) throws IOException {
        long lastModifiedTimeMs = Files.getLastModifiedTime(Paths.get(path, new String[0]), new LinkOption[0]).toMillis();
        return this.loadSecretsFileInfo(path, now, lastModifiedTimeMs, null);
    }

    private synchronized SecretsFileInfo loadSecretsFileInfo(String path, long now, long lastModifiedTimeMs, MultiTenantSaslSecrets oldSecrets) throws IOException {
        byte[] buf = Files.readAllBytes(Paths.get(path, new String[0]));
        long crc = Crc32C.compute((byte[])buf, (int)0, (int)buf.length);
        return this.loadSecretsFileInfo(path, now, lastModifiedTimeMs, buf, crc, oldSecrets);
    }

    private synchronized SecretsFileInfo loadSecretsFileInfo(String path, long now, long lastModifiedTimeMs, byte[] buf, long crc, MultiTenantSaslSecrets oldSecrets) {
        MultiTenantSaslSecrets secrets;
        try {
            secrets = (MultiTenantSaslSecrets)this.objectMapper.readValue(buf, MultiTenantSaslSecrets.class);
        }
        catch (Exception e) {
            throw new ConfigException("Error parsing secret file " + path, (Object)e);
        }
        SecretsFileInfo info = new SecretsFileInfo(secrets, lastModifiedTimeMs, buf.length, crc, now);
        log.debug("Loaded secrets file with keys {}", secrets.entries().keySet());
        if (oldSecrets != null) {
            this.processDeletedApiKeys(oldSecrets, secrets);
        }
        this.mostRecentlyLoaded = path;
        this.files.put(path, info);
        if (this.files.size() > this.maxCachedInfos) {
            Iterator<Map.Entry<String, SecretsFileInfo>> iter = this.files.entrySet().iterator();
            iter.next();
            iter.remove();
        }
        return info;
    }

    private synchronized void processDeletedApiKeys(MultiTenantSaslSecrets oldSecrets, MultiTenantSaslSecrets newSecrets) {
        newSecrets.deletedCredentials(oldSecrets).forEach(deleted -> {
            log.info("Deleting API key {}", deleted);
            BrokerSession.handleCredentialDeleteForAllSessions((PublicCredential)deleted);
        });
    }

    synchronized boolean contains(String path) {
        return this.files.containsKey(path);
    }

    synchronized String mostRecentlyLoaded() {
        return this.mostRecentlyLoaded;
    }

    static class SecretsFileInfo {
        private final MultiTenantSaslSecrets secrets;
        private final long lastModifiedTimeMs;
        private final long fileLength;
        private final long crc;
        private long lastCheckMs;

        SecretsFileInfo(MultiTenantSaslSecrets secrets, long lastModifiedTimeMs, long fileLength, long crc, long lastCheckMs) {
            this.secrets = secrets;
            this.lastModifiedTimeMs = lastModifiedTimeMs;
            this.fileLength = fileLength;
            this.crc = crc;
            this.lastCheckMs = lastCheckMs;
        }
    }
}

