/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.utilization;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import io.confluent.ksql.KsqlExecutionContext;
import io.confluent.ksql.engine.KsqlEngine;
import io.confluent.ksql.internal.MetricsReporter;
import io.confluent.ksql.query.QueryId;
import io.confluent.ksql.util.PersistentQueryMetadata;
import io.confluent.ksql.util.QueryMetadata;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.kafka.common.Metric;
import org.apache.kafka.streams.KafkaStreams;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class PersistentQuerySaturationMetrics
implements Runnable {
    private static final Logger LOGGER = LogManager.getLogger(PersistentQuerySaturationMetrics.class);
    private static final String QUERY_SATURATION = "node-query-saturation";
    private static final String NODE_QUERY_SATURATION = "max-node-query-saturation";
    private static final String QUERY_THREAD_SATURATION = "query-thread-saturation";
    private static final String STREAMS_TOTAL_BLOCKED_TIME = "blocked-time-ns-total";
    private static final String STREAMS_THREAD_START_TIME = "thread-start-time";
    private static final String STREAMS_THREAD_METRICS_GROUP = "stream-thread-metrics";
    private static final String THREAD_ID = "thread-id";
    private static final String QUERY_ID = "query-id";
    private Map<String, String> customTags;
    private final Map<String, KafkaStreamsSaturation> perKafkaStreamsStats = new HashMap<String, KafkaStreamsSaturation>();
    private final KsqlExecutionContext engine;
    private final Supplier<Instant> time;
    private final MetricsReporter reporter;
    private final Duration window;
    private final Duration sampleMargin;

    public PersistentQuerySaturationMetrics(KsqlEngine engine, MetricsReporter reporter, Duration window, Duration sampleMargin, Map<String, String> customTags) {
        this(Instant::now, engine, reporter, window, sampleMargin, customTags);
    }

    @VisibleForTesting
    PersistentQuerySaturationMetrics(Supplier<Instant> time, KsqlExecutionContext engine, MetricsReporter reporter, Duration window, Duration sampleMargin, Map<String, String> customTags) {
        this.time = Objects.requireNonNull(time, "time");
        this.engine = Objects.requireNonNull(engine, "engine");
        this.reporter = Objects.requireNonNull(reporter, "reporter");
        this.window = Objects.requireNonNull(window, "window");
        this.sampleMargin = Objects.requireNonNull(sampleMargin, "sampleMargin");
        this.customTags = Objects.requireNonNull(customTags, "customTags");
    }

    @Override
    public void run() {
        Instant now = this.time.get();
        try {
            List<PersistentQueryMetadata> queries = this.engine.getPersistentQueries();
            Optional<Double> saturation = queries.stream().collect(Collectors.groupingBy(QueryMetadata::getQueryApplicationId)).entrySet().stream().map(e -> this.measure(now, (String)e.getKey(), (List)e.getValue())).max(PersistentQuerySaturationMetrics::compareSaturation).orElse(Optional.of(0.0));
            saturation.ifPresent(s -> this.report(now, (double)s));
            Set appIds = queries.stream().map(QueryMetadata::getQueryApplicationId).collect(Collectors.toSet());
            for (String appId : Sets.difference(new HashSet<String>(this.perKafkaStreamsStats.keySet()), appIds)) {
                this.perKafkaStreamsStats.get(appId).cleanup(this.reporter);
                this.perKafkaStreamsStats.remove(appId);
            }
        }
        catch (RuntimeException e2) {
            LOGGER.error("Error collecting saturation", (Throwable)e2);
            throw e2;
        }
    }

    private static int compareSaturation(Optional<Double> a, Optional<Double> b) {
        if (!a.isPresent()) {
            return 1;
        }
        if (!b.isPresent()) {
            return -1;
        }
        return Double.compare(a.get(), b.get());
    }

    private Optional<Double> measure(Instant now, String appId, List<PersistentQueryMetadata> queryMetadata) {
        KafkaStreamsSaturation ksSaturation = this.perKafkaStreamsStats.computeIfAbsent(appId, k -> new KafkaStreamsSaturation(this.window, this.sampleMargin));
        Optional<KafkaStreams> kafkaStreams = queryMetadata.stream().filter(q -> q.getKafkaStreams() != null).map(QueryMetadata::getKafkaStreams).findFirst();
        if (!kafkaStreams.isPresent()) {
            return Optional.of(0.0);
        }
        List<QueryId> queryIds = queryMetadata.stream().map(QueryMetadata::getQueryId).collect(Collectors.toList());
        return ksSaturation.measure(now, queryIds, kafkaStreams.get(), this.reporter);
    }

    private void report(Instant now, double saturation) {
        LOGGER.info("reporting node-level saturation {}", (Object)saturation);
        this.reporter.report((List<MetricsReporter.DataPoint>)ImmutableList.of((Object)new MetricsReporter.DataPoint(now, NODE_QUERY_SATURATION, saturation, this.customTags)));
    }

    private Map<String, String> getTags(String key, String value) {
        HashMap<String, String> newTags = new HashMap<String, String>(this.customTags);
        newTags.put(key, value);
        return newTags;
    }

    private final class KafkaStreamsSaturation {
        private final Set<QueryId> queryIds = new HashSet<QueryId>();
        private final Map<String, ThreadSaturation> perThreadSaturation = new HashMap<String, ThreadSaturation>();
        private final Duration window;
        private final Duration sampleMargin;

        private KafkaStreamsSaturation(Duration window, Duration sampleMargin) {
            this.window = Objects.requireNonNull(window, "window");
            this.sampleMargin = Objects.requireNonNull(sampleMargin, "sampleMargin");
        }

        private void reportThreadSaturation(Instant now, double saturation, String name, MetricsReporter reporter) {
            LOGGER.info("Reporting thread saturation {} for {}", (Object)saturation, (Object)name);
            reporter.report((List<MetricsReporter.DataPoint>)ImmutableList.of((Object)new MetricsReporter.DataPoint(now, PersistentQuerySaturationMetrics.QUERY_THREAD_SATURATION, saturation, PersistentQuerySaturationMetrics.this.getTags(PersistentQuerySaturationMetrics.THREAD_ID, name))));
        }

        private void reportQuerySaturation(Instant now, double saturation, MetricsReporter reporter) {
            for (QueryId queryId : this.queryIds) {
                LOGGER.info("Reporting query saturation {} for {}", (Object)saturation, (Object)queryId);
                reporter.report((List<MetricsReporter.DataPoint>)ImmutableList.of((Object)new MetricsReporter.DataPoint(now, PersistentQuerySaturationMetrics.QUERY_SATURATION, saturation, PersistentQuerySaturationMetrics.this.getTags(PersistentQuerySaturationMetrics.QUERY_ID, queryId.toString()))));
            }
        }

        private void updateQueryIds(List<QueryId> newQueryIds, MetricsReporter reporter) {
            for (QueryId queryId : Sets.difference(this.queryIds, new HashSet<QueryId>(newQueryIds))) {
                reporter.cleanup(PersistentQuerySaturationMetrics.QUERY_SATURATION, (Map<String, String>)ImmutableMap.of((Object)PersistentQuerySaturationMetrics.QUERY_ID, (Object)queryId.toString()));
            }
            this.queryIds.clear();
            this.queryIds.addAll(newQueryIds);
        }

        private Map<String, Map<String, Metric>> metricsByThread(KafkaStreams kafkaStreams) {
            return kafkaStreams.metrics().values().stream().filter(m -> m.metricName().group().equals(PersistentQuerySaturationMetrics.STREAMS_THREAD_METRICS_GROUP)).collect(Collectors.groupingBy(m -> (String)m.metricName().tags().get(PersistentQuerySaturationMetrics.THREAD_ID), Collectors.toMap(m -> m.metricName().name(), m -> m, (a, b) -> a)));
        }

        private Optional<Double> measure(Instant now, List<QueryId> queryIds, KafkaStreams kafkaStreams, MetricsReporter reporter) {
            this.updateQueryIds(queryIds, reporter);
            Map<String, Map<String, Metric>> byThread = this.metricsByThread(kafkaStreams);
            Optional<Double> saturation = Optional.of(0.0);
            for (Map.Entry<String, Map<String, Metric>> entry : byThread.entrySet()) {
                String threadName = entry.getKey();
                Map<String, Metric> metricsForThread = entry.getValue();
                if (!metricsForThread.containsKey(PersistentQuerySaturationMetrics.STREAMS_TOTAL_BLOCKED_TIME) || !metricsForThread.containsKey(PersistentQuerySaturationMetrics.STREAMS_THREAD_START_TIME)) {
                    LOGGER.info("Missing required metrics for thread: {}", (Object)threadName);
                    continue;
                }
                double totalBlocked = (Double)metricsForThread.get(PersistentQuerySaturationMetrics.STREAMS_TOTAL_BLOCKED_TIME).metricValue();
                long startTime = (Long)metricsForThread.get(PersistentQuerySaturationMetrics.STREAMS_THREAD_START_TIME).metricValue();
                ThreadSaturation threadSaturation = this.perThreadSaturation.computeIfAbsent(threadName, k -> {
                    LOGGER.debug("Adding saturation for new thread: {}", k);
                    return new ThreadSaturation(threadName, startTime, this.window, this.sampleMargin);
                });
                LOGGER.debug("Record thread {} sample {} {}", (Object)threadName, (Object)totalBlocked, (Object)startTime);
                BlockedTimeSample blockedTimeSample = new BlockedTimeSample(now, totalBlocked);
                Optional<Double> measured = threadSaturation.measure(now, blockedTimeSample);
                LOGGER.debug("Measured value for thread {}: {}", (Object)threadName, (Object)measured.map(Object::toString).orElse(""));
                measured.ifPresent(m -> this.reportThreadSaturation(now, (double)m, threadName, reporter));
                saturation = PersistentQuerySaturationMetrics.compareSaturation(saturation, measured) > 0 ? saturation : measured;
            }
            saturation.ifPresent(s -> this.reportQuerySaturation(now, (double)s, reporter));
            for (String threadName : Sets.difference(new HashSet<String>(this.perThreadSaturation.keySet()), byThread.keySet())) {
                this.perThreadSaturation.remove(threadName);
                reporter.cleanup(PersistentQuerySaturationMetrics.QUERY_THREAD_SATURATION, (Map<String, String>)ImmutableMap.of((Object)PersistentQuerySaturationMetrics.THREAD_ID, (Object)threadName));
            }
            return saturation;
        }

        private void cleanup(MetricsReporter reporter) {
            for (String threadName : this.perThreadSaturation.keySet()) {
                reporter.cleanup(PersistentQuerySaturationMetrics.QUERY_THREAD_SATURATION, (Map<String, String>)ImmutableMap.of((Object)PersistentQuerySaturationMetrics.THREAD_ID, (Object)threadName));
            }
            for (QueryId queryId : this.queryIds) {
                reporter.cleanup(PersistentQuerySaturationMetrics.QUERY_SATURATION, (Map<String, String>)ImmutableMap.of((Object)PersistentQuerySaturationMetrics.QUERY_ID, (Object)queryId.toString()));
            }
        }
    }

    private static final class BlockedTimeSample {
        private final Instant timestamp;
        private final double totalBlockedTime;

        private BlockedTimeSample(Instant timestamp, double totalBlockedTime) {
            this.timestamp = timestamp;
            this.totalBlockedTime = totalBlockedTime;
        }

        public String toString() {
            return "BlockedTimeSample{timestamp=" + String.valueOf(this.timestamp) + ", totalBlockedTime=" + this.totalBlockedTime + "}";
        }
    }

    private static final class ThreadSaturation {
        private final String threadName;
        private final List<BlockedTimeSample> samples = new LinkedList<BlockedTimeSample>();
        private final Instant startTime;
        private final Duration window;
        private final Duration sampleMargin;

        private ThreadSaturation(String threadName, long startTime, Duration window, Duration sampleMargin) {
            this.threadName = Objects.requireNonNull(threadName, "threadName");
            this.startTime = Instant.ofEpochMilli(startTime);
            this.window = Objects.requireNonNull(window, "window");
            this.sampleMargin = Objects.requireNonNull(sampleMargin, "sampleMargin");
        }

        private boolean inRange(Instant time, Instant start, Instant end) {
            return time.isAfter(start) && time.isBefore(end);
        }

        private Optional<Double> measure(Instant now, BlockedTimeSample current) {
            Instant windowStart = now.minus(this.window);
            Instant earliest = now.minus(this.window.plus(this.sampleMargin));
            Instant latest = now.minus(this.window.minus(this.sampleMargin));
            LOGGER.debug("{}: record and measure with now {},  window {} ({} : {})", (Object)this.threadName, (Object)now, (Object)windowStart, (Object)earliest, (Object)latest);
            this.samples.add(current);
            this.samples.removeIf(s -> s.timestamp.isBefore(earliest));
            if (!this.inRange(this.samples.get((int)0).timestamp, earliest, latest) && !this.startTime.isAfter(windowStart)) {
                return Optional.empty();
            }
            BlockedTimeSample startSample = this.samples.get(0);
            LOGGER.debug("{}: start sample {}", (Object)this.threadName, (Object)startSample);
            double blocked = Math.max(current.totalBlockedTime - startSample.totalBlockedTime, 0.0);
            Instant observedStart = this.samples.get((int)0).timestamp;
            if (this.startTime.isAfter(windowStart)) {
                LOGGER.debug("{}: start time {} is after window start", (Object)this.threadName, (Object)this.startTime);
                blocked += (double)Duration.between(windowStart, this.startTime).toNanos();
                observedStart = windowStart;
            }
            Duration duration = Duration.between(observedStart, current.timestamp);
            double durationNs = duration.toNanos();
            return Optional.of((durationNs - Math.min(blocked, durationNs)) / durationNs);
        }
    }
}

