/*
 * Decompiled with CFR 0.152.
 */
package kafka.restore.schedulers;

import java.util.concurrent.ArrayBlockingQueue;
import kafka.restore.messages.MessageRequest;
import kafka.restore.messages.MessageStatusCode;
import kafka.restore.schedulers.AsyncServiceSchedulerResultsReceiver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractAsyncServiceScheduler {
    private static final int DEFAULT_REQUEST_QUEUE_CAPACITY = 500000;
    private static final String REQUEST_QUEUE_CONSUMER_THREAD_NAME = "requestQueueConsumerThread";
    private static final Logger log = LoggerFactory.getLogger(AbstractAsyncServiceScheduler.class);
    private final AsyncServiceSchedulerResultsReceiver resultsReceiver;
    private ArrayBlockingQueue<MessageRequest> requestQueue;
    private Thread requestQueueConsumerThread;
    private final int requestQueueCapacity;
    private volatile AsyncServiceSchedulerStatus status;

    public AbstractAsyncServiceScheduler(AsyncServiceSchedulerResultsReceiver resultsReceiver) {
        this(resultsReceiver, 500000);
    }

    public AbstractAsyncServiceScheduler(AsyncServiceSchedulerResultsReceiver resultsReceiver, int requestQueueCapacity) {
        if (requestQueueCapacity < 1) {
            throw new IllegalArgumentException("Request Queue Capacity must be at least one");
        }
        this.resultsReceiver = resultsReceiver;
        this.requestQueueCapacity = requestQueueCapacity;
        this.status = AsyncServiceSchedulerStatus.NOT_STARTED;
    }

    public synchronized boolean startUp() {
        if (this.status != AsyncServiceSchedulerStatus.NOT_STARTED && this.status != AsyncServiceSchedulerStatus.SHUTDOWN) {
            log.error("startUp() can only be called on a service scheduler that has not been started or is shutdown.");
            return false;
        }
        this.requestQueue = new ArrayBlockingQueue(this.requestQueueCapacity);
        this.startRequestQueueConsumerThread();
        return true;
    }

    private void consumeRequestsFromRequestQueue() {
        while (this.status == AsyncServiceSchedulerStatus.RUNNING) {
            MessageRequest request = null;
            try {
                request = this.requestQueue.take();
                log.info(String.format("[%s]: took a request out of requestQueue: %s, messages in queue: %s", request.getTopicPartition(), request.getClass().getSimpleName(), this.requestQueue.size()));
                this.processRequestFromRequestQueue(request);
            }
            catch (InterruptedException e) {
                log.warn("requestConsumerThread received InterruptedException while waiting for or processing request. exit consuming request queue loop.", (Throwable)e);
                break;
            }
            catch (Exception e) {
                log.error(String.format("[%s]: Unexpected exception caught while processing request: " + String.valueOf(request) + ", swallow it", request.getTopicPartition()), (Throwable)e);
            }
        }
        if (this.status != AsyncServiceSchedulerStatus.PAUSING && this.status != AsyncServiceSchedulerStatus.SHUTTING_DOWN) {
            log.error("Async Service Scheduler has entered illegal state " + this.status.name() + " while exiting from request queue consumer thread.");
            this.status = AsyncServiceSchedulerStatus.ERROR;
        }
        this.status = this.status == AsyncServiceSchedulerStatus.PAUSING ? AsyncServiceSchedulerStatus.PAUSED : AsyncServiceSchedulerStatus.SHUTDOWN;
        log.info("Request Consumer Thread exiting upon entry to " + this.status.name() + " state");
    }

    public MessageStatusCode submitRequest(MessageRequest request) {
        if (this.status != AsyncServiceSchedulerStatus.RUNNING) {
            log.error("submitRequest() can only be called on a service scheduler that is running. A non-running service scheduler was likely shut down via a call to shutdown().");
            return MessageStatusCode.ILLEGAL_STATE_ERROR;
        }
        boolean scheduled = this.requestQueue.offer(request);
        return scheduled ? MessageStatusCode.SCHEDULED : MessageStatusCode.REQUEST_QUEUE_FULL;
    }

    protected abstract void processRequestFromRequestQueue(MessageRequest var1);

    public synchronized boolean shutdown() {
        if (this.status == AsyncServiceSchedulerStatus.PAUSED || this.status == AsyncServiceSchedulerStatus.ERROR) {
            this.status = AsyncServiceSchedulerStatus.SHUTDOWN;
            return true;
        }
        if (this.status != AsyncServiceSchedulerStatus.RUNNING) {
            log.error("shutdown() can only be called on a service scheduler that is running, paused, or in error state.");
            return false;
        }
        this.status = AsyncServiceSchedulerStatus.SHUTTING_DOWN;
        this.requestQueueConsumerThread.interrupt();
        try {
            this.requestQueueConsumerThread.join();
        }
        catch (InterruptedException e) {
            log.error("shutdown() was interrupted prior to background thread terminating", (Throwable)e);
            this.status = AsyncServiceSchedulerStatus.ERROR;
            return false;
        }
        return true;
    }

    public AsyncServiceSchedulerStatus getStatus() {
        return this.status;
    }

    protected AsyncServiceSchedulerResultsReceiver getResultsReceiver() {
        return this.resultsReceiver;
    }

    public synchronized boolean pause() {
        if (this.status != AsyncServiceSchedulerStatus.RUNNING) {
            log.error("pause() can only be called on service scheduler that is running.");
            return false;
        }
        this.status = AsyncServiceSchedulerStatus.PAUSING;
        this.requestQueueConsumerThread.interrupt();
        try {
            this.requestQueueConsumerThread.join();
        }
        catch (InterruptedException e) {
            log.error("Pause was interrupted prior to completing");
            this.status = AsyncServiceSchedulerStatus.ERROR;
            return false;
        }
        return true;
    }

    public synchronized boolean resume() {
        if (this.status != AsyncServiceSchedulerStatus.PAUSED) {
            log.error("resume() can only be called on service scheduler in PAUSED state.");
            return false;
        }
        this.startRequestQueueConsumerThread();
        return true;
    }

    private void startRequestQueueConsumerThread() {
        this.status = AsyncServiceSchedulerStatus.RUNNING;
        this.requestQueueConsumerThread = new Thread(this::consumeRequestsFromRequestQueue, REQUEST_QUEUE_CONSUMER_THREAD_NAME);
        this.requestQueueConsumerThread.start();
    }

    public static enum AsyncServiceSchedulerStatus {
        NOT_STARTED,
        RUNNING,
        PAUSING,
        PAUSED,
        SHUTTING_DOWN,
        SHUTDOWN,
        ERROR;

    }
}

