/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.backupObjectLifecycle;

import io.confluent.kafka.storage.tier.domain.AbstractTierMetadata;
import io.confluent.kafka.storage.tier.domain.AbstractTierSegmentMetadata;
import io.confluent.kafka.storage.tier.domain.TierRecordType;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.function.Function;
import kafka.tier.backupObjectLifecycle.ObjectStoreUtils;
import kafka.tier.backupObjectLifecycle.RetryFramework;
import kafka.tier.backupObjectLifecycle.TierTopicReaderConfig;
import kafka.tier.topic.TierTopicManager;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndTimestamp;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TierTopicReader {
    private static final Logger log = LoggerFactory.getLogger(TierTopicReader.class);
    private final TierTopicReaderConfig config;
    private Consumer<byte[], byte[]> consumer = null;
    private Map<TopicPartition, Long> endOffsets;
    private long[] currentPositions;

    public TierTopicReader(TierTopicReaderConfig config) throws InterruptedException {
        this.config = config;
        this.consumer = new KafkaConsumer(TierTopicReader.getProperties(config));
        Set<TopicPartition> tierTopicPartitions = TierTopicManager.partitions("_confluent-tier-state", config.tierMetadataNumPartitions);
        this.consumer.assign(tierTopicPartitions);
        this.seekConsumer();
        Function<Set, Map> fn = partitions -> this.consumer.endOffsets((Collection)partitions);
        this.endOffsets = new RetryFramework<Set<TopicPartition>, Map>(config.maxRetries).call(fn, tierTopicPartitions);
        this.currentPositions = new long[config.tierMetadataNumPartitions];
        this.recordConsumerPositionsAndLag();
    }

    private void canTierTopicReaderRunElseThrow() throws InterruptedException {
        if (!this.config.canCLMRun.get().booleanValue()) {
            throw new RuntimeException("TierTopicReader will stop because CLM needs to stop");
        }
        if (Thread.interrupted() || this.config.isShutdownInitiated.get().booleanValue()) {
            throw new InterruptedException("TierTopicReader will stop because current thread is interrupted");
        }
    }

    public List<ObjectStoreUtils.DeletionRecord> deletedSegments() throws InterruptedException {
        ArrayList<ObjectStoreUtils.DeletionRecord> deletedSegments = new ArrayList<ObjectStoreUtils.DeletionRecord>();
        try {
            while (this.hasMoreRecordsToConsume()) {
                this.canTierTopicReaderRunElseThrow();
                ConsumerRecords records = this.consumer.poll(Duration.ofMillis(this.config.tierMetadataMaxPollMs));
                deletedSegments.addAll(TierTopicReader.filterDeleteRecords((ConsumerRecords<byte[], byte[]>)records));
                if ((long)deletedSegments.size() < this.config.maxDeleteCompleteRecordsToConsumeInOneIteration) continue;
                break;
            }
            log.debug("TierTopicReader has consumed " + deletedSegments.size() + " SEGMENT_DELETE_COMPLETE records");
        }
        finally {
            if (this.consumer != null) {
                this.recordConsumerPositionsAndLag();
            }
        }
        return deletedSegments;
    }

    public long[] currentPositions() {
        return this.currentPositions;
    }

    private void seekConsumer() throws InterruptedException {
        if (this.config.tierOffsets.size() == this.config.tierMetadataNumPartitions) {
            log.debug("Consumer to seek to the offsets supplied by LifecycleManagerState");
            for (int i = 0; i < this.config.tierOffsets.size(); ++i) {
                TopicPartition tp = new TopicPartition("_confluent-tier-state", i);
                this.consumer.seek(tp, this.config.tierOffsets.get(i).longValue());
            }
            return;
        }
        HashMap<TopicPartition, Long> timesToSearch = new HashMap<TopicPartition, Long>();
        Long lookBackTimeInMs = this.config.time.milliseconds() - (long)this.config.maxLookBackInDays.get().intValue() * 86400000L;
        log.warn("LifecycleManagerState not available. Try to seek consumer to " + String.valueOf(new Date(lookBackTimeInMs)));
        for (int i = 0; i < this.config.tierMetadataNumPartitions; ++i) {
            timesToSearch.put(new TopicPartition("_confluent-tier-state", i), lookBackTimeInMs);
        }
        Function<Map, Map> fn = partitionToTimes -> this.consumer.offsetsForTimes(partitionToTimes);
        Map offsets = new RetryFramework<HashMap<TopicPartition, Long>, Map>(this.config.maxRetries).call(fn, timesToSearch);
        for (int i = 0; i < this.config.tierMetadataNumPartitions; ++i) {
            TopicPartition tp = new TopicPartition("_confluent-tier-state", i);
            if (offsets.get(tp) != null) {
                log.warn("Consumer to seek to " + String.valueOf(offsets.get(tp)) + " for " + String.valueOf(tp));
                this.consumer.seek(tp, ((OffsetAndTimestamp)offsets.get(tp)).offset());
                continue;
            }
            log.warn("Consumer to seek to the beginning for " + String.valueOf(tp));
            this.consumer.seekToBeginning(Collections.singletonList(tp));
        }
    }

    private static List<ObjectStoreUtils.DeletionRecord> filterDeleteRecords(ConsumerRecords<byte[], byte[]> records) {
        int numMessages = 0;
        ArrayList<ObjectStoreUtils.DeletionRecord> deletedSegments = new ArrayList<ObjectStoreUtils.DeletionRecord>();
        for (ConsumerRecord record : records) {
            ++numMessages;
            Optional<AbstractTierMetadata> metadataEntryOpt = AbstractTierMetadata.deserialize((byte[])record.key(), (byte[])record.value(), record.timestamp());
            if (metadataEntryOpt.isPresent()) {
                AbstractTierMetadata tierMetadata = metadataEntryOpt.get();
                if (tierMetadata.type() != TierRecordType.SegmentDeleteComplete) continue;
                AbstractTierSegmentMetadata deleteEvent = (AbstractTierSegmentMetadata)tierMetadata;
                deletedSegments.add(new ObjectStoreUtils.DeletionRecord(deleteEvent.objectId(), deleteEvent.topicIdPartition(), deleteEvent.timestamp()));
                continue;
            }
            throw new RuntimeException("Could not deserialize tier metadata record from partition: " + record.partition() + " at offset: " + record.offset());
        }
        log.debug("Processed " + numMessages + " total records and filtered " + deletedSegments.size() + " records of type SegmentDeleteComplete");
        return deletedSegments;
    }

    public boolean hasMoreRecordsToConsume() throws InterruptedException {
        for (Map.Entry<TopicPartition, Long> entry : this.endOffsets.entrySet()) {
            TopicPartition tierPartition = entry.getKey();
            Long endOffset = entry.getValue();
            Function<TopicPartition, Long> fn = topicPartition -> this.consumer.position(topicPartition);
            long position = new RetryFramework<TopicPartition, Long>(this.config.maxRetries).call(fn, tierPartition);
            if (position >= endOffset) continue;
            log.debug("Consumer position for " + String.valueOf(tierPartition) + " is at " + position + " and end offset is " + endOffset);
            return true;
        }
        log.debug("Consumer has consumed till the end of all the tier state partitions");
        return false;
    }

    public void maybeCloseConsumer() {
        if (this.consumer != null) {
            this.consumer.close();
        }
    }

    private void recordConsumerPositionsAndLag() throws InterruptedException {
        long totalLag = 0L;
        for (int i = 0; i < this.config.tierMetadataNumPartitions; ++i) {
            TopicPartition tp = new TopicPartition("_confluent-tier-state", i);
            Function<TopicPartition, Long> fn = topicPartition -> this.consumer.position(topicPartition);
            this.currentPositions[i] = new RetryFramework<TopicPartition, Long>(this.config.maxRetries).call(fn, tp);
            if (!this.endOffsets.containsKey(tp)) continue;
            totalLag += Math.max(0L, this.endOffsets.get(tp) - this.currentPositions[i]);
        }
        log.info("Total lag (in offsets) for LifecycleManager's consumer " + totalLag);
        this.config.recordLag.accept(totalLag);
    }

    private static String clientId(String clusterId, int brokerId) {
        String separator = "-";
        return "consumer" + separator + clusterId + separator + brokerId + separator + "CustomLifecycleManager";
    }

    private static String groupId(String clusterId) {
        String separator = "-";
        return clusterId + separator + "CustomLifecycleManager";
    }

    public static Properties getProperties(TierTopicReaderConfig config) {
        Properties properties = new Properties();
        properties.putAll(config.interBrokerClientConfigs.get());
        properties.put("client.id", TierTopicReader.clientId(config.clusterId, config.brokerId));
        properties.put("group.id", TierTopicReader.groupId(config.clusterId));
        properties.put("enable.auto.commit", "false");
        properties.put("key.deserializer", ByteArrayDeserializer.class.getName());
        properties.put("value.deserializer", ByteArrayDeserializer.class.getName());
        properties.put("enable.metrics.push", "false");
        properties.remove("metric.reporters");
        return properties;
    }
}

