/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.fetcher.objectcache;

import io.confluent.kafka.storage.tier.store.objects.FragmentType;
import io.confluent.kafka.storage.tier.store.objects.metadata.ObjectMetadata;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufInputStream;
import io.netty.buffer.CompositeByteBuf;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.util.ReferenceCounted;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import kafka.tier.exceptions.TierObjectStoreRetriableException;
import kafka.tier.fetcher.TierFetcherMetrics;
import kafka.tier.fetcher.objectcache.ObjectCacheConfig;
import kafka.tier.store.TierObjectStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PrefetchCache {
    private static final Logger log = LoggerFactory.getLogger(PrefetchCache.class);
    private static final ByteBufAllocator ALLOCATOR = new PooledByteBufAllocator(true);
    public final LinkedHashMap<CacheKey, CacheEntry> cacheMap;
    private final ReadWriteLock rwLock;
    public volatile long cacheMaxSizeBytes;
    private final long entrySizeBytes;
    private final long prefetchRangeBytes;
    private final TierObjectStore objectStore;
    public volatile Optional<TierFetcherMetrics> tierFetcherMetrics;

    public PrefetchCache(ObjectCacheConfig config, TierObjectStore objectStore, Optional<TierFetcherMetrics> tierFetcherMetrics) {
        this.cacheMaxSizeBytes = config.prefetchCacheMaxSizeBytes;
        this.entrySizeBytes = config.prefetchCacheEntrySizeBytes;
        this.prefetchRangeBytes = config.prefetchCacheRangeBytes;
        this.cacheMap = new LinkedHashMap(20, 0.75f, false);
        this.rwLock = new ReentrantReadWriteLock();
        this.objectStore = objectStore;
        this.tierFetcherMetrics = tierFetcherMetrics;
    }

    public CompletableFuture<CacheEntryInputStream> get(ObjectMetadata objectMetadata, long startBytePosition, long endBytePositionExclusive, int segmentSize) {
        log.debug("Fetching data by prefetch cache: {}, start: {}, end: {}, segmentSize: {}", new Object[]{objectMetadata, startBytePosition, endBytePositionExclusive, segmentSize});
        long alignedStart = this.alignToEntrySize(startBytePosition);
        long maxBytePositionExclusive = Math.min(endBytePositionExclusive, (long)segmentSize);
        ArrayList<CacheEntry> entries = new ArrayList<CacheEntry>();
        log.debug("Fetching requested range by prefetch cache: {}, aligned start: {}, end position: {}", new Object[]{objectMetadata, alignedStart, maxBytePositionExclusive});
        for (long pos = alignedStart; pos < maxBytePositionExclusive; pos += this.entrySizeBytes) {
            CacheKey key = new CacheKey(objectMetadata.objectId(), pos);
            CacheEntry entry2 = this.getOrCreateCacheEntry(key, Math.min(pos + this.entrySizeBytes, (long)segmentSize), objectMetadata, true);
            entries.add(entry2);
        }
        long maxPrefetchEndPositionExclusive = Math.min(alignedStart + this.prefetchRangeBytes, (long)segmentSize);
        if (maxPrefetchEndPositionExclusive > maxBytePositionExclusive) {
            long prefetchAlignedStart = this.alignToEntrySize(maxBytePositionExclusive);
            log.debug("Prefetching data by prefetch cache: {}, range: {} - {}", new Object[]{objectMetadata, prefetchAlignedStart, maxPrefetchEndPositionExclusive});
            try {
                for (long pos = prefetchAlignedStart; pos < maxPrefetchEndPositionExclusive; pos += this.entrySizeBytes) {
                    CacheKey key = new CacheKey(objectMetadata.objectId(), pos);
                    CacheEntry entry3 = this.getOrCreateCacheEntry(key, Math.min(pos + this.entrySizeBytes, (long)segmentSize), objectMetadata, false);
                    entry3.release();
                }
            }
            catch (Exception e2) {
                log.error("Failed to prefetch data: {}, range: {} - {}", new Object[]{objectMetadata, prefetchAlignedStart, maxPrefetchEndPositionExclusive, e2});
            }
        }
        return ((CompletableFuture)CompletableFuture.allOf((CompletableFuture[])entries.stream().map(entry -> entry.future).toArray(CompletableFuture[]::new)).thenApply(v -> new CacheEntryInputStream(entries, startBytePosition - alignedStart, maxBytePositionExclusive - alignedStart))).exceptionally(e -> {
            entries.forEach(CacheEntry::release);
            e = e.getCause() != null ? e.getCause() : e;
            throw e instanceof RuntimeException ? (RuntimeException)e : new RuntimeException((Throwable)e);
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CacheEntry getOrCreateCacheEntry(CacheKey key, long endBytePositionExclusive, ObjectMetadata objectMetadata, boolean updateAccessStats) {
        this.rwLock.readLock().lock();
        CacheEntry entry = this.cacheMap.get(key);
        if (entry != null) {
            entry.retain();
        }
        this.rwLock.readLock().unlock();
        if (updateAccessStats) {
            this.tierFetcherMetrics.ifPresent(metrics -> metrics.objectCacheAccesses().record());
        }
        if (entry == null || entry.future.isCompletedExceptionally()) {
            this.rwLock.writeLock().lock();
            try {
                entry = this.cacheMap.get(key);
                if (entry == null || entry.future.isCompletedExceptionally()) {
                    entry = new CacheEntry(this.fetchDataAsync(objectMetadata, key.bytePosition, endBytePositionExclusive));
                    this.cacheMap.remove(key);
                    this.cacheMap.put(key, entry);
                } else if (updateAccessStats) {
                    this.tierFetcherMetrics.ifPresent(metrics -> metrics.objectCacheHits().record());
                }
                entry.retain();
                this.evictEntries();
            }
            finally {
                this.rwLock.writeLock().unlock();
            }
        } else if (updateAccessStats) {
            this.tierFetcherMetrics.ifPresent(metrics -> metrics.objectCacheHits().record());
        }
        return entry;
    }

    private CompletableFuture<ByteBuf> fetchDataAsync(ObjectMetadata objectMetadata, long startPosition, long endPositionExclusive) {
        return this.objectStore.getObjectStoreFragmentAsync(objectMetadata, FragmentType.SEGMENT, startPosition, endPositionExclusive).thenApply(response -> {
            ReferenceCounted buf = null;
            try {
                int expectedLen = (int)(endPositionExclusive - startPosition);
                buf = ALLOCATOR.buffer((int)this.entrySizeBytes);
                int actualLen = ((ByteBuf)buf).writeBytes(response.getInputStream(), expectedLen);
                if (actualLen != expectedLen) {
                    throw new IllegalStateException(String.format("Failed to read %d bytes from TierObjectStoreResponse, actual read bytes: %d", expectedLen, actualLen));
                }
                this.tierFetcherMetrics.ifPresent(metrics -> metrics.bytesPrefetched().record((double)expectedLen));
                ReferenceCounted referenceCounted = buf;
                return referenceCounted;
            }
            catch (Exception e) {
                if (buf != null) {
                    buf.release();
                }
                log.error("Failed to read data from TierObjectStoreResponse", (Throwable)e);
                throw new TierObjectStoreRetriableException("Failed to read data from TierObjectStoreResponse", e);
            }
            finally {
                try {
                    response.close();
                }
                catch (Exception e) {
                    log.warn("failed to close TierObjectStoreResponse", (Throwable)e);
                }
            }
        });
    }

    private long alignToEntrySize(long startBytePosition) {
        return startBytePosition / this.entrySizeBytes * this.entrySizeBytes;
    }

    private void evictEntries() {
        while (this.cacheTotalSize() > this.cacheMaxSizeBytes) {
            CacheKey key = this.cacheMap.keySet().iterator().next();
            CacheEntry entry = (CacheEntry)this.cacheMap.remove(key);
            entry.release();
        }
        this.tierFetcherMetrics.ifPresent(metrics -> {
            metrics.objectCacheTotalSizeBytes = this.cacheTotalSize();
        });
    }

    private long cacheTotalSize() {
        return this.entrySizeBytes * (long)this.cacheMap.size();
    }

    public void close() {
        this.cacheMaxSizeBytes = 0L;
        this.rwLock.writeLock().lock();
        try {
            this.evictEntries();
        }
        finally {
            this.rwLock.writeLock().unlock();
            this.tierFetcherMetrics = Optional.empty();
        }
    }

    public static class CacheKey {
        public final UUID objectId;
        public final long bytePosition;

        public CacheKey(UUID objectId, long bytePosition) {
            this.objectId = objectId;
            this.bytePosition = bytePosition;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CacheKey that = (CacheKey)o;
            return this.bytePosition == that.bytePosition && Objects.equals(this.objectId, that.objectId);
        }

        public int hashCode() {
            return Objects.hash(this.objectId, this.bytePosition);
        }

        public String toString() {
            return "CacheKey(objectId=" + String.valueOf(this.objectId) + ", bytePosition=" + this.bytePosition + ")";
        }
    }

    public static class CacheEntry {
        public final CompletableFuture<ByteBuf> future;
        public final AtomicInteger refCnt = new AtomicInteger(1);

        public CacheEntry(CompletableFuture<ByteBuf> future) {
            this.future = future;
        }

        public void release() {
            if (this.refCnt.decrementAndGet() <= 0) {
                this.future.thenApply(ReferenceCounted::release);
            }
        }

        public void retain() {
            this.refCnt.incrementAndGet();
        }
    }

    public static class CacheEntryInputStream
    extends ByteBufInputStream {
        private final List<CacheEntry> entries;

        public CacheEntryInputStream(List<CacheEntry> entries, long relativeStartBytePosition, long relativeEndBytePositionExclusive) {
            super((ByteBuf)CacheEntryInputStream.createCompositeBuf(entries, relativeStartBytePosition, relativeEndBytePositionExclusive), true);
            this.entries = entries;
        }

        private static CompositeByteBuf createCompositeBuf(List<CacheEntry> entries, long relativeStartBytePosition, long relativeEndBytePositionExclusive) {
            List<ByteBuf> buffers = entries.stream().map(entry -> entry.future.join()).collect(Collectors.toList());
            CompositeByteBuf compositeBuf = ALLOCATOR.compositeBuffer(buffers.size());
            try {
                buffers.forEach(buffer -> compositeBuf.addComponent(true, buffer.retainedDuplicate()));
                compositeBuf.readerIndex((int)relativeStartBytePosition);
                compositeBuf.writerIndex((int)Math.min((long)compositeBuf.writerIndex(), relativeEndBytePositionExclusive));
                return compositeBuf;
            }
            catch (Exception e) {
                log.error("Failed to create InputStream from buffers", (Throwable)e);
                compositeBuf.release();
                throw e;
            }
        }

        @Override
        public void close() throws IOException {
            try {
                super.close();
            }
            finally {
                this.entries.forEach(CacheEntry::release);
            }
        }
    }
}

