/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.tools;

import io.confluent.kafka.storage.tier.store.DataTypePathPrefix;
import io.confluent.kafka.storage.tier.store.objects.FragmentType;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.TierObjectStoreFunctionUtils;
import kafka.tier.store.TierObjectStoreResponse;
import kafka.tier.store.TierObjectStoreUtils;
import kafka.tier.store.VersionInformation;
import kafka.tier.store.objects.metadata.TierTopicHeadDataLossReportMetadata;
import kafka.tier.tools.RecoveryUtils;
import kafka.tier.tools.TierMetadataRecoveryOrchestrator;
import kafka.tier.tools.TierMetadataRecoveryUtils;
import kafka.tier.tools.TierToolsUtils;
import kafka.tier.tools.commands.TierTopicHeadDataLossDetectionCommandResponse;
import kafka.tier.topic.TierTopicHeadDataLossReportReconciler;
import kafka.tier.topic.recovery.ReconciledTierTopicHeadDataLossReport;
import kafka.tier.topic.recovery.TierTopicHeadDataLossReport;
import net.sourceforge.argparse4j.impl.Arguments;
import net.sourceforge.argparse4j.inf.MutuallyExclusiveGroup;
import net.sourceforge.argparse4j.inf.Namespace;
import net.sourceforge.argparse4j.inf.Subparser;
import net.sourceforge.argparse4j.inf.Subparsers;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TierTopicHeadDataLossDetectionCommand {
    private static final Logger LOGGER = LoggerFactory.getLogger(TierTopicHeadDataLossDetectionCommand.class);
    static final String COMMAND = "tier-topic-head-data-loss-detection-command";
    private static final String IDENTIFIER = "identifier";
    private static final String IDENTIFIER_DOC = "Unique identifier representing the on-demand data loss validation run.";
    private static final String TTP_ALLOW_LIST_FILE = "ttp-allow-list-file";
    private static final String TTP_ALLOW_LIST_FILE_DOC = "Name of the file that contains the list of tier topic partitions we want to validate. If not provided, defaults to running on all tier topic partitions.";
    private static final String SKIP_RECONCILE = "skip-reconcile";
    private static final String SKIP_RECONCILE_DOC = "If provided, only runs the data loss validator and skips the reconcile operation. This is useful when you want to run the data loss validator on-demand and then run the reconcile operation later. By default / if not provided, we run the reconcile operation after the data loss validator.";
    private static final String RECONCILE_ONLY = "reconcile-only";
    private static final String RECONCILE_ONLY_DOC = "If provided, downloads the previous data loss validation reports from the object store under the prefix <identifier>, runs the reconcile operation and does not run the data loss validator. This will also write the reconciled data loss report to a local working directory named <identifier>. This flag is useful when you want to run the reconcile operation on-demand. By default / if not provided, we run the data loss validator and then run the reconcile operation.";

    static void addCommand(Subparsers subparsers) {
        Subparser tierTopicHeadDataLossDetection = subparsers.addParser(COMMAND).help("Runs on-demand data loss validator on provided tier topic partitions.");
        MutuallyExclusiveGroup runScopeGroup = tierTopicHeadDataLossDetection.addMutuallyExclusiveGroup().required(false);
        runScopeGroup.addArgument(RecoveryUtils.makeArgument("broker-ids")).dest("broker-ids").action(Arguments.store()).type(String.class).help("Runs the operation only on the specified brokers supplied as a comma-separated list. (i.e., 0,1,2,3)");
        runScopeGroup.addArgument(RecoveryUtils.makeArgument("all-brokers")).dest("all-brokers").action(Arguments.storeTrue()).help("Runs the operation for all brokers in the cluster");
        tierTopicHeadDataLossDetection.addArgument(RecoveryUtils.makeArgument("tier.config")).dest("tier.config").action(Arguments.store()).type(Arguments.fileType().verifyCanRead()).required(true).help("The path to a configuration file containing the required properties");
        tierTopicHeadDataLossDetection.addArgument(RecoveryUtils.makeArgument("rest-server-port-override")).dest("rest-server-port-override").type(Integer.class).action(Arguments.store()).setDefault(TierToolsUtils.DEFAULT_REST_SERVER_PORT_OVERRIDE).help(TierToolsUtils.REST_SERVER_PORT_OVERRIDE_DOC);
        tierTopicHeadDataLossDetection.addArgument(RecoveryUtils.makeArgument(IDENTIFIER)).dest(IDENTIFIER).type(String.class).action(Arguments.store()).required(true).help(IDENTIFIER_DOC);
        tierTopicHeadDataLossDetection.addArgument(RecoveryUtils.makeArgument(TTP_ALLOW_LIST_FILE)).dest(TTP_ALLOW_LIST_FILE).type(Arguments.fileType().verifyCanRead()).action(Arguments.store()).required(false).help(TTP_ALLOW_LIST_FILE_DOC);
        MutuallyExclusiveGroup reconcileRunGroup = tierTopicHeadDataLossDetection.addMutuallyExclusiveGroup().required(false);
        reconcileRunGroup.addArgument(RecoveryUtils.makeArgument(SKIP_RECONCILE)).dest(SKIP_RECONCILE).action(Arguments.storeTrue()).setDefault((Object)false).help(SKIP_RECONCILE_DOC);
        reconcileRunGroup.addArgument(RecoveryUtils.makeArgument(RECONCILE_ONLY)).dest(RECONCILE_ONLY).action(Arguments.storeTrue()).setDefault((Object)false).help(RECONCILE_ONLY_DOC);
    }

    static Set<TopicPartition> parseTierTopicPartitionAllowList(String tierTopicPartitionAllowListFile) throws IOException {
        HashSet<TopicPartition> topicPartitions = new HashSet<TopicPartition>();
        try (BufferedReader reader = new BufferedReader(new FileReader(tierTopicPartitionAllowListFile));){
            String line;
            while ((line = reader.readLine()) != null) {
                TopicPartition topicPartition = TopicPartition.fromString((String)line.trim());
                if (!topicPartition.topic().equals("_confluent-tier-state")) {
                    throw new IllegalArgumentException(String.format("Expected tier topic: %s, Received topic: %s", "_confluent-tier-state", topicPartition.topic()));
                }
                topicPartitions.add(topicPartition);
            }
        }
        return topicPartitions;
    }

    private static String getPerBrokerDataLossReportFileName(String dirName, int broker) {
        return String.format("%s/%d-data-loss-report.json", dirName, broker);
    }

    static void printDataLossValidatorSummary(TierTopicHeadDataLossDetectionCommandResponse response, String identifier, boolean printLocalFile) {
        StringBuilder formattedDataLossValidatorResponse = new StringBuilder("Data loss detection succeeded completely for the following brokers:\n");
        int successIndex = 1;
        for (TierTopicHeadDataLossDetectionCommandResponse.SuccessBrokerDetail broker : response.getSuccess()) {
            if (broker.dataLossReportPath() == null) {
                formattedDataLossValidatorResponse.append(String.format("%d. Broker %d, Data loss report not created.\n", successIndex++, broker.broker()));
                continue;
            }
            if (printLocalFile) {
                formattedDataLossValidatorResponse.append(String.format("%d. Broker %d, Data loss report URI: %s, Data loss report local file: %s\n", successIndex++, broker.broker(), broker.dataLossReportPath(), TierTopicHeadDataLossDetectionCommand.getPerBrokerDataLossReportFileName(identifier, broker.broker())));
                continue;
            }
            formattedDataLossValidatorResponse.append(String.format("%d. Broker %d, Data loss report URI: %s\n", successIndex++, broker.broker(), broker.dataLossReportPath()));
        }
        int partialSuccessIndex = 1;
        int failedIndex = 1;
        boolean hasPartialSuccesses = false;
        boolean hasFailures = false;
        StringBuilder partialSuccessResponse = new StringBuilder("\nData loss detection succeeded partially for the following brokers:\n");
        StringBuilder failedBrokersResponse = new StringBuilder("\nData loss detection failed completely for the following brokers:\n");
        for (TierTopicHeadDataLossDetectionCommandResponse.FailedBrokerDetail broker : response.getFailed()) {
            StringBuilder errorMessages = new StringBuilder();
            for (String errorMessage : broker.errorMessages()) {
                errorMessages.append(String.format("\t- %s\n", errorMessage));
            }
            if (broker.dataLossReportPath() == null) {
                hasFailures = true;
                failedBrokersResponse.append(String.format("%d. Broker %d, Error messages:\n%s", failedIndex++, broker.broker(), errorMessages));
                continue;
            }
            hasPartialSuccesses = true;
            if (printLocalFile) {
                partialSuccessResponse.append(String.format("%d. Broker %d, Data loss report URI: %s, Data loss report local file: %s, Error messages:\n%s", partialSuccessIndex++, broker.broker(), broker.dataLossReportPath(), TierTopicHeadDataLossDetectionCommand.getPerBrokerDataLossReportFileName(identifier, broker.broker()), errorMessages));
                continue;
            }
            partialSuccessResponse.append(String.format("%d. Broker %d, Data loss report URI: %s, Error messages:\n%s", partialSuccessIndex++, broker.broker(), broker.dataLossReportPath(), errorMessages));
        }
        if (hasPartialSuccesses) {
            formattedDataLossValidatorResponse.append((CharSequence)partialSuccessResponse);
        }
        if (hasFailures) {
            formattedDataLossValidatorResponse.append((CharSequence)failedBrokersResponse);
        }
        System.out.print(formattedDataLossValidatorResponse);
    }

    private static int runDataLossValidator(Namespace namespace, boolean printLocalFile) throws IOException, ExecutionException, InterruptedException {
        String bootstrapServers = namespace.getString("bootstrap-servers");
        String adminConfig = namespace.getString("admin.config");
        Integer restServerPort = namespace.getInt("rest-server-port-override");
        TierMetadataRecoveryOrchestrator orchestrator = TierMetadataRecoveryUtils.getTierMetadataRecoveryOrchestrator(adminConfig, bootstrapServers, restServerPort);
        String identifier = namespace.getString(IDENTIFIER);
        String tierTopicPartitionAllowListFile = namespace.getString(TTP_ALLOW_LIST_FILE);
        HashSet<TopicPartition> tierTopicPartitionsAllowList = new HashSet();
        if (tierTopicPartitionAllowListFile != null) {
            tierTopicPartitionsAllowList = TierTopicHeadDataLossDetectionCommand.parseTierTopicPartitionAllowList(tierTopicPartitionAllowListFile);
        }
        TierTopicHeadDataLossDetectionCommandResponse response = new TierTopicHeadDataLossDetectionCommandResponse();
        if (namespace.getBoolean("all-brokers").booleanValue()) {
            response = orchestrator.detectDataLossInTierTopicForCluster(identifier, tierTopicPartitionsAllowList);
        } else if (namespace.getString("broker-ids") != null) {
            Set<Integer> brokers = TierMetadataRecoveryUtils.getBrokerList(namespace);
            for (Integer broker : brokers) {
                TierTopicHeadDataLossDetectionCommandResponse brokerRes = orchestrator.detectDataLossInTierTopicForBroker(broker, identifier, tierTopicPartitionsAllowList);
                response.getSuccess().addAll(brokerRes.getSuccess());
                response.getFailed().addAll(brokerRes.getFailed());
            }
        } else {
            throw new IllegalArgumentException(String.format("Either --%s OR --%s must be provided when running data loss validator.", "all-brokers", "broker-ids"));
        }
        TierTopicHeadDataLossDetectionCommand.printDataLossValidatorSummary(response, identifier, printLocalFile);
        if (response.getFailed().isEmpty()) {
            return 0;
        }
        return 1;
    }

    private static TierObjectStore createTierObjectStore(String tierConfigFile) {
        TierObjectStore objectStore;
        try {
            objectStore = TierObjectStoreUtils.objectStore(tierConfigFile);
        }
        catch (IOException e) {
            LOGGER.error("Failed to create object store instance with config: {}", (Object)tierConfigFile);
            throw new UncheckedIOException(e);
        }
        return objectStore;
    }

    static Map.Entry<List<TierTopicHeadDataLossReport>, Integer> listAndDownloadDataLossReports(String identifier, String tierConfigFile) throws InterruptedException, IOException {
        ArrayList<TierTopicHeadDataLossReport> reports = new ArrayList<TierTopicHeadDataLossReport>();
        int exitCode = 0;
        TierObjectStore objectStore = TierTopicHeadDataLossDetectionCommand.createTierObjectStore(tierConfigFile);
        String metadataPrefix = DataTypePathPrefix.TIER_RECOVERY_DATA_UPLOAD.prefix() + "/recovery-" + identifier + "/";
        Map<String, List<VersionInformation>> objects = TierObjectStoreFunctionUtils.listObject(() -> false, objectStore, metadataPrefix, true);
        Path outputDir = Paths.get(identifier, new String[0]);
        if (Files.exists(outputDir, new LinkOption[0])) {
            try {
                FileUtils.deleteDirectory(outputDir.toFile());
            }
            catch (IOException e) {
                LOGGER.error("Failed to clean up stale directory: {}", (Object)outputDir);
                throw e;
            }
        }
        try {
            Files.createDirectories(outputDir, new FileAttribute[0]);
        }
        catch (IOException e) {
            LOGGER.error("Failed to create directory: {}", (Object)identifier);
            throw e;
        }
        for (Map.Entry<String, List<VersionInformation>> entry : objects.entrySet()) {
            String uri = entry.getKey();
            if (!uri.contains("tier-topic-head-data-loss-report")) continue;
            LOGGER.info("Found data loss report at uri: {}", (Object)uri);
            try {
                TierTopicHeadDataLossReportMetadata metadata = TierTopicHeadDataLossReportMetadata.fromPath(uri);
                String reportJson = TierTopicHeadDataLossDetectionCommand.readDataLossReport(metadata, objectStore);
                reports.add(TierTopicHeadDataLossReport.readJsonFromString(reportJson));
                Path reportPath = Paths.get(String.format("%s/%d-data-loss-report.json", identifier, metadata.broker()), new String[0]);
                Files.write(reportPath, reportJson.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
            }
            catch (IOException | InterruptedException e) {
                LOGGER.error("Failed to read data loss report from uri: {}", (Object)uri);
                exitCode = 1;
            }
        }
        return new AbstractMap.SimpleEntry<List<TierTopicHeadDataLossReport>, Integer>(reports, exitCode);
    }

    private static String readDataLossReport(TierTopicHeadDataLossReportMetadata metadata, TierObjectStore objectStore) throws IOException, InterruptedException {
        TierObjectStoreResponse response = TierObjectStoreFunctionUtils.getObjectStoreFragment(() -> false, objectStore, metadata, FragmentType.TIER_TOPIC_HEAD_DATA_LOSS_REPORT);
        return IOUtils.toString(response.getInputStream(), StandardCharsets.UTF_8);
    }

    private static void reconcileDataLossValidatorReports(List<TierTopicHeadDataLossReport> reports, String identifier) throws IOException {
        ReconciledTierTopicHeadDataLossReport reconciledReport = ReconciledTierTopicHeadDataLossReport.createEmptyReport();
        for (TierTopicHeadDataLossReport report : reports) {
            reconciledReport = TierTopicHeadDataLossReportReconciler.reconcileReportIncrementally(report, reconciledReport);
        }
        Path reportPath = Paths.get(String.format("%s/reconciled-data-loss-report.json", identifier), new String[0]);
        ReconciledTierTopicHeadDataLossReport.writeJsonToFile(reconciledReport, Files.newOutputStream(reportPath, new OpenOption[0]));
        LOGGER.info("Reconciled data loss report is written locally to: {}", (Object)reportPath);
    }

    static int execute(Namespace namespace) throws IOException, ExecutionException, InterruptedException {
        boolean runReconciler;
        int exitCode = 0;
        boolean skipReconcile = namespace.getBoolean(SKIP_RECONCILE);
        boolean reconcileOnly = namespace.getBoolean(RECONCILE_ONLY);
        String identifier = namespace.getString(IDENTIFIER);
        String tierConfigFile = namespace.getString("tier.config");
        boolean runDlv = !reconcileOnly;
        boolean bl = runReconciler = !skipReconcile || reconcileOnly;
        if (reconcileOnly) {
            LOGGER.info("Running reconcile operation only.");
        } else if (skipReconcile) {
            LOGGER.info("Running data loss validator only.");
        }
        if (runDlv) {
            exitCode = TierTopicHeadDataLossDetectionCommand.runDataLossValidator(namespace, runReconciler);
        }
        Map.Entry<List<TierTopicHeadDataLossReport>, Integer> dataLossReports = TierTopicHeadDataLossDetectionCommand.listAndDownloadDataLossReports(identifier, tierConfigFile);
        if (exitCode == 0) {
            exitCode = dataLossReports.getValue();
        }
        if (runReconciler) {
            TierTopicHeadDataLossDetectionCommand.reconcileDataLossValidatorReports(dataLossReports.getKey(), identifier);
        }
        return exitCode;
    }
}

