/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.security.auth.ipFiltering;

import io.confluent.security.auth.ipFiltering.IpFilteringAccessRule;
import io.confluent.security.auth.ipFiltering.IpFilteringUtils;
import io.confluent.security.auth.store.data.AuthEntryType;
import io.confluent.security.auth.store.data.AuthKey;
import io.confluent.security.auth.store.data.AuthValue;
import io.confluent.security.auth.store.data.IpFilteringKey;
import io.confluent.security.auth.store.data.IpFilteringValue;
import io.confluent.security.auth.store.data.StatusValue;
import io.confluent.security.authorizer.AccessRule;
import io.confluent.security.authorizer.Action;
import io.confluent.security.authorizer.Scope;
import io.confluent.security.authorizer.provider.AuthorizeRule;
import io.confluent.security.authorizer.provider.InvalidScopeException;
import io.confluent.security.authorizer.provider.ResourceAuthorizeRules;
import io.confluent.security.operationgroupdefinitions.IpfOperationGroupDefinitions;
import io.confluent.security.operationgroupdefinitions.OperationGroupDefinition;
import io.confluent.security.roledefinitions.AccessPolicy;
import io.confluent.security.roledefinitions.Operation;
import io.confluent.security.roledefinitions.PermissionType;
import io.confluent.security.roledefinitions.ResourceType;
import io.confluent.security.store.KeyValueStore;
import io.confluent.security.store.MetadataStoreStatus;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.apache.kafka.common.errors.CorruptRecordException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IpFilteringAuthCache
implements KeyValueStore<AuthKey, AuthValue> {
    private static final Logger log = LoggerFactory.getLogger(IpFilteringAuthCache.class);
    private static final Map<IpFilteringKey, IpFilteringValue> NO_IP_FILTERING = Collections.emptyMap();
    private static final String IP_FILTERING_DOES_NOT_SUPPORT_MSG = "IpFiltering does not support key entry type: ";
    private final IpfOperationGroupDefinitions ipfOperationGroupDefinitions;
    private final Map<String, Map<String, String>> ipFilteringOperationGroupMappings;
    private final Map<String, Map<IpFilteringKey, IpFilteringValue>> ipFilteringMap;
    private final Map<String, Map<Scope, IpFilteringAccessRule>> ipFilteringAccessRules;
    private final Map<Integer, StatusValue> partitionStatus;

    public IpFilteringAuthCache(IpfOperationGroupDefinitions ipfOperationGroupDefinitions) {
        this.ipfOperationGroupDefinitions = ipfOperationGroupDefinitions;
        this.ipFilteringMap = new ConcurrentHashMap<String, Map<IpFilteringKey, IpFilteringValue>>();
        this.ipFilteringAccessRules = new ConcurrentHashMap<String, Map<Scope, IpFilteringAccessRule>>();
        this.partitionStatus = new ConcurrentHashMap<Integer, StatusValue>();
        this.ipFilteringOperationGroupMappings = new ConcurrentHashMap<String, Map<String, String>>();
        this.convertIpfOperationGroupDefinitionsToMapping();
    }

    public AuthValue get(AuthKey key) {
        AuthEntryType entryType = key.entryType();
        switch (entryType) {
            case IP_FILTERING: {
                IpFilteringKey ipFilteringKey = (IpFilteringKey)key;
                return (AuthValue)this.ipFilteringMap.getOrDefault(ipFilteringKey.operationGroup(), NO_IP_FILTERING).get(ipFilteringKey);
            }
        }
        log.info(IP_FILTERING_DOES_NOT_SUPPORT_MSG + String.valueOf(entryType));
        return null;
    }

    public AuthValue put(AuthKey key, AuthValue value) {
        try {
            if (value == null) {
                throw new IllegalArgumentException("Value must not be null");
            }
            if (key.entryType() != value.entryType()) {
                throw new CorruptRecordException("Invalid IpFiltering record with key entry type =" + String.valueOf(key.entryType()));
            }
            AuthEntryType entryType = key.entryType();
            switch (entryType) {
                case IP_FILTERING: {
                    return this.updateIpFilteringKeyValue(key, value);
                }
            }
            log.info(IP_FILTERING_DOES_NOT_SUPPORT_MSG + String.valueOf(entryType));
            return null;
        }
        catch (Exception e) {
            log.error("Error while putting key: {} and value: {} in IP Filtering Auth Cache", new Object[]{key, value, e});
            throw e;
        }
    }

    public AuthValue remove(AuthKey key) {
        AuthEntryType entryType = key.entryType();
        switch (entryType) {
            case IP_FILTERING: {
                return this.removeIpFilteringKey(key);
            }
        }
        log.info(IP_FILTERING_DOES_NOT_SUPPORT_MSG + String.valueOf(entryType));
        return null;
    }

    public Map<? extends AuthKey, ? extends AuthValue> map(String type) {
        AuthEntryType entryType = AuthEntryType.valueOf((String)type);
        switch (entryType) {
            case IP_FILTERING: {
                Map<IpFilteringKey, IpFilteringValue> ipFilteringKeyValueMap = this.ipFilteringMap.values().stream().flatMap(map -> map.entrySet().stream()).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
                return Collections.unmodifiableMap(ipFilteringKeyValueMap);
            }
        }
        log.info(IP_FILTERING_DOES_NOT_SUPPORT_MSG + String.valueOf(entryType));
        return null;
    }

    public void fail(int partition, String errorMessage) {
        this.partitionStatus.put(partition, new StatusValue(MetadataStoreStatus.FAILED, -1, null, errorMessage));
    }

    public MetadataStoreStatus status(int partition) {
        StatusValue statusValue = this.partitionStatus.get(partition);
        return statusValue != null ? statusValue.status() : MetadataStoreStatus.UNKNOWN;
    }

    public AuthorizeRule findRule(String host, Action action) {
        AuthorizeRule authorizeRule = new AuthorizeRule();
        IpFilteringAccessRule matchingRule = this.findMatchingAccessRule(action.scope(), host, action.resourceType(), action.operation());
        if (matchingRule != null) {
            authorizeRule.addRuleIfNotExist((AccessRule)matchingRule);
        }
        return authorizeRule;
    }

    public void addMatchingRule(ResourceAuthorizeRules matchingRules, Scope resourceScope, String host, Operation operation, ResourceType resourceType) {
        IpFilteringAccessRule matchingRule = this.findMatchingAccessRule(resourceScope, host, resourceType, operation);
        if (matchingRule != null) {
            matchingRules.addRuleIfNotExist((AccessRule)matchingRule);
        }
    }

    public int totalIpFilterings() {
        return this.ipFilteringMap.values().stream().mapToInt(Map::size).sum();
    }

    public long totalIpFilteringAccessRules() {
        return this.ipFilteringAccessRules.values().stream().mapToLong(Map::size).sum();
    }

    private IpFilteringAccessRule findMatchingAccessRule(Scope scope, String host, ResourceType resourceType, Operation operation) {
        String operationGroup = (String)this.ipFilteringOperationGroupMappings.getOrDefault(resourceType.name(), Collections.emptyMap()).get(operation.name());
        if (operationGroup == null) {
            log.error("Operation group mapping not found for resource type: {} and operation: {}", (Object)resourceType, (Object)operation);
            return null;
        }
        Map scopeIpFilteringRuleMap = this.ipFilteringAccessRules.getOrDefault(operationGroup, Collections.emptyMap());
        if (!scopeIpFilteringRuleMap.isEmpty()) {
            for (Scope nextScope = scope; nextScope != null; nextScope = nextScope.parent()) {
                IpFilteringAccessRule ipFilteringAccessRule = (IpFilteringAccessRule)((Object)scopeIpFilteringRuleMap.get(nextScope));
                if (ipFilteringAccessRule == null || !ipFilteringAccessRule.matches(host, operation, PermissionType.DENY)) continue;
                return ipFilteringAccessRule;
            }
        }
        return null;
    }

    private AuthValue updateIpFilteringKeyValue(AuthKey key, AuthValue value) {
        IpFilteringKey ipFilteringKey = (IpFilteringKey)key;
        IpFilteringValue ipFilteringValue = (IpFilteringValue)value;
        log.debug("Updating IP Filtering Auth Cache with key: {} and value: {}", (Object)ipFilteringKey, (Object)ipFilteringValue);
        this.validateScopePath(ipFilteringKey.scope());
        this.validateCidrs(ipFilteringValue.ipFilteringCidrs());
        IpFilteringValue oldValue = this.ipFilteringMap.computeIfAbsent(ipFilteringKey.operationGroup(), k -> new ConcurrentHashMap()).put(ipFilteringKey, ipFilteringValue);
        this.updateIpFilteringAccessRules(ipFilteringKey, ipFilteringValue);
        return oldValue;
    }

    private void updateIpFilteringAccessRules(IpFilteringKey ipFilteringKey, IpFilteringValue ipFilteringValue) {
        String operationGroup = ipFilteringKey.operationGroup();
        Scope scope = ipFilteringKey.scope();
        IpFilteringAccessRule ipFilteringAccessRule = new IpFilteringAccessRule(operationGroup, ipFilteringValue.ipFilteringCidrs(), scope);
        this.ipFilteringAccessRules.computeIfAbsent(operationGroup, k -> new ConcurrentHashMap()).put(scope, ipFilteringAccessRule);
        log.debug("Updated IP Filtering Access Rule: {}", (Object)ipFilteringAccessRule);
    }

    private AuthValue removeIpFilteringKey(AuthKey key) {
        IpFilteringKey ipFilteringKey = (IpFilteringKey)key;
        Map<IpFilteringKey, IpFilteringValue> map = this.ipFilteringMap.get(ipFilteringKey.operationGroup());
        if (map == null) {
            return null;
        }
        IpFilteringValue removedValue = map.remove(ipFilteringKey);
        if (map.isEmpty()) {
            this.ipFilteringMap.remove(ipFilteringKey.operationGroup());
        }
        log.info("Removed IP Filtering Auth Key: {} and value: {}", (Object)ipFilteringKey, (Object)removedValue);
        this.removeIpFilteringAccessRule(ipFilteringKey);
        return removedValue;
    }

    private void validateCidrs(Collection<String> cidrs) {
        for (String cidr : cidrs) {
            IpFilteringUtils.validateCidr(cidr);
        }
    }

    private void validateScopePath(Scope scope) {
        if (scope == null || scope.path() == null || scope.path().isEmpty()) {
            throw new InvalidScopeException("Scope and scope path must not be null or empty");
        }
        scope.validate();
        if (scope.path().size() > 2 || !((String)scope.path().get(0)).startsWith("organization") || scope.path().size() == 2 && !((String)scope.path().get(1)).startsWith("environment")) {
            throw new InvalidScopeException("IP Filtering only supports org level and env level scopes");
        }
        log.debug("Validated Scope: {}", (Object)scope);
    }

    private void removeIpFilteringAccessRule(IpFilteringKey ipFilteringKey) {
        String operationGroup = ipFilteringKey.operationGroup();
        Map<Scope, IpFilteringAccessRule> scopeIpFilteringRuleMap = this.ipFilteringAccessRules.get(operationGroup);
        if (scopeIpFilteringRuleMap != null) {
            scopeIpFilteringRuleMap.remove(ipFilteringKey.scope());
            if (scopeIpFilteringRuleMap.isEmpty()) {
                this.ipFilteringAccessRules.remove(operationGroup);
            }
        }
        log.debug("Removed IP Filtering Access Rule for the key: {}", (Object)ipFilteringKey);
    }

    private void convertIpfOperationGroupDefinitionsToMapping() {
        for (OperationGroupDefinition operationGroupDefinition : this.ipfOperationGroupDefinitions.operationGroupDefinitions()) {
            String operationGroup = operationGroupDefinition.name();
            Collection allowedOperations = operationGroupDefinition.allowedOperations();
            for (AccessPolicy.ResourceOperations allowedOperation : allowedOperations) {
                String resourceType = allowedOperation.resourceType();
                Collection operations = allowedOperation.operations();
                for (String operation : operations) {
                    this.ipFilteringOperationGroupMappings.computeIfAbsent(resourceType, k -> new ConcurrentHashMap()).put(operation, operationGroup);
                }
            }
        }
    }

    boolean accessRuleExists(IpFilteringKey ipFilteringKey) {
        Map<Scope, IpFilteringAccessRule> scopeIpFilteringRuleMap = this.ipFilteringAccessRules.get(ipFilteringKey.operationGroup());
        return scopeIpFilteringRuleMap != null && scopeIpFilteringRuleMap.get(ipFilteringKey.scope()) != null;
    }
}

