/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.secretregistry.storage;

import io.confluent.kafka.secretregistry.rest.SecretRegistryConfig;
import io.confluent.kafka.secretregistry.storage.KafkaStore;
import io.confluent.kafka.secretregistry.storage.Store;
import io.confluent.kafka.secretregistry.storage.StoreUpdateHandler;
import io.confluent.kafka.secretregistry.storage.exceptions.SerializationException;
import io.confluent.kafka.secretregistry.storage.exceptions.StoreException;
import io.confluent.kafka.secretregistry.storage.exceptions.StoreTimeoutException;
import io.confluent.kafka.secretregistry.storage.serialization.Serializer;
import io.confluent.kafka.secretregistry.utils.ShutdownableThread;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.RecordTooLargeException;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaStoreReaderThread<K, V>
extends ShutdownableThread {
    private static final Logger log = LoggerFactory.getLogger(KafkaStoreReaderThread.class);
    private final String topic;
    private final TopicPartition topicPartition;
    private final String groupId;
    private final StoreUpdateHandler<K, V> storeUpdateHandler;
    private final Serializer<K, V> serializer;
    private final Store<K, V> localStore;
    private final ReentrantLock offsetUpdateLock;
    private final Condition offsetReachedThreshold;
    private Consumer<byte[], byte[]> consumer;
    private final Producer<byte[], byte[]> producer;
    private volatile long offsetInSecretsTopic = -1L;
    private final K noopKey;
    private Properties consumerProps = new Properties();

    public KafkaStoreReaderThread(String bootstrapBrokers, String topic, String groupId, StoreUpdateHandler<K, V> storeUpdateHandler, Serializer<K, V> serializer, Store<K, V> localStore, Producer<byte[], byte[]> producer, K noopKey, SecretRegistryConfig config) {
        super("kafka-store-reader-thread-" + topic, false);
        this.offsetUpdateLock = new ReentrantLock();
        this.offsetReachedThreshold = this.offsetUpdateLock.newCondition();
        this.topic = topic;
        this.groupId = groupId;
        this.storeUpdateHandler = storeUpdateHandler;
        this.serializer = serializer;
        this.localStore = localStore;
        this.producer = producer;
        this.noopKey = noopKey;
        KafkaStore.addSecretRegistryConfigsToClientProperties(config, this.consumerProps);
        this.consumerProps.put("group.id", this.groupId);
        this.consumerProps.put("client.id", "KafkaStore-reader-" + this.topic);
        this.consumerProps.put("bootstrap.servers", bootstrapBrokers);
        this.consumerProps.put("auto.offset.reset", "earliest");
        this.consumerProps.put("enable.auto.commit", "false");
        this.consumerProps.put("key.deserializer", ByteArrayDeserializer.class);
        this.consumerProps.put("value.deserializer", ByteArrayDeserializer.class);
        log.info("Kafka store reader thread starting consumer");
        this.consumer = new KafkaConsumer(this.consumerProps);
        int retries = 0;
        List partitions = null;
        while (retries++ < 10 && ((partitions = this.consumer.partitionsFor(this.topic)) == null || partitions.size() < 1)) {
            try {
                Thread.sleep(1000L);
            }
            catch (InterruptedException interruptedException) {}
        }
        if (partitions == null || partitions.size() < 1) {
            throw new IllegalArgumentException("Unable to subscribe to the Kafka topic " + topic + " backing this data store. Topic may not exist.");
        }
        if (partitions.size() > 1) {
            throw new IllegalStateException("Unexpected number of partitions in the " + topic + " topic. Expected 1 and instead got " + partitions.size());
        }
        this.topicPartition = new TopicPartition(topic, 0);
        this.consumer.assign(Arrays.asList(this.topicPartition));
        this.consumer.seekToBeginning(Arrays.asList(this.topicPartition));
        log.info("Initialized last consumed offset to " + this.offsetInSecretsTopic);
        log.debug("Kafka store reader thread started");
    }

    protected void readToEnd() {
        Set assignment = this.consumer.assignment();
        Map endOffsets = this.consumer.endOffsets((Collection)assignment);
        log.debug("Reading to end of offsets {}", (Object)endOffsets);
        block0: while (!endOffsets.isEmpty()) {
            Iterator it = endOffsets.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry entry = it.next();
                if (this.consumer.position((TopicPartition)entry.getKey()) >= (Long)entry.getValue()) {
                    it.remove();
                    continue;
                }
                this.poll();
                continue block0;
            }
        }
    }

    @Override
    public void doWork() {
        this.poll();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void poll() {
        try {
            ConsumerRecords records = this.consumer.poll(Duration.ofMillis(Long.MAX_VALUE));
            for (ConsumerRecord record : records) {
                Object messageKey = null;
                try {
                    messageKey = this.serializer.deserializeKey((byte[])record.key());
                }
                catch (SerializationException e) {
                    log.error("Failed to deserialize the secret or config key", (Throwable)e);
                    continue;
                }
                if (messageKey.equals(this.noopKey)) {
                    try {
                        this.offsetUpdateLock.lock();
                        this.offsetInSecretsTopic = record.offset();
                        this.offsetReachedThreshold.signalAll();
                        continue;
                    }
                    finally {
                        this.offsetUpdateLock.unlock();
                        continue;
                    }
                }
                Object message = null;
                try {
                    message = record.value() == null ? null : (Object)this.serializer.deserializeValue(messageKey, (byte[])record.value());
                }
                catch (SerializationException e) {
                    log.error("Failed to deserialize a secret or config update", (Throwable)e);
                    continue;
                }
                try {
                    log.trace("Applying update from record in topic {}, partition {} with offset {} to the local store", new Object[]{record.topic(), record.partition(), record.offset()});
                    boolean valid = this.storeUpdateHandler.validateUpdate(messageKey, message);
                    if (valid) {
                        if (message == null) {
                            this.localStore.delete(messageKey);
                        } else {
                            this.localStore.put(messageKey, message);
                        }
                        this.storeUpdateHandler.handleUpdate(messageKey, message);
                    } else if (this.localStore.get(messageKey) == null) {
                        try {
                            ProducerRecord producerRecord = new ProducerRecord(this.topic, Integer.valueOf(0), (Object)((byte[])record.key()), null);
                            this.producer.send(producerRecord);
                            log.debug("Tombstoned invalid key {}", messageKey);
                        }
                        catch (KafkaException ke) {
                            log.warn("Failed to tombstone invalid key {}", messageKey, (Object)ke);
                        }
                    }
                    try {
                        this.offsetUpdateLock.lock();
                        this.offsetInSecretsTopic = record.offset();
                        this.offsetReachedThreshold.signalAll();
                    }
                    finally {
                        this.offsetUpdateLock.unlock();
                    }
                }
                catch (StoreException se) {
                    log.error("Failed to add record from the Kafka topic" + this.topic + " the local store");
                }
            }
        }
        catch (WakeupException records) {
        }
        catch (RecordTooLargeException rtle) {
            throw new IllegalStateException("Consumer threw RecordTooLargeException. A secret has been written that exceeds the default maximum fetch size.", rtle);
        }
        catch (RuntimeException e) {
            log.error("KafkaStoreReader thread has died for an unknown reason.");
            throw e;
        }
    }

    @Override
    public void shutdown() {
        try {
            log.debug("Starting shutdown of KafkaStoreReaderThread.");
            super.initiateShutdown();
            if (this.consumer != null) {
                this.consumer.wakeup();
            }
            if (this.localStore != null) {
                this.localStore.close();
            }
            super.awaitShutdown();
            if (this.consumer != null) {
                this.consumer.close();
            }
            log.info("KafkaStoreReaderThread shutdown complete.");
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void waitUntilOffset(long offset, long timeout, TimeUnit timeUnit) throws StoreException {
        if (offset < 0L) {
            throw new StoreException("KafkaStoreReaderThread can't wait for a negative offset.");
        }
        log.trace("Waiting to read offset {}. Currently at offset {}", (Object)offset, (Object)this.offsetInSecretsTopic);
        try {
            this.offsetUpdateLock.lock();
            long timeoutNs = TimeUnit.NANOSECONDS.convert(timeout, timeUnit);
            while (this.offsetInSecretsTopic < offset && timeoutNs > 0L) {
                try {
                    timeoutNs = this.offsetReachedThreshold.awaitNanos(timeoutNs);
                }
                catch (InterruptedException e) {
                    log.debug("Interrupted while waiting for the background store reader thread to reach the specified offset: " + offset, (Throwable)e);
                }
            }
        }
        finally {
            this.offsetUpdateLock.unlock();
        }
        if (this.offsetInSecretsTopic < offset) {
            throw new StoreTimeoutException("KafkaStoreReaderThread failed to reach target offset within the timeout interval. targetOffset: " + offset + ", offsetReached: " + this.offsetInSecretsTopic + ", timeout(ms): " + TimeUnit.MILLISECONDS.convert(timeout, timeUnit));
        }
    }

    public String getConsumerProperty(String key) {
        return this.consumerProps.getProperty(key);
    }
}

