/*
 * Decompiled with CFR 0.152.
 */
package kafka.network.netty;

import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.stats.CumulativeSum;
import org.apache.kafka.common.metrics.stats.Value;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionMaxAgeManager {
    private static final Logger log = LoggerFactory.getLogger(ConnectionMaxAgeManager.class);
    private final Metrics metrics;
    private final CumulativeSum agedConnectionsKilledCount;
    private final MetricName agedConnectionsKilledCountMetricName;
    private final Value oldestConnectionAgeMs;
    private final MetricName oldestConnectionMsMetricName;
    private final Time time;
    private final SortedSet<Item> trackedConnections;
    private final ScheduledExecutorService executor;
    private final AtomicLong connectionExpiryTimeoutMs = new AtomicLong();
    private final AtomicLong connectionCloseIntervalMs = new AtomicLong();
    private volatile ScheduledFuture<?> connectionExpiryTask;

    public ConnectionMaxAgeManager(Metrics metrics, String metricsGroup, Map<String, String> metricsTags, Time time, long connectionExpiryTimeoutMs, long connectionCloseIntervalMs) {
        this.metrics = metrics;
        this.agedConnectionsKilledCount = new CumulativeSum();
        this.agedConnectionsKilledCountMetricName = metrics.metricName("aged-connections-killed-count", metricsGroup, metricsTags);
        this.metrics.addMetric(this.agedConnectionsKilledCountMetricName, this.agedConnectionsKilledCount);
        this.oldestConnectionAgeMs = new Value();
        this.oldestConnectionMsMetricName = metrics.metricName("oldest-connection-age-ms", metricsGroup, metricsTags);
        this.metrics.addMetric(this.oldestConnectionMsMetricName, this.oldestConnectionAgeMs);
        this.time = time;
        this.connectionExpiryTimeoutMs.set(connectionExpiryTimeoutMs);
        this.connectionCloseIntervalMs.set(connectionCloseIntervalMs);
        this.trackedConnections = new TreeSet<Item>();
        this.executor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread thread = new Thread(r, "ConnectionExpiryManagerThread");
            thread.setDaemon(true);
            return thread;
        });
        this.setupConnectionExpiryTask();
    }

    public synchronized void setConnectionMaxAgeTime(long connectionExpiryTimeoutMs) {
        this.connectionExpiryTimeoutMs.set(connectionExpiryTimeoutMs);
        this.setupConnectionExpiryTask();
    }

    public void trackConnectionAsync(String id, long creationTimeMs, long seqId, Runnable closeAction) {
        Item item = new Item(id, creationTimeMs, seqId, closeAction);
        this.executor.execute(() -> {
            boolean added = this.trackedConnections.add(item);
            log.debug("Tracked connection {}, added? {}", (Object)item, (Object)added);
        });
    }

    public void untrackConnectionAsync(String id, long creationTimeMs, long seqId) {
        this.executor.execute(() -> {
            Item item = Item.create(id, creationTimeMs, seqId);
            boolean removed = this.trackedConnections.remove(item);
            log.debug("Untracked connection {}, removed? {}", (Object)item, (Object)removed);
        });
    }

    public void close() {
        this.metrics.removeMetric(this.oldestConnectionMsMetricName);
        this.metrics.removeMetric(this.agedConnectionsKilledCountMetricName);
        if (this.connectionExpiryTask != null) {
            this.connectionExpiryTask.cancel(false);
        }
        this.trackedConnections.clear();
        this.executor.shutdown();
    }

    private void setupConnectionExpiryTask() {
        if (this.connectionExpiryTask != null) {
            this.connectionExpiryTask.cancel(false);
            this.connectionExpiryTask = null;
        }
        if (this.connectionExpiryTask == null) {
            this.connectionExpiryTask = this.executor.scheduleWithFixedDelay(this::closeOldestExpiredConnection, this.connectionCloseIntervalMs.get(), this.connectionCloseIntervalMs.get(), TimeUnit.MILLISECONDS);
        }
    }

    private void closeOldestExpiredConnection() {
        try {
            if (this.trackedConnections.isEmpty()) {
                log.debug("No connections to close");
                this.oldestConnectionAgeMs.record(null, 0.0, 0L);
                return;
            }
            Item oldestConnection = this.trackedConnections.first();
            long ageMs = this.time.milliseconds() - oldestConnection.creationTimeMs();
            this.oldestConnectionAgeMs.record(null, ageMs, 0L);
            if (ageMs >= this.connectionExpiryTimeoutMs.get()) {
                log.debug("Closing connection {} due to max age exceeded", (Object)oldestConnection);
                oldestConnection.closeAction().run();
                this.trackedConnections.remove(oldestConnection);
                this.agedConnectionsKilledCount.record(null, 1.0, 0L);
            }
        }
        catch (Throwable e) {
            log.debug("Error while closing the oldest expired connection", e);
        }
    }

    private record Item(String id, long creationTimeMs, long seqId, Runnable closeAction) implements Comparable<Item>
    {
        private static final Runnable NOOP_CLOSE_ACTION = () -> {};

        @Override
        public int compareTo(Item other) {
            if (this.creationTimeMs == other.creationTimeMs) {
                return Long.compare(this.seqId, other.seqId);
            }
            return Long.compare(this.creationTimeMs, other.creationTimeMs);
        }

        static Item create(String id, long creationTimeMs, long seqId) {
            return new Item(id, creationTimeMs, seqId, NOOP_CLOSE_ACTION);
        }

        @Override
        public String toString() {
            return "Item{id=" + this.id + ", creationTimeMs=" + this.creationTimeMs + ", seqId=" + this.seqId + "}";
        }
    }
}

