/*
 * Decompiled with CFR 0.152.
 */
package kafka.restore.snapshot;

import io.confluent.kafka.storage.cloud.LinkedCloudObject;
import io.confluent.kafka.storage.cloud.SequencedObject;
import io.confluent.kafka.storage.tier.TopicIdPartition;
import io.confluent.kafka.storage.tier.domain.AbstractTierMetadata;
import io.confluent.kafka.storage.tier.domain.TierPartitionForceRestore;
import io.confluent.kafka.storage.tier.state.FileTierPartitionStateSnapshotObject;
import io.confluent.kafka.storage.tier.store.DataTypePathPrefix;
import io.confluent.kafka.storage.tier.store.objects.FragmentType;
import io.confluent.kafka.storage.tier.store.objects.metadata.ObjectStoreMetadata;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import kafka.restore.RestoreMetricsManager;
import kafka.restore.db.PartitionRestoreContext;
import kafka.restore.snapshot.PointInTimeTierPartitionStateBuilder;
import kafka.tier.backupObjectLifecycle.ObjectStoreUtils;
import kafka.tier.backupObjectLifecycle.ObjectStoreUtilsContext;
import kafka.tier.backupObjectLifecycle.RetryPolicy;
import kafka.tier.exceptions.TierMetadataFatalException;
import kafka.tier.exceptions.TierObjectStoreFatalException;
import kafka.tier.snapshot.TierTopicSnapshotManager;
import kafka.tier.snapshot.TierTopicSnapshotObject;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.TierObjectStoreResponse;
import kafka.tier.store.VersionInformation;
import kafka.tier.store.objects.metadata.TierPartitionStateSnapshotMetadata;
import kafka.tier.store.objects.metadata.TierTopicSnapshotMetadata;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnapshotObjectStoreUtils {
    private static final Logger log = LoggerFactory.getLogger(SnapshotObjectStoreUtils.class);
    private static final String TIER_TOPIC_SNAPSHOT_PREFIX = DataTypePathPrefix.TIER_TOPIC_SNAPSHOT.prefix() + "/";
    private final Pattern ftpsSnapshotPathPattern;
    private final String keyPrefix;
    private final ObjectStoreUtilsContext storeCtx;
    private final RetryPolicy retryPolicy;
    private final ThreadPoolExecutor pool;
    private final Time time;
    private final RestoreMetricsManager metricsManager;

    public SnapshotObjectStoreUtils(TierObjectStore store, ThreadPoolExecutor pool, RetryPolicy retryPolicy, String keyPrefix, Time time, RestoreMetricsManager metricsManager) {
        this.pool = pool;
        this.retryPolicy = retryPolicy;
        this.keyPrefix = keyPrefix;
        this.time = time;
        this.metricsManager = metricsManager;
        this.storeCtx = new ObjectStoreUtilsContext(store, () -> true, () -> false);
        this.ftpsSnapshotPathPattern = Pattern.compile(Pattern.quote(keyPrefix + DataTypePathPrefix.TIER_PARTITION_STATE_METADATA_SNAPSHOT.prefix() + "/") + "(?<topicId>[^/]+)/(?<partition>\\d+)/(?<snapshotName>.*)_v.*");
    }

    private void downloadFragmentToFile(ObjectStoreMetadata metadata, FragmentType fragmentType, Path destination) throws IOException, InterruptedException {
        try (TierObjectStoreResponse response = ObjectStoreUtils.getObjectStoreFragment(this.storeCtx, metadata, fragmentType, this.retryPolicy);
             InputStream in = response.getInputStream();){
            Files.copy(in, destination, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (InterruptedException e) {
            log.error("Execution of downloadObjectToFile cancelled", e);
            throw new TierObjectStoreFatalException("Execution of downloadObjectToFile cancelled", e);
        }
    }

    protected Set<String> listFilenames(String key) {
        try {
            long startTimeMs = this.time.hiResClockMs();
            Set<String> fileNames = ObjectStoreUtils.listObject(this.storeCtx, this.keyPrefix + key, false, this.retryPolicy).keySet();
            long timeToListObjectsMs = this.time.hiResClockMs() - startTimeMs;
            this.metricsManager.restoreListVersionsMs().record(timeToListObjectsMs);
            for (String filename : fileNames) {
                log.debug("found file: " + filename);
            }
            return fileNames;
        }
        catch (InterruptedException e) {
            log.error("Execution of listObject cancelled", e);
            throw new TierObjectStoreFatalException("Execution of listObject cancelled", e);
        }
    }

    protected List<String> listTtpsSnapshotsByStartTimestamp(long startTimestamp) {
        ArrayList<SequencedObject> ttpsObjects = new ArrayList<SequencedObject>();
        long daysFromEpochNow = TierTopicSnapshotObject.getDaysFromEpoch(this.time.milliseconds());
        long ts = startTimestamp;
        while (TierTopicSnapshotObject.getDaysFromEpoch(ts) <= daysFromEpochNow) {
            String ttpsSnapshotDayPrefix = TIER_TOPIC_SNAPSHOT_PREFIX + TierTopicSnapshotObject.getDirNameWithDayPrefix(ts);
            List<SequencedObject> snapshotObjects = this.listTtpsSnapshotsByPrefix(ttpsSnapshotDayPrefix);
            log.debug(String.format("found ttps objects, {timestamp: %s, prefix: %s, ttps snapshot count: %s", ts, ttpsSnapshotDayPrefix, snapshotObjects.size()));
            ttpsObjects.addAll(snapshotObjects);
            ts += TimeUnit.DAYS.toMillis(1L);
        }
        List<String> ttpsSnapshots = ttpsObjects.stream().map(snapshot -> new TierTopicSnapshotMetadata((TierTopicSnapshotObject)snapshot).toFragmentLocation("", FragmentType.TIER_TOPIC_SNAPSHOT).get().objectPath()).collect(Collectors.toList());
        return ttpsSnapshots;
    }

    private List<SequencedObject> listTtpsSnapshotsByPrefix(String prefix) {
        try {
            long startTimeMs = this.time.hiResClockMs();
            Map<String, List<VersionInformation>> objects = ObjectStoreUtils.listObject(this.storeCtx, this.keyPrefix + prefix, false, this.retryPolicy);
            long timeToListObjectsMs = this.time.hiResClockMs() - startTimeMs;
            this.metricsManager.restoreListVersionsMs().record(timeToListObjectsMs);
            return LinkedCloudObject.getOptimalChain(TierTopicSnapshotManager.convertObjListToSeqList(objects));
        }
        catch (InterruptedException e) {
            log.error("Execution of listObject cancelled", e);
            throw new TierObjectStoreFatalException("Execution of listObject cancelled", e);
        }
    }

    public Map<TopicIdPartition, Future<Path>> downloadFtpsSnapshotsInParallel(Map<TopicIdPartition, TierPartitionStateSnapshotMetadata> ftpsSnapshots, Path toLocalDir) {
        HashMap<TopicIdPartition, Future<Path>> snapshotFileFutures = new HashMap<TopicIdPartition, Future<Path>>(ftpsSnapshots.size());
        ftpsSnapshots.forEach((tpid, snapshotMetadata) -> {
            if (snapshotMetadata == null) {
                log.warn("Tier Partition Snapshot for TopicIdPartition " + String.valueOf(tpid) + " not found in object store");
                return;
            }
            snapshotFileFutures.put((TopicIdPartition)tpid, this.pool.submit(() -> {
                String snapshotPathInCloud = snapshotMetadata.toFragmentLocation(this.keyPrefix, FragmentType.TIER_PARTITION_STATE_METADATA_SNAPSHOT).get().objectPath();
                Path snapshotPathInLocal = Paths.get(toLocalDir.toString(), snapshotPathInCloud);
                if (!snapshotPathInLocal.getParent().toFile().exists()) {
                    snapshotPathInLocal.getParent().toFile().mkdirs();
                }
                snapshotPathInLocal.toFile().createNewFile();
                try {
                    log.debug(String.format("[%s]: download snapshot and store it to %s", tpid.topicPartition(), snapshotPathInLocal));
                    long startTimeMs = this.time.hiResClockMs();
                    this.downloadFragmentToFile((ObjectStoreMetadata)snapshotMetadata, FragmentType.TIER_PARTITION_STATE_METADATA_SNAPSHOT, snapshotPathInLocal);
                    long timeToFetchFtpsMs = this.time.hiResClockMs() - startTimeMs;
                    this.metricsManager.restoreFetchFtpsMs().record(timeToFetchFtpsMs);
                }
                catch (InterruptedException | TierObjectStoreFatalException e) {
                    log.error(String.format("[%s]: getObject failed, related snapshot metadata: %s", tpid.topicPartition(), snapshotMetadata), e);
                    return null;
                }
                return snapshotPathInLocal;
            }));
        });
        return snapshotFileFutures;
    }

    public List<Future<Path>> downloadTierTopicSnapshotsInParallel(List<TierTopicSnapshotMetadata> allObjectMetadata, Path toLocalDir) {
        ArrayList<Future<Path>> tierTopicSnapshotFileFutures = new ArrayList<Future<Path>>(allObjectMetadata.size());
        for (TierTopicSnapshotMetadata objectMetadata : allObjectMetadata) {
            tierTopicSnapshotFileFutures.add(this.pool.submit(() -> {
                String filePathInCloud = objectMetadata.toFragmentLocation(this.keyPrefix, FragmentType.TIER_TOPIC_SNAPSHOT).get().objectPath();
                Path filePathInLocal = Paths.get(toLocalDir.toString(), filePathInCloud);
                if (!filePathInLocal.getParent().toFile().exists()) {
                    filePathInLocal.getParent().toFile().mkdirs();
                }
                filePathInLocal.toFile().createNewFile();
                log.debug(String.format("download tier topic snapshot and store it to %s", filePathInLocal.toString()));
                long startMs = this.time.hiResClockMs();
                this.downloadFragmentToFile(objectMetadata, FragmentType.TIER_TOPIC_SNAPSHOT, filePathInLocal);
                long timeToDownloadTierTopicSnapshotsMs = this.time.hiResClockMs() - startMs;
                this.metricsManager.restoreFetchTierTopicSnapshotMs().record(timeToDownloadTierTopicSnapshotsMs);
                return filePathInLocal;
            }));
        }
        return tierTopicSnapshotFileFutures;
    }

    public PointInTimeTierPartitionStateBuilder.FtpsSnapshotsMetadata locateFtpsSnapshotsByTimestamp(Map<TopicPartition, PartitionRestoreContext> partitionRestoreContextMap) {
        ArrayList futures = new ArrayList(partitionRestoreContextMap.size());
        partitionRestoreContextMap.values().forEach(pCtx -> {
            TopicIdPartition topicIdPartition = pCtx.topicIdPartition();
            String snapshotDir = this.keyPrefix + DataTypePathPrefix.TIER_PARTITION_STATE_METADATA_SNAPSHOT.prefix() + "/" + topicIdPartition.topicIdAsBase64() + "/" + topicIdPartition.partition() + "/";
            long timestamp = pCtx.revertCompactionSinceTimestamp;
            futures.add(this.pool.submit(() -> this.listFilenames(snapshotDir).stream().map(filename -> this.parseTierPartitionSnapshotFilename(topicIdPartition.topic(), (String)filename)).filter(Optional::isPresent).map(Optional::get).filter(metadata -> metadata.snapshotObject().snapshotTimestampMs() <= timestamp).collect(Collectors.toMap(TierPartitionStateSnapshotMetadata::topicIdPartition, Function.identity(), (oldMetadata, newMetadata) -> newMetadata.snapshotObject().snapshotTimestampMs() > oldMetadata.snapshotObject().snapshotTimestampMs() ? newMetadata : oldMetadata))));
        });
        HashMap<TopicIdPartition, TierPartitionStateSnapshotMetadata> results = new HashMap<TopicIdPartition, TierPartitionStateSnapshotMetadata>();
        partitionRestoreContextMap.values().forEach(pCtx -> results.put(pCtx.topicIdPartition(), null));
        for (Future fut : futures) {
            try {
                Map topicMetadata = (Map)fut.get();
                results.putAll(topicMetadata);
            }
            catch (InterruptedException | ExecutionException e) {
                log.error("Parallel execution interrupted", e);
            }
        }
        return new PointInTimeTierPartitionStateBuilder.FtpsSnapshotsMetadata(results);
    }

    public List<TierTopicSnapshotMetadata> locateTierTopicSnapshotsByTimestamp(long startTimestamp, OptionalLong endTimestamp) {
        TierTopicSnapshotMetadata firstSnapshotMetadata;
        if (endTimestamp.isPresent() && startTimestamp >= endTimestamp.getAsLong()) {
            log.warn("Attempted to fetch range of Tier Topic Snapshots with starting timestamp >= ending timestamp");
            return Collections.emptyList();
        }
        long endTs = endTimestamp.orElse(Long.MAX_VALUE);
        List<TierTopicSnapshotMetadata> snapshotMetadatas = this.listTtpsSnapshotsByStartTimestamp(startTimestamp).stream().map(this::parseTierTopicSnapshotFilename).filter(Optional::isPresent).map(Optional::get).filter(metadata -> {
            long maxTs;
            long minTs = metadata.snapshotObject().startTimestampMs();
            return minTs <= (maxTs = metadata.snapshotObject().endTimestampMs()) && startTimestamp <= maxTs && minTs <= endTs;
        }).sorted(new TierTopicSnapshotMetadataComparator()).collect(Collectors.toList());
        if (log.isDebugEnabled()) {
            for (TierTopicSnapshotMetadata snapshot : snapshotMetadatas) {
                log.debug("found snapshot: " + String.valueOf(snapshot.snapshotObject()));
            }
        }
        if (snapshotMetadatas.size() != 0 && (firstSnapshotMetadata = snapshotMetadatas.get(0)).snapshotObject().startTimestampMs() > startTimestamp) {
            log.warn("there is a gap between events of ttps snapshots and the ftps snapshots, return empty ttps list");
            return new ArrayList<TierTopicSnapshotMetadata>();
        }
        return snapshotMetadatas;
    }

    protected Optional<TierPartitionStateSnapshotMetadata> parseTierPartitionSnapshotFilename(String topicName, String filename) {
        try {
            TierPartitionStateSnapshotMetadata metadata = this.decodeFtpsSnapshotPath(filename, topicName);
            return Optional.of(metadata);
        }
        catch (IllegalArgumentException e) {
            log.warn("Invalid tier metadata snapshot filename: " + filename, e);
            return Optional.empty();
        }
    }

    private Optional<TierTopicSnapshotMetadata> parseTierTopicSnapshotFilename(String filename) {
        try {
            TierTopicSnapshotMetadata metadata = TierTopicSnapshotMetadata.fromPath(filename);
            return Optional.of(metadata);
        }
        catch (Exception e) {
            log.warn("Invalid tier topic snapshot filename: " + filename, e);
            return Optional.empty();
        }
    }

    public TierPartitionStateSnapshotMetadata decodeFtpsSnapshotPath(String path, String topicName) {
        Matcher m = this.ftpsSnapshotPathPattern.matcher(path);
        if (!m.matches()) {
            throw new IllegalArgumentException("Invalid tier partition state snapshot filename: " + path);
        }
        int partition = Integer.parseInt(m.group("partition"));
        UUID topicId = Utils.uuidFromBase64(m.group("topicId"));
        TopicIdPartition tpid = new TopicIdPartition(topicName, topicId, partition);
        FileTierPartitionStateSnapshotObject snapshotObject = FileTierPartitionStateSnapshotObject.decodeSnapshotName(m.group("snapshotName"));
        return new TierPartitionStateSnapshotMetadata(tpid, snapshotObject);
    }

    public ByteBuffer fetchRecoverSnapshot(TierPartitionForceRestore forceRestoreEvent) throws InterruptedException {
        return ObjectStoreUtils.fetchRecoverSnapshot(this.storeCtx, forceRestoreEvent, this.retryPolicy);
    }

    public static AbstractTierMetadata deserializeRecord(ConsumerRecord<byte[], byte[]> record) {
        Optional<AbstractTierMetadata> entryOpt = AbstractTierMetadata.deserialize(record.key(), record.value(), record.timestamp());
        if (!entryOpt.isPresent()) {
            throw new TierMetadataFatalException(String.format("Fatal Exception message for %s and unknown type: %d cannot be deserialized.", AbstractTierMetadata.deserializeKey(record.key()), AbstractTierMetadata.getTypeId(record.value())));
        }
        return entryOpt.get();
    }

    private static class TierTopicSnapshotMetadataComparator
    implements Comparator<TierTopicSnapshotMetadata> {
        private TierTopicSnapshotMetadataComparator() {
        }

        @Override
        public int compare(TierTopicSnapshotMetadata o1, TierTopicSnapshotMetadata o2) {
            if (Objects.equals(o1.snapshotObject().currentEpochAndSeqNumber(), o2.snapshotObject().currentEpochAndSeqNumber())) {
                if (Objects.equals(o1.snapshotObject().startTimestampMs(), o2.snapshotObject().startTimestampMs())) {
                    return Long.compare(o1.snapshotObject().endTimestampMs(), o2.snapshotObject().endTimestampMs());
                }
                return Long.compare(o1.snapshotObject().startTimestampMs(), o2.snapshotObject().startTimestampMs());
            }
            return o1.snapshotObject().currentEpochAndSeqNumber().compareTo(o2.snapshotObject().currentEpochAndSeqNumber());
        }
    }
}

