/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.store.encryption;

import io.confluent.kafka.multitenant.ByokMetadata;
import io.confluent.kafka.multitenant.KafkaLogicalClusterMetadata;
import io.confluent.kafka.storage.tier.TopicIdPartition;
import io.confluent.kafka.storage.tier.store.DataTypePathPrefix;
import io.confluent.kafka.storage.tier.store.OpaqueData;
import java.io.Closeable;
import java.nio.ByteBuffer;
import java.security.GeneralSecurityException;
import java.security.SecureRandom;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import kafka.tier.exceptions.TierObjectStoreFatalException;
import kafka.tier.exceptions.TierObjectStoreRetriableException;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.encryption.CleartextDataKey;
import kafka.tier.store.encryption.DataEncryptionKeyHolder;
import kafka.tier.store.encryption.EncryptedDataKey;
import kafka.tier.store.encryption.KeySha;
import kafka.tier.store.encryption.TenantAwareEncryptionKeyManagerMetrics;
import kafka.tier.store.encryption.TenantKeyCache;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.multitenant.LogicalClusterMetadata;
import org.apache.kafka.server.multitenant.MultiTenantMetadata;
import org.apache.kafka.server.util.KafkaScheduler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class TenantAwareEncryptionKeyManager
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(TenantAwareEncryptionKeyManager.class);
    private static final String TOPIC_LOGICAL_CLUSTER_ID_DELIMITER = "_";
    private static final Integer PROACTIVE_KEY_GENERATION_RETRY_DELAY_MS = 300000;
    public static final String METADATA_SHA_KEY = "ioconfluentkeysha256";
    public static final String METADATA_SHA_KEY_LEGACY1 = "ioConfluentKeySha256";
    public static final String METADATA_SHA_KEY_LEGACY2 = "io.confluent/key-sha-256";
    public static final String METADATA_DATA_KEY = "ioconfluentbase64encrypteddatakey";
    public static final String METADATA_DATA_KEY_LEGACY1 = "ioConfluentBase64EncryptedDataKey";
    public static final String METADATA_DATA_KEY_LEGACY2 = "io.confluent/base64-encrypted-data-key";
    public static final String METADATA_KEY_CREATION_TIME = "ioconfluentkeycreationtime";
    public static final String METADATA_KEY_CREATION_TIME_LEGACY1 = "ioConfluentKeyCreationTime";
    public static final String METADATA_KEY_CREATION_TIME_LEGACY2 = "io.confluent/key-creation-time";
    public static final String LKC_PREFIX = "lkc-";
    private final Boolean proactiveKeyGenerationEnable;
    private final KafkaScheduler scheduler;
    private final MultiTenantMetadata multiTenantMetadata;
    private final Duration keyRefreshInterval;
    private final int maxTenantKeyCacheSize;
    final Time time;
    final TenantAwareEncryptionKeyManagerMetrics metrics;
    private final Map<String, TenantKeyCache> keyCache = new ConcurrentHashMap<String, TenantKeyCache>();

    public TenantAwareEncryptionKeyManager(MultiTenantMetadata multiTenantMetadata, Duration maxKeyAge, int maxTenantKeyCacheSize, long tenantKeyCacheEvictionTimeSeconds, Time time, Metrics metrics, KafkaScheduler scheduler, boolean proactiveKeyGenerationEnable) {
        this.multiTenantMetadata = multiTenantMetadata;
        this.keyRefreshInterval = maxKeyAge;
        this.maxTenantKeyCacheSize = maxTenantKeyCacheSize;
        this.time = time;
        if (metrics != null) {
            this.metrics = new TenantAwareEncryptionKeyManagerMetrics(metrics);
            this.metrics.updateMaxKeyAge(maxKeyAge);
        } else {
            this.metrics = null;
        }
        this.scheduler = scheduler;
        scheduler.schedule("tenant-key-cache-time-based-eviction", () -> this.keyCache.values().forEach(tenantKeyCacheMap -> tenantKeyCacheMap.maybeEvictEntriesTimeBased(tenantKeyCacheEvictionTimeSeconds)), TimeUnit.SECONDS.toMillis(tenantKeyCacheEvictionTimeSeconds), TimeUnit.SECONDS.toMillis(tenantKeyCacheEvictionTimeSeconds));
        this.proactiveKeyGenerationEnable = proactiveKeyGenerationEnable;
    }

    @Override
    public void close() {
        if (this.metrics != null) {
            this.metrics.close();
        }
        if (this.scheduler != null) {
            try {
                this.scheduler.shutdown();
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public void clear() {
        this.keyCache.clear();
    }

    public void clear(String logicalClusterId) {
        TenantKeyCache tenantKeyCache = this.keyCache.remove(logicalClusterId);
        if (this.metrics != null) {
            this.metrics.decrementNumberOfByokTenants();
        }
        tenantKeyCache.clear();
    }

    protected Time getTime() {
        return this.time;
    }

    public Duration getKeyRefreshInterval() {
        return this.keyRefreshInterval;
    }

    public int getMaxTenantKeyCacheSize() {
        return this.maxTenantKeyCacheSize;
    }

    private ByokMetadata getByokMetadata(String logicalClusterId) {
        if (!this.multiTenantMetadata.isUp()) {
            throw new TierObjectStoreRetriableException("Cannot access BYOK metadata because MultiTenant Metadata is not up yet.");
        }
        LogicalClusterMetadata logicalClusterMetadata = this.multiTenantMetadata.metadata(logicalClusterId);
        if (logicalClusterMetadata == null) {
            return null;
        }
        if (!(logicalClusterMetadata instanceof KafkaLogicalClusterMetadata)) {
            throw new IllegalStateException(String.format("[%s] Unexpected instance of type: %s, we always expect %s", logicalClusterId, logicalClusterMetadata.getClass(), KafkaLogicalClusterMetadata.class));
        }
        return ((KafkaLogicalClusterMetadata)logicalClusterMetadata).byokMetadata();
    }

    private TenantKeyCache getOrCreateTenantKeyCache(String logicalClusterId) {
        TenantKeyCache newCache = new TenantKeyCache(this.time, logicalClusterId, this.getMaxTenantKeyCacheSize());
        TenantKeyCache existingCache = this.keyCache.putIfAbsent(logicalClusterId, newCache);
        if (existingCache == null && this.metrics != null) {
            this.metrics.incrementNumberOfByokTenants();
        }
        return existingCache == null ? newCache : existingCache;
    }

    private final KeySha activeKeySha(String logicalClusterId) {
        log.debug("[{}] Trying to get active key", (Object)logicalClusterId);
        ByokMetadata byokMetadata = this.getByokMetadata(logicalClusterId);
        TenantKeyCache tenantKeyCache = this.keyCache.get(logicalClusterId);
        KeySha emptyActiveKey = KeySha.fromRawBytes(OpaqueData.ZEROED.intoByteArray());
        if (byokMetadata == null) {
            if (tenantKeyCache == null) {
                log.debug("[{}] No ByokMetadata found, and no active key found in cache, returning empty active key.", (Object)logicalClusterId);
                return emptyActiveKey;
            }
            KeySha existingActiveKey = tenantKeyCache.activeKeySha();
            if (existingActiveKey == null) {
                log.warn("[{}] No ByokMetadata found, but tenant present in cache, so tenant may be deactivated. Active key is missing, so returning empty active key.", (Object)logicalClusterId);
                return emptyActiveKey;
            }
            log.warn("[{}] No ByokMetadata found, but tenant present in cache, so tenant may be deactivated. Trying to reuse existing active key: {}.", (Object)logicalClusterId, (Object)existingActiveKey);
            return existingActiveKey;
        }
        if (tenantKeyCache == null) {
            tenantKeyCache = this.getOrCreateTenantKeyCache(logicalClusterId);
        }
        this.maybeRotateActiveKey(logicalClusterId, tenantKeyCache, byokMetadata);
        return tenantKeyCache.activeKeySha();
    }

    public DataEncryptionKeyHolder key(String logicalClusterId, KeySha keySha) {
        TenantKeyCache tenantKeyCache = this.tenantKeyCache(logicalClusterId);
        if (tenantKeyCache == null) {
            return null;
        }
        return tenantKeyCache.get(keySha);
    }

    TenantKeyCache tenantKeyCache(String logicalClusterId) {
        return this.keyCache.get(logicalClusterId);
    }

    public final DataEncryptionKeyHolder registerKeyFromObjectMetadata(String logicalClusterId, Map<String, String> objectMetadata) {
        ByokMetadata byokMetadata = this.getByokMetadata(logicalClusterId);
        if (byokMetadata == null) {
            throw new TierObjectStoreFatalException(String.format("[%s] Found BYOK metadata inconsistency, object metadata can't be registered when BYOK metadata is absent.", logicalClusterId));
        }
        TenantKeyCache tenantKeyCache = this.getOrCreateTenantKeyCache(logicalClusterId);
        DataEncryptionKeyHolder holder = this.parseKeyFromObjectMetadata(logicalClusterId, byokMetadata, objectMetadata, true);
        log.info("[{}] Registering key {} decoded from object metadata", (Object)logicalClusterId, (Object)holder.keySha);
        tenantKeyCache.add(holder);
        return holder;
    }

    private void maybeRotateActiveKey(String logicalClusterId, TenantKeyCache tenantKeyCache, ByokMetadata byokMetadata) {
        KeySha active = tenantKeyCache.activeKeySha();
        if (active == null) {
            log.info("[{}] No active key found, seeding key cache", (Object)logicalClusterId);
            this.forceRotate(logicalClusterId, tenantKeyCache, byokMetadata);
            active = tenantKeyCache.activeKeySha();
        }
        DataEncryptionKeyHolder holder = tenantKeyCache.get(active);
        if (holder.keyCreationTimeOpt.isPresent()) {
            Instant creationTime = holder.keyCreationTimeOpt.get();
            Instant timeNow = Instant.ofEpochMilli(this.time.milliseconds());
            Instant deadline = creationTime.plus(this.keyRefreshInterval);
            Optional<Instant> expiredKeyToleranceTimeOpt = holder.expiredKeyToleranceTimeOpt;
            if (timeNow.isAfter(deadline)) {
                if (expiredKeyToleranceTimeOpt.isPresent()) {
                    if (timeNow.isBefore(expiredKeyToleranceTimeOpt.get())) {
                        log.info("[{}] Key corresponding to {} created at {} has expired, but is still within the expired key tolerance time {}.", logicalClusterId, active, creationTime, expiredKeyToleranceTimeOpt.get());
                        return;
                    }
                    log.info("[{}] Key corresponding to {} created at {} has expired and is no longer within the expired key tolerance time {}, seeding key cache", logicalClusterId, active, creationTime, expiredKeyToleranceTimeOpt.get());
                } else {
                    log.info("[{}] Key corresponding to {} created at {} has expired determined by the refresh interval {} and there is no provided expired key tolerance time, seeding key cache", logicalClusterId, active, creationTime, this.keyRefreshInterval);
                }
                this.forceRotate(logicalClusterId, tenantKeyCache, byokMetadata);
            }
        } else {
            throw new TierObjectStoreFatalException(String.format("[%s] Key corresponding to %s has not been checked for rotation since no corresponding creation time has been found.", logicalClusterId, active));
        }
    }

    private boolean forceRotate(String logicalClusterId, TenantKeyCache tenantKeyCache, ByokMetadata byokMetadata) {
        DataEncryptionKeyHolder newKey;
        DataEncryptionKeyHolder fallbackActiveKey;
        block13: {
            Map<String, String> metadata;
            fallbackActiveKey = null;
            if (tenantKeyCache.activeKeySha() != null && tenantKeyCache.get(tenantKeyCache.activeKeySha()) != null) {
                fallbackActiveKey = tenantKeyCache.get(tenantKeyCache.activeKeySha());
            }
            if ((metadata = this.fetchWellKnownPathMetadata(logicalClusterId)) != null && !metadata.isEmpty()) {
                try {
                    DataEncryptionKeyHolder newKeyHolder = this.parseActiveKeyFromObjectMetadata(logicalClusterId, byokMetadata, metadata);
                    if (newKeyHolder.keyCreationTimeOpt.isPresent()) {
                        Instant creationTime = newKeyHolder.keyCreationTimeOpt.get();
                        Instant timeNow = Instant.ofEpochMilli(this.time.milliseconds());
                        Instant deadline = creationTime.plus(this.keyRefreshInterval);
                        log.info("[{}] Recovered previously written active key {} created at {} from the well-known keypath", logicalClusterId, newKeyHolder.keySha, creationTime);
                        if (timeNow.isBefore(deadline)) {
                            log.info("[{}] Using key {} as the active key", (Object)logicalClusterId, (Object)newKeyHolder.keySha);
                            if (this.metrics != null) {
                                this.metrics.updateActiveKeyCreationTime(newKeyHolder.keyCreationTimeOpt.get());
                            }
                            tenantKeyCache.replaceActiveKey(newKeyHolder);
                            return true;
                        }
                        log.info("[{}] Key {} recovered from the well-known key path is old to use as the active key, but we could still consider it as a fallback key", (Object)logicalClusterId, (Object)newKeyHolder.keySha);
                        tenantKeyCache.add(newKeyHolder);
                        if (fallbackActiveKey != null) {
                            fallbackActiveKey = newKeyHolder;
                        }
                        break block13;
                    }
                    throw new TierObjectStoreFatalException(String.format("[%s] Key %s could not be rotated since creation time was not found. This should never happen because an active key is always supposed to have a creation time associated with it.", logicalClusterId, newKeyHolder.keySha));
                }
                catch (TierObjectStoreRetriableException ex) {
                    log.error("[{}] Failed to decrypt active key from the well-known key path.", (Object)logicalClusterId, (Object)ex);
                    this.maybeFallbackToExpiredActiveKey(logicalClusterId, tenantKeyCache, fallbackActiveKey);
                    return false;
                }
            }
        }
        log.info("[{}] Unable to restore a valid active key from the well-known key path, trying to generate a new one", (Object)logicalClusterId);
        try {
            byte[] encodedDataKeyBytes = this.getEncodedDataKeyBytes(logicalClusterId);
            Optional<CleartextDataKey> cleartextDataKeyOpt = Optional.empty();
            if (encodedDataKeyBytes != null && encodedDataKeyBytes.length > 0) {
                cleartextDataKeyOpt = Optional.of(new CleartextDataKey(encodedDataKeyBytes));
            }
            newKey = this.encryptKey(logicalClusterId, cleartextDataKeyOpt, byokMetadata);
            log.info("[{}] Using key {} as the active key", (Object)logicalClusterId, (Object)newKey.keySha);
            if (this.metrics != null) {
                this.metrics.updateActiveKeyCreationTime(newKey.keyCreationTimeOpt.get());
            }
            tenantKeyCache.replaceActiveKey(newKey);
        }
        catch (TierObjectStoreFatalException e) {
            log.error("[{}] Failed to generate a new key due to a permanent failure.", (Object)logicalClusterId, (Object)e);
            throw e;
        }
        catch (TierObjectStoreRetriableException e) {
            log.error("[{}] Failed to generate a new key due to a retriable failure.", (Object)logicalClusterId, (Object)e);
            this.maybeFallbackToExpiredActiveKey(logicalClusterId, tenantKeyCache, fallbackActiveKey);
            return false;
        }
        log.info("[{}] Writing out newly generated key {} to the well-known key path", (Object)logicalClusterId, (Object)newKey.keySha);
        this.writeWellKnownPathMetadata(logicalClusterId, this.keyToObjectMetadata(newKey));
        return true;
    }

    private void maybeFallbackToExpiredActiveKey(String logicalClusterId, TenantKeyCache tenantKeyCache, DataEncryptionKeyHolder fallbackExpiredActiveKey) {
        if (fallbackExpiredActiveKey == null) {
            throw new TierObjectStoreRetriableException(String.format("[%s] Failed to rotate key and could not find an existing active key to fallback to... Retrying", logicalClusterId));
        }
        log.error("[{}] Failed to rotate key, falling back to expired active key {}...", (Object)logicalClusterId, (Object)fallbackExpiredActiveKey.keySha);
        Instant expiredKeyToleranceTime = Instant.ofEpochMilli(this.time.milliseconds()).plus(Duration.of(2L, ChronoUnit.HOURS));
        fallbackExpiredActiveKey = new DataEncryptionKeyHolder(fallbackExpiredActiveKey.encryptedDataKey, fallbackExpiredActiveKey.cleartextDataKey, fallbackExpiredActiveKey.keyCreationTimeOpt, Optional.of(expiredKeyToleranceTime));
        tenantKeyCache.replaceActiveKey(fallbackExpiredActiveKey);
    }

    public synchronized void maybeCreateActiveKeysForAllAssignedTenants(Set<String> logicalClusterIds) {
        if (!this.proactiveKeyGenerationEnable.booleanValue()) {
            return;
        }
        if (this.multiTenantMetadata.isUp()) {
            logicalClusterIds.forEach(logicalClusterId -> this.scheduler.scheduleOnce(String.format("proactive-key-generation-%s", logicalClusterId), () -> this.maybeCreateActiveKeyForTenant((String)logicalClusterId)));
        } else {
            log.error("MultiTenantMetadata is not up, unable to create active key for assigned tenants. Retrying after 5 minutes...");
            this.scheduler.scheduleOnce("proactive-key-generation-retry-all-tenants", () -> this.maybeCreateActiveKeysForAllAssignedTenants(logicalClusterIds), PROACTIVE_KEY_GENERATION_RETRY_DELAY_MS.intValue());
        }
    }

    public void scheduleProactiveKeyGenerationForTopicIdPartition(TopicIdPartition topicIdPartition, String suffix) {
        if (!this.proactiveKeyGenerationEnable.booleanValue()) {
            return;
        }
        String logicalClusterId = this.maybeGetLogicalClusterId(topicIdPartition);
        if (logicalClusterId == null) {
            return;
        }
        if (this.multiTenantMetadata.isUp()) {
            this.scheduler.scheduleOnce(String.format("proactive-key-generation-%s-%s", suffix, logicalClusterId), () -> this.maybeCreateActiveKeyForTenant(logicalClusterId));
        } else {
            log.error("MultiTenantMetadata is not up, unable to create active key for tenant. Retrying after 5 minutes...");
            this.scheduler.scheduleOnce(String.format("proactive-key-generation-retry-%s-%s", suffix, logicalClusterId), () -> this.scheduleProactiveKeyGenerationForTopicIdPartition(topicIdPartition, suffix), PROACTIVE_KEY_GENERATION_RETRY_DELAY_MS.intValue());
        }
    }

    private synchronized void maybeCreateActiveKeyForTenant(String logicalClusterId) {
        if (!this.proactiveKeyGenerationEnable.booleanValue()) {
            return;
        }
        ByokMetadata byokMetadata = this.getByokMetadata(logicalClusterId);
        if (byokMetadata == null) {
            return;
        }
        TenantKeyCache tenantKeyCache = this.getOrCreateTenantKeyCache(logicalClusterId);
        if (tenantKeyCache.activeKeySha() == null) {
            log.info("[{}] Tenant key cache does not contain active key, generating one proactively.", (Object)logicalClusterId);
            if (!this.forceRotate(logicalClusterId, tenantKeyCache, byokMetadata)) {
                log.error("[{}] Failed to create active key, retrying after 5 minutes...", (Object)logicalClusterId);
                this.scheduler.scheduleOnce(String.format("proactive-key-generation-retry-%s", logicalClusterId), () -> this.forceRotate(logicalClusterId, tenantKeyCache, byokMetadata), PROACTIVE_KEY_GENERATION_RETRY_DELAY_MS.intValue());
            }
        } else {
            log.debug("[{}] Tenant key cache contains active key, skipping proactive key generation.", (Object)logicalClusterId);
        }
    }

    public static String getKeyWithFallbackToLegacy(Map<String, String> metadata, String key, List<String> legacyKeys) {
        String value;
        block1: {
            String legacyKey;
            value = metadata.get(key);
            if (value != null && !value.isEmpty()) break block1;
            Iterator<String> iterator = legacyKeys.iterator();
            while (iterator.hasNext() && ((value = metadata.get(legacyKey = iterator.next())) == null || value.isEmpty())) {
            }
        }
        return value;
    }

    private DataEncryptionKeyHolder parseKeyFromObjectMetadata(String logicalClusterId, ByokMetadata byokMetadata, Map<String, String> metadata, boolean tolerateMissingCreationTime) {
        String keySha = TenantAwareEncryptionKeyManager.getKeyWithFallbackToLegacy(metadata, METADATA_SHA_KEY, List.of(METADATA_SHA_KEY_LEGACY1, METADATA_SHA_KEY_LEGACY2));
        if (keySha == null || keySha.isEmpty()) {
            throw new TierObjectStoreFatalException(String.format("[%s] Neither %s, %s, or %s metadata fields are present", logicalClusterId, METADATA_SHA_KEY, METADATA_SHA_KEY_LEGACY1, METADATA_SHA_KEY_LEGACY2));
        }
        String encryptedDataKey = TenantAwareEncryptionKeyManager.getKeyWithFallbackToLegacy(metadata, METADATA_DATA_KEY, List.of(METADATA_DATA_KEY_LEGACY1, METADATA_DATA_KEY_LEGACY2));
        if (encryptedDataKey == null || encryptedDataKey.isEmpty()) {
            throw new TierObjectStoreFatalException(String.format("[%s] Neither %s, %s, or %s metadata fields are present", logicalClusterId, METADATA_DATA_KEY, METADATA_DATA_KEY_LEGACY1, METADATA_DATA_KEY_LEGACY2));
        }
        String keyCreationTime = TenantAwareEncryptionKeyManager.getKeyWithFallbackToLegacy(metadata, METADATA_KEY_CREATION_TIME, List.of(METADATA_KEY_CREATION_TIME_LEGACY1, METADATA_KEY_CREATION_TIME_LEGACY2));
        if (!tolerateMissingCreationTime && (keyCreationTime == null || keyCreationTime.isEmpty())) {
            throw new TierObjectStoreFatalException(String.format("[%s] Neither %s, %s, or %s metadata fields are present", logicalClusterId, METADATA_KEY_CREATION_TIME, METADATA_KEY_CREATION_TIME_LEGACY1, METADATA_KEY_CREATION_TIME_LEGACY2));
        }
        KeySha parsedKeySha = KeySha.fromBase64Encoded(keySha);
        EncryptedDataKey parsedEncryptedDataKey = EncryptedDataKey.fromBase64Encoded(encryptedDataKey);
        Optional<Instant> parsedKeyCreationTime = keyCreationTime == null || keyCreationTime.isEmpty() ? Optional.empty() : Optional.of(Instant.ofEpochMilli(Long.parseLong(keyCreationTime)));
        CleartextDataKey cleartextDataKey = this.decryptKey(logicalClusterId, parsedEncryptedDataKey, byokMetadata);
        DataEncryptionKeyHolder decoded = new DataEncryptionKeyHolder(parsedEncryptedDataKey, cleartextDataKey, parsedKeyCreationTime);
        if (!decoded.keySha.equals(parsedKeySha)) {
            throw new TierObjectStoreFatalException(String.format("[%s] KeySha parsed from object metadata '%s' does not match decoded KeySha '%s'", logicalClusterId, parsedKeySha, decoded.keySha));
        }
        return decoded;
    }

    private DataEncryptionKeyHolder parseActiveKeyFromObjectMetadata(String logicalClusterId, ByokMetadata byokMetadata, Map<String, String> metadata) {
        return this.parseKeyFromObjectMetadata(logicalClusterId, byokMetadata, metadata, false);
    }

    public HashMap<String, String> keyToObjectMetadata(DataEncryptionKeyHolder holder) {
        return this.keyToObjectMetadata(holder.keySha, holder.encryptedDataKey, holder.keyCreationTimeOpt);
    }

    public HashMap<String, String> keyToObjectMetadata(KeySha keySha, EncryptedDataKey encryptedDataKey, Optional<Instant> keyCreationTimeOpt) {
        HashMap<String, String> metadata = new HashMap<String, String>();
        metadata.put(METADATA_SHA_KEY, keySha.base64Encoded());
        keyCreationTimeOpt.ifPresent(instant -> metadata.put(METADATA_KEY_CREATION_TIME, Long.toString(instant.toEpochMilli())));
        metadata.put(METADATA_DATA_KEY, encryptedDataKey.base64Encoded());
        return metadata;
    }

    public TierObjectStore.ByokKeyHolder getActiveKey(String logicalClusterId) {
        KeySha active = this.activeKeySha(logicalClusterId);
        OpaqueData opaqueData = OpaqueData.fromByteArray(active.toRawBytes());
        if (opaqueData.isEmpty()) {
            return TierObjectStore.ByokKeyHolder.defaultByokKeyHolder();
        }
        TenantKeyCache tenantKeyCache = this.keyCache.get(logicalClusterId);
        DataEncryptionKeyHolder holder = tenantKeyCache.get(active);
        return new TierObjectStore.ByokKeyHolder(opaqueData, ByteBuffer.wrap(holder.encryptedDataKey.keyMaterial()));
    }

    public static boolean isTenantPrefixed(String prefixedName) {
        return prefixedName != null && !prefixedName.startsWith(TOPIC_LOGICAL_CLUSTER_ID_DELIMITER) && prefixedName.contains(TOPIC_LOGICAL_CLUSTER_ID_DELIMITER);
    }

    private static String extractTenant(String prefixedName) {
        if (!TenantAwareEncryptionKeyManager.isTenantPrefixed(prefixedName)) {
            throw new IllegalArgumentException("Name is not tenant-prefixed: " + prefixedName);
        }
        return prefixedName.substring(0, prefixedName.indexOf(TOPIC_LOGICAL_CLUSTER_ID_DELIMITER));
    }

    public String getLogicalClusterIdOrThrow(TopicIdPartition partition) {
        String logicalClusterId = this.maybeGetLogicalClusterId(partition);
        if (logicalClusterId == null) {
            throw new IllegalArgumentException("Can't find logical cluster ID in the partition: " + String.valueOf(partition));
        }
        return logicalClusterId;
    }

    public String maybeGetLogicalClusterId(TopicIdPartition partition) {
        String logicalClusterId;
        try {
            logicalClusterId = TenantAwareEncryptionKeyManager.extractTenant(partition.topic());
        }
        catch (IllegalArgumentException e) {
            return null;
        }
        if (!logicalClusterId.startsWith(LKC_PREFIX)) {
            throw new IllegalArgumentException(String.format("Found a logical cluster ID: %s that does not start with the prefix: %s", logicalClusterId, LKC_PREFIX));
        }
        return logicalClusterId;
    }

    protected abstract Map<String, String> fetchWellKnownPathMetadata(String var1);

    protected abstract void writeWellKnownPathMetadata(String var1, Map<String, String> var2);

    protected abstract CleartextDataKey decryptKey(String var1, EncryptedDataKey var2, ByokMetadata var3);

    protected abstract DataEncryptionKeyHolder encryptKey(String var1, Optional<CleartextDataKey> var2, ByokMetadata var3);

    protected byte[] getEncodedDataKeyBytes(String logicalClusterId) {
        KeyGenerator keyGenerator;
        try {
            keyGenerator = KeyGenerator.getInstance(this.getDataEncryptionKeyAlgorithm());
        }
        catch (GeneralSecurityException e) {
            throw new TierObjectStoreFatalException(String.format("[%s] Failed to generate data key due to a non-retryable failure.", logicalClusterId), e);
        }
        keyGenerator.init(this.getDataEncryptionKeySizeBytes(), new SecureRandom());
        SecretKey dataKey = keyGenerator.generateKey();
        return dataKey.getEncoded();
    }

    protected abstract String getDataEncryptionKeyAlgorithm();

    protected abstract int getDataEncryptionKeySizeBytes();

    public abstract String getKeyManagementClientEncryptionAlgorithm();

    protected static String getLastActiveKeyPath(String prefix, String logicalClusterId) {
        return String.format("%s%s/last-active-key/%s", prefix, DataTypePathPrefix.LAST_ACTIVE_ENCRYPTION_KEY.prefix(), logicalClusterId);
    }
}

