/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.analyzer.goals;

import com.linkedin.kafka.cruisecontrol.analyzer.ActionAcceptance;
import com.linkedin.kafka.cruisecontrol.analyzer.ActionType;
import com.linkedin.kafka.cruisecontrol.analyzer.AnalyzerUtils;
import com.linkedin.kafka.cruisecontrol.analyzer.BalancingConstraint;
import com.linkedin.kafka.cruisecontrol.analyzer.OptimizationOptions;
import com.linkedin.kafka.cruisecontrol.analyzer.PartitionBalancingAction;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.AcceptanceResult;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.EntityFilter;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.Goal;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.NoOpReplicaFilter;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.internals.CandidateBroker;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.util.HotPartitionsInfo;
import com.linkedin.kafka.cruisecontrol.common.Resource;
import com.linkedin.kafka.cruisecontrol.exception.OptimizationFailureException;
import com.linkedin.kafka.cruisecontrol.model.Broker;
import com.linkedin.kafka.cruisecontrol.model.ClusterModel;
import com.linkedin.kafka.cruisecontrol.model.Disk;
import com.linkedin.kafka.cruisecontrol.model.Partition;
import com.linkedin.kafka.cruisecontrol.model.Replica;
import com.linkedin.kafka.cruisecontrol.model.ReplicaWrapper;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.SortedSet;
import java.util.stream.Collectors;
import kafka.common.TenantHelpers;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GoalUtils {
    private static final Logger LOG = LoggerFactory.getLogger(GoalUtils.class);
    public static final int MIN_NUM_VALID_WINDOWS_FOR_SELF_HEALING = 1;
    public static final Double MAXIMUM_CAPACITY_DEVIATION_PERCENTAGE = 0.05;
    public static final double DEAD_BROKER_UTILIZATION = 1.0;
    private static final double DEAD_DISK_UTILIZATION = 1.0;

    private GoalUtils() {
    }

    static void filterOutBrokersExcludedForLeadership(List<Broker> originalBrokers, OptimizationOptions optimizationOptions, Replica replica, ActionType action) {
        GoalUtils.filterOutBrokersExcludedForLeadership(originalBrokers, optimizationOptions.excludedBrokersForLeadership(), replica, action);
    }

    public static Optional<Double> validateEvenBrokerResourceCapacities(Map<Integer, Double> capacityByBrokerId, Resource resource) {
        if (capacityByBrokerId.isEmpty()) {
            LOG.warn("GoalUtils#validateEvenBrokerResourceCapacities called with empty capacity map - this is not expected.");
            return Optional.empty();
        }
        ArrayList<Double> allCapacities = new ArrayList<Double>(capacityByBrokerId.values());
        HashSet<Double> capacities = new HashSet<Double>(allCapacities);
        if (capacities.size() != 1) {
            double averageCapacity = allCapacities.stream().mapToDouble(a -> a).average().orElse(0.0);
            double maxCapacity = averageCapacity + averageCapacity * MAXIMUM_CAPACITY_DEVIATION_PERCENTAGE;
            double minCapacity = averageCapacity - averageCapacity * MAXIMUM_CAPACITY_DEVIATION_PERCENTAGE;
            List brokersOutsideAllowedCapacity = capacityByBrokerId.entrySet().stream().filter(kv -> (Double)kv.getValue() > maxCapacity || (Double)kv.getValue() < minCapacity).map(kv -> new BrokerCapacity((Integer)kv.getKey(), (Double)kv.getValue())).collect(Collectors.toList());
            if (!brokersOutsideAllowedCapacity.isEmpty()) {
                String errorMsg = String.format("Found inconsistent broker capacities for resource %s. Computed min/average/max capacity %.2f/%.2f/%.2f. The maximum allowed deviation was configured to %.2f%% outside the average and brokers %s were outside that threshold.", resource.name(), minCapacity, averageCapacity, maxCapacity, MAXIMUM_CAPACITY_DEVIATION_PERCENTAGE * 100.0, brokersOutsideAllowedCapacity);
                LOG.warn(errorMsg);
                return Optional.empty();
            }
            return Optional.of(averageCapacity);
        }
        return Optional.of((Double)capacities.iterator().next());
    }

    static void filterOutBrokersExcludedForLeadership(List<Broker> originalBrokers, Set<Integer> excludedBrokers, Replica replica, ActionType action) {
        boolean isLiveLeaderReplica;
        if (excludedBrokers.isEmpty()) {
            return;
        }
        boolean bl = isLiveLeaderReplica = replica.originalBroker().isAlive() && replica.isLeader();
        if (action == ActionType.LEADERSHIP_MOVEMENT || isLiveLeaderReplica) {
            originalBrokers.removeIf(broker -> excludedBrokers.contains(broker.id()));
        }
    }

    static void filterOutBrokersExcludedForReplicaMove(List<Broker> originalBrokers, OptimizationOptions optimizationOptions, Replica replica, ActionType action) {
        Set<Integer> excludedBrokers = optimizationOptions.excludedBrokersForReplicaMove();
        if (excludedBrokers.isEmpty()) {
            return;
        }
        if (action == ActionType.INTER_BROKER_REPLICA_MOVEMENT && replica.originalBroker().isAlive()) {
            originalBrokers.removeIf(broker -> excludedBrokers.contains(broker.id()));
        }
    }

    static List<Broker> eligibleBrokers(ClusterModel clusterModel, Replica replica, Collection<Broker> candidates, ActionType action, OptimizationOptions optimizationOptions) {
        ArrayList<Broker> filteredBrokers = new ArrayList<Broker>(candidates);
        GoalUtils.filterOutBrokersExcludedForLeadership(filteredBrokers, optimizationOptions, replica, action);
        GoalUtils.filterOutBrokersExcludedForReplicaMove(filteredBrokers, optimizationOptions, replica, action);
        if (clusterModel.newBrokers().isEmpty()) {
            return filteredBrokers;
        }
        boolean allowLeadershipMove = replica.isObserver() && replica.isLeader() && action == ActionType.LEADERSHIP_MOVEMENT;
        return filteredBrokers.stream().filter(b -> b.isNew() || b == replica.originalBroker() || allowLeadershipMove).collect(Collectors.toList());
    }

    public static boolean legitMove(Replica replica, Broker destinationBroker, ClusterModel clusterModel, ActionType actionType) {
        if (replica.broker().id() == destinationBroker.id()) {
            return false;
        }
        switch (actionType) {
            case INTER_BROKER_REPLICA_MOVEMENT: {
                return clusterModel.partition(replica.topicPartition()).canAssignReplicaToBroker(destinationBroker) && !destinationBroker.hasReplicaOfPartition(replica.topicPartition());
            }
            case LEADERSHIP_MOVEMENT: {
                return replica.isLeader() && destinationBroker.hasReplicaOfPartition(replica.topicPartition());
            }
        }
        return false;
    }

    static SortedSet<Replica> eligibleReplicasForSwap(ClusterModel clusterModel, Replica sourceReplica, CandidateBroker cb) {
        Broker destinationBroker;
        if (cb.shouldExcludeForLeadership(sourceReplica) || cb.shouldExcludeForReplicaMove(sourceReplica)) {
            return Collections.emptySortedSet();
        }
        SortedSet<Replica> candidateReplicasToSwapWith = cb.replicas();
        Broker sourceBroker = sourceReplica.broker();
        Broker broker = destinationBroker = candidateReplicasToSwapWith.isEmpty() ? null : candidateReplicasToSwapWith.first().broker();
        if (clusterModel.newBrokers().isEmpty() || destinationBroker == null || sourceBroker.isNew() && (destinationBroker.isNew() || sourceReplica.originalBroker() == destinationBroker)) {
            return candidateReplicasToSwapWith;
        }
        if (destinationBroker.isNew()) {
            candidateReplicasToSwapWith.removeIf(replica -> replica.originalBroker() != sourceBroker);
            return candidateReplicasToSwapWith;
        }
        return Collections.emptySortedSet();
    }

    public static void ensureNoOfflineReplicas(ClusterModel clusterModel, String goalName) throws OptimizationFailureException {
        for (Replica replica : clusterModel.selfHealingEligibleReplicas()) {
            if (!replica.isCurrentOffline()) continue;
            throw new OptimizationFailureException(String.format("[%s] Self healing failed to move the replica %s from %s broker %d (contains %d replicas).", new Object[]{goalName, replica, replica.broker().strategy(), replica.broker().id(), replica.broker().replicas().size()}));
        }
    }

    public static void ensureReplicasMoveOffBrokersWithBadDisks(ClusterModel clusterModel, String goalName) throws OptimizationFailureException {
        for (Broker broker : clusterModel.brokersWithBadDisks()) {
            for (Replica replica : broker.replicas()) {
                if (clusterModel.partition(replica.topicPartition()).canAssignReplicaToBroker(broker)) continue;
                throw new OptimizationFailureException(String.format("[%s] A replica of partition %s has been moved back to broker %d, where it was originally hosted on a broken disk.", goalName, clusterModel.partition(replica.topicPartition()), replica.broker().id()));
            }
        }
    }

    private static Set<Replica> filterLeaders(Broker broker, boolean immigrantsOnly) {
        HashSet<Replica> filteredLeaders;
        if (immigrantsOnly) {
            filteredLeaders = new HashSet<Replica>(broker.immigrantReplicas());
            filteredLeaders.removeIf(replica -> !replica.isLeader());
        } else {
            filteredLeaders = new HashSet<Replica>(broker.leaderReplicas());
        }
        return filteredLeaders;
    }

    public static Set<Replica> filterReplicas(Broker broker, boolean followersOnly, boolean leadersOnly, boolean immigrantsOnly) {
        return GoalUtils.filterReplicas(broker, followersOnly, leadersOnly, immigrantsOnly, new NoOpReplicaFilter());
    }

    public static Set<Replica> filterReplicas(Broker broker, boolean followersOnly, boolean leadersOnly, boolean immigrantsOnly, EntityFilter<Replica> replicaFilter) {
        if (leadersOnly) {
            Set<Replica> leaders = followersOnly ? Collections.emptySet() : GoalUtils.filterLeaders(broker, immigrantsOnly);
            replicaFilter.filterEntities(leaders);
            return leaders;
        }
        HashSet<Replica> filteredReplicas = new HashSet<Replica>(immigrantsOnly ? broker.immigrantReplicas() : broker.replicas());
        if (followersOnly) {
            filteredReplicas.removeAll(broker.leaderReplicas());
        }
        replicaFilter.filterEntities(filteredReplicas);
        return filteredReplicas;
    }

    public static double utilizationPercentage(Broker broker, Resource resource) {
        double brokerCapacity = broker.capacity().totalCapacityFor(resource);
        return brokerCapacity > 0.0 ? broker.load().expectedUtilizationFor(resource) / brokerCapacity : 1.0;
    }

    public static double diskUtilizationPercentage(Disk disk) {
        double diskCapacity = disk.capacity();
        return diskCapacity > 0.0 ? disk.utilization() / diskCapacity : 1.0;
    }

    static void sortReplicasInAscendingOrderByBrokerRackLessResourceUtilization(List<Replica> replicas, Resource.CompositeResource resource) {
        replicas.sort((r1, r2) -> {
            double expectedBrokerLoad2;
            double expectedBrokerLoad1 = r1.broker().load().expectedUtilizationForRackLessResources(resource);
            int result = Double.compare(expectedBrokerLoad1, expectedBrokerLoad2 = r2.broker().load().expectedUtilizationForRackLessResources(resource));
            return result == 0 ? Integer.compare(r1.broker().id(), r2.broker().id()) : result;
        });
    }

    protected static Broker minResourceInBrokers(Set<Broker> brokersToAnalyze, Resource.CompositeResource resource, Broker.ResourceComparator.Mode resourceComparatorMode, BalancingConstraint balancingConstraint) {
        Optional<Broker> minBroker = brokersToAnalyze.stream().min(new Broker.ResourceComparator(resource, balancingConstraint, resourceComparatorMode, true));
        return minBroker.orElseThrow(() -> new IllegalArgumentException(String.format("Broker set is empty when computing minimum resource among brokers", new Object[0])));
    }

    private static double minHotPartitionResourceUsage(Set<Broker> brokersToAnalyze, Resource.CompositeResource resource, BalancingConstraint balancingConstraint) {
        return balancingConstraint.hotPartitionUtilizationThreshold() * GoalUtils.minResourceInBrokers(brokersToAnalyze, resource, Broker.ResourceComparator.Mode.TOTAL, balancingConstraint).capacity().totalCapacityFor(resource.capacityResource);
    }

    private static double minSaturatedReplicaResourceUsage(Set<Broker> brokersToAnalyze, Resource.CompositeResource resource, BalancingConstraint balancingConstraint) {
        return GoalUtils.minResourceInBrokers(brokersToAnalyze, resource, Broker.ResourceComparator.Mode.ALLOWED, balancingConstraint).allowedCapacity(resource.capacityResource, balancingConstraint);
    }

    public static HotPartitionsInfo analyzeHotPartitions(Set<Broker> brokersToAnalyze, Resource.CompositeResource resource, BalancingConstraint balancingConstraint, String goalName, String sortName) {
        HashMap<Integer, List<TopicPartition>> hotPartitionsByBroker = new HashMap<Integer, List<TopicPartition>>();
        LinkedHashMap<TopicPartition, Double> hotPartitionsWithCorrespondingUsages = new LinkedHashMap<TopicPartition, Double>();
        double maxReplicaLoad = 0.0;
        HashMap<Replica, Double> saturatedReplicasWithCorrespondingUsages = new HashMap<Replica, Double>();
        double minSaturatedReplicaResourceUsage = GoalUtils.minSaturatedReplicaResourceUsage(brokersToAnalyze, resource, balancingConstraint);
        double minHotPartitionResourceUsage = GoalUtils.minHotPartitionResourceUsage(brokersToAnalyze, resource, balancingConstraint);
        HashMap<Integer, Double> maxReplicaLoadPerBroker = new HashMap<Integer, Double>();
        for (Broker broker : brokersToAnalyze) {
            Iterator<ReplicaWrapper> iterator = broker.trackedSortedReplicas(sortName).reverselySortedIterator();
            double maxReplicaLoadForBroker = 0.0;
            while (iterator.hasNext()) {
                ReplicaWrapper replicaWrapper = iterator.next();
                Replica replica = replicaWrapper.replica();
                double avgResourceUsage = replicaWrapper.score();
                maxReplicaLoad = Math.max(maxReplicaLoad, avgResourceUsage);
                maxReplicaLoadForBroker = Math.max(maxReplicaLoadForBroker, avgResourceUsage);
                if (avgResourceUsage > minHotPartitionResourceUsage) {
                    hotPartitionsByBroker.putIfAbsent(replica.broker().id(), new ArrayList());
                    ((List)hotPartitionsByBroker.get(replica.broker().id())).add(replica.topicPartition());
                    hotPartitionsWithCorrespondingUsages.put(replica.topicPartition(), avgResourceUsage);
                }
                if (avgResourceUsage > minSaturatedReplicaResourceUsage) {
                    saturatedReplicasWithCorrespondingUsages.put(replica, avgResourceUsage);
                    replica.markSaturatedResource(resource.capacityResource);
                }
                if (!(avgResourceUsage <= Math.min(minHotPartitionResourceUsage, minSaturatedReplicaResourceUsage))) continue;
                break;
            }
            maxReplicaLoadPerBroker.put(broker.id(), maxReplicaLoadForBroker);
        }
        LOG.info("Max replica load per broker for resource {} in {} is: {}", new Object[]{resource.capacityResource, resource.capacityResource.unit(), maxReplicaLoadPerBroker.entrySet()});
        HotPartitionsInfo hotPartitionsInfo = new HotPartitionsInfo(hotPartitionsByBroker, hotPartitionsWithCorrespondingUsages, saturatedReplicasWithCorrespondingUsages, maxReplicaLoad);
        if (!hotPartitionsWithCorrespondingUsages.isEmpty()) {
            LOG.warn(hotPartitionsInfo.hotPartitionsLog(goalName, resource.capacityResource, balancingConstraint.hotPartitionUtilizationThreshold() * 100.0));
        }
        return hotPartitionsInfo;
    }

    public static Map<Replica, Broker> getPartitionMoves(ClusterModel clusterModel, Set<Goal> optimizedGoals, List<Replica> replicasToMove, Iterable<List<Broker>> brokersIterable) {
        for (List<Broker> brokers : brokersIterable) {
            PartitionBalancingAction proposal;
            AcceptanceResult isProposalAcceptable;
            HashMap<Replica, Broker> replicaMoves = new HashMap<Replica, Broker>();
            int replicasThatNeedToBeMoved = replicasToMove.size();
            for (Replica replicaToMove : replicasToMove) {
                if (brokers.contains(replicaToMove.broker())) {
                    --replicasThatNeedToBeMoved;
                    continue;
                }
                Broker broker = brokers.stream().filter(b -> b.attributes().equals(replicaToMove.broker().attributes())).findFirst().orElse(brokers.iterator().next());
                if (!GoalUtils.legitMove(replicaToMove, broker, clusterModel, ActionType.INTER_BROKER_REPLICA_MOVEMENT)) break;
                replicaMoves.put(replicaToMove, broker);
                brokers.remove(broker);
            }
            if (replicaMoves.size() != replicasThatNeedToBeMoved || (isProposalAcceptable = AnalyzerUtils.isProposalAcceptableForOptimizedGoals(optimizedGoals, proposal = new PartitionBalancingAction(replicaMoves), clusterModel)).acceptance() != ActionAcceptance.ACCEPT) continue;
            return replicaMoves;
        }
        return Collections.emptyMap();
    }

    public static int numPartitionCells(Partition partition) {
        return (int)partition.partitionBrokers().stream().map(Broker::cell).distinct().count();
    }

    public static boolean nonTenantTopic(String topic) {
        return TenantHelpers.extractTenantPrefix((String)topic, (boolean)false) == null;
    }

    public static boolean partitionHasReplicaOnIgnoredBroker(Partition partition) {
        return partition.partitionBrokers().stream().anyMatch(broker -> broker.strategy() == Broker.Strategy.IGNORE);
    }

    static boolean isUtilAcceptableForCapacity(double utilization, double capacity) {
        return utilization <= capacity;
    }

    private static class BrokerCapacity {
        private final int brokerId;
        private final double capacity;

        public BrokerCapacity(int brokerId, double capacity) {
            this.brokerId = brokerId;
            this.capacity = capacity;
        }

        public String toString() {
            return "BrokerCapacity{brokerId=" + this.brokerId + ", capacity=" + this.capacity + "}";
        }
    }
}

