/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.detector.tenantstriping;

import com.google.common.collect.ImmutableList;
import com.linkedin.kafka.cruisecontrol.config.ConfigSupplier;
import com.linkedin.kafka.cruisecontrol.config.KafkaCruiseControlConfig;
import com.linkedin.kafka.cruisecontrol.detector.ResourceUtilizationDetector;
import com.linkedin.kafka.cruisecontrol.detector.tenantstriping.CellSelector;
import com.linkedin.kafka.cruisecontrol.detector.tenantstriping.DesiredStripeFactorCalculator;
import com.linkedin.kafka.cruisecontrol.detector.tenantstriping.TenantOverloadDetector;
import com.linkedin.kafka.cruisecontrol.detector.utils.CellUtils;
import com.linkedin.kafka.cruisecontrol.detector.utils.TenantResourceUsageComparator;
import com.linkedin.kafka.cruisecontrol.detector.utils.TenantUtils;
import com.linkedin.kafka.cruisecontrol.model.CellResourceUsage;
import com.linkedin.kafka.cruisecontrol.model.ClusterModel;
import com.linkedin.kafka.cruisecontrol.model.ResourceDetectorSharedState;
import com.linkedin.kafka.cruisecontrol.model.Tenant;
import com.linkedin.kafka.cruisecontrol.model.TenantResource;
import com.linkedin.kafka.cruisecontrol.model.TenantResourceUsage;
import com.linkedin.kafka.cruisecontrol.model.TenantResourceUsageMappings;
import com.linkedin.kafka.cruisecontrol.model.TenantStripingInfo;
import com.linkedin.kafka.cruisecontrol.model.TenantStripingSharedState;
import com.linkedin.kafka.cruisecontrol.model.view.CellTenantView;
import com.linkedin.kafka.cruisecontrol.model.view.ClusterModelCellView;
import com.linkedin.kafka.cruisecontrol.monitor.ModelGeneration;
import io.confluent.databalancer.metrics.TenantStripingMetrics;
import io.confluent.kafka.clients.AssignTenantsToCellResult;
import io.confluent.kafka.clients.CloudAdmin;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.kafka.common.CellState;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.message.AssignTenantsToCellRequestData;
import org.apache.kafka.common.message.AssignTenantsToCellResponseData;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TenantAutoScaler
implements ResourceUtilizationDetector {
    private static final Logger LOG = LoggerFactory.getLogger(TenantAutoScaler.class);
    private final CloudAdmin cloudAdmin;
    private final TenantOverloadDetector tenantOverloadDetector;
    private final CellSelector cellSelector;
    private final TenantStripingSharedState tenantStripingSharedState;
    private final DesiredStripeFactorCalculator desiredStripeFactorCalculator;
    private final TenantSortingHelper tenantSortingHelper = new TenantSortingHelper();
    private final ConfigSupplier configSupplier;
    private final long maxReplicasPerBroker;
    private final TenantStripingMetrics tenantStripingMetrics;
    private final Time time;
    private ModelGeneration lastCheckedGeneration;

    public TenantAutoScaler(CloudAdmin cloudAdmin, ConfigSupplier configSupplier, Time time) {
        this.cloudAdmin = cloudAdmin;
        this.configSupplier = configSupplier;
        KafkaCruiseControlConfig kccConfig = configSupplier.getConfig();
        this.tenantStripingMetrics = new TenantStripingMetrics();
        this.desiredStripeFactorCalculator = new DesiredStripeFactorCalculator(kccConfig);
        this.maxReplicasPerBroker = kccConfig.getLong("max.replicas");
        this.tenantOverloadDetector = new TenantOverloadDetector(kccConfig, time, this.desiredStripeFactorCalculator, this.tenantStripingMetrics);
        this.cellSelector = new CellSelector(kccConfig, this.tenantStripingMetrics);
        this.tenantStripingSharedState = new TenantStripingSharedState();
        this.time = time;
        this.lastCheckedGeneration = null;
        this.tenantStripingMetrics.newGauge("dry-run-mode", this::isDryRunMode);
    }

    TenantAutoScaler(CloudAdmin cloudAdmin, ConfigSupplier configSupplier, TenantOverloadDetector tenantOverloadDetector, DesiredStripeFactorCalculator desiredStripeFactorCalculator, CellSelector cellSelector, long maxReplicasPerBroker, TenantStripingMetrics tenantStripingMetrics, Time time) {
        this.cloudAdmin = cloudAdmin;
        this.configSupplier = configSupplier;
        this.tenantOverloadDetector = tenantOverloadDetector;
        this.desiredStripeFactorCalculator = desiredStripeFactorCalculator;
        this.cellSelector = cellSelector;
        this.tenantStripingSharedState = new TenantStripingSharedState();
        this.maxReplicasPerBroker = maxReplicasPerBroker;
        this.tenantStripingMetrics = tenantStripingMetrics;
        this.time = time;
        this.lastCheckedGeneration = null;
    }

    @Override
    public void detectResourceUtilization(ClusterModel clusterModel, ResourceDetectorSharedState detectorSharedState) {
        if (this.lastCheckedGeneration != null && this.lastCheckedGeneration.loadGeneration() == clusterModel.generation().loadGeneration()) {
            LOG.info("Skipping Tenant Auto Scaler as the metric's generation have not changed.");
            return;
        }
        this.lastCheckedGeneration = clusterModel.generation();
        if (!this.tenantStripingSharedState.pendingTenantsExecutions().isEmpty()) {
            LOG.info("There are tenants whose executions are not completed yet: {}, hence skipping tenant auto striping.", (Object)this.tenantStripingSharedState.pendingTenantsExecutions().toString());
            return;
        }
        this.tenantStripingSharedState.resourceDetectorSharedState(detectorSharedState);
        Optional<ClusterModelCellView> cellViewOpt = CellUtils.clusterModelCellView(clusterModel, LOG.getName());
        if (cellViewOpt.isEmpty()) {
            LOG.warn("TenantAutoScaler skipping detection: No cluster model cellview available");
            return;
        }
        ClusterModelCellView cellView = cellViewOpt.get();
        List<CellTenantView> readyCells = cellView.cells().stream().filter(cellTenantView -> cellTenantView.state().equals((Object)CellState.READY)).collect(Collectors.toList());
        this.cellSelector.updateReadyCells(readyCells);
        Map<Integer, CellResourceUsage> cellResourceLoadStats = cellView.cellResourceLoadStats(this.maxReplicasPerBroker);
        this.cellSelector.updateCellResourceLoadStats(cellResourceLoadStats);
        Map<String, TenantStripingInfo> tenantStripingInfoMap = this.populateTenantStripesAndUsages(cellView);
        Set<String> excludedTenants = cellView.cells().stream().filter(cellTenantView -> cellTenantView.state().equals((Object)CellState.QUARANTINED)).flatMap(cellTenantView -> cellTenantView.tenants().stream()).map(Tenant::tenantId).collect(Collectors.toSet());
        excludedTenants.addAll(TenantUtils.excludedTenants(this.configSupplier));
        Map<String, TenantResourceUsageMappings> candidateTenantsToStripeUp = this.tenantOverloadDetector.getTenantsToStripeUp(cellView, tenantStripingInfoMap, excludedTenants);
        Set<String> sortedCandidateTenantIds = this.tenantSortingHelper.getSortedTenants(tenantStripingInfoMap, candidateTenantsToStripeUp);
        boolean isDryRunMode = this.isDryRunMode();
        int stripesPlacedForCurrentRound = 0;
        int totalEvaluatedTenants = 0;
        int totalStripedTenants = 0;
        StringBuilder tenantStripeSummary = new StringBuilder();
        LOG.info("TenantAutoScaler initiated striping process for {} tenants: {} (dry-run: {})", new Object[]{sortedCandidateTenantIds.size(), String.join((CharSequence)", ", sortedCandidateTenantIds), isDryRunMode});
        for (String tenantId : sortedCandidateTenantIds) {
            if (this.cellSelector.isRateLimitReached()) {
                LOG.info("TenantAutoScaler rate-limit reached after processing {} tenants. Skipped tenants in this round: {}", (Object)totalEvaluatedTenants, (Object)sortedCandidateTenantIds.stream().skip(totalEvaluatedTenants).collect(Collectors.joining(", ")));
                break;
            }
            TenantStripingInfo tenantStripingInfo = tenantStripingInfoMap.get(tenantId);
            tenantStripingInfo.spikedResourceUsages().addResourceMappings(candidateTenantsToStripeUp.get(tenantId));
            List<Integer> selectedCells = this.maybeSelectCells(tenantStripingInfo, tenantStripeSummary);
            if (!selectedCells.isEmpty()) {
                ImmutableList<Integer> existingStripes = tenantStripingInfo.existingStripes();
                tenantStripeSummary.append("Tenant ").append(tenantId).append(" residing on cells: ").append(existingStripes).append(" striped to new cells :").append(selectedCells).append("\n");
                this.cellSelector.updateTenantStripes(tenantId, selectedCells);
                stripesPlacedForCurrentRound += selectedCells.size();
                Map<Integer, CellResourceUsage> updatedCellResourceLoadStats = this.cellSelector.getUpdatedCellResourceLoadStats(readyCells, selectedCells, tenantStripingInfo);
                this.cellSelector.updateCellResourceLoadStats(updatedCellResourceLoadStats);
                ++totalStripedTenants;
            } else {
                LOG.info("TenantAutoScaler found no suitable cells available for striping tenant: {}", (Object)tenantId);
            }
            ++totalEvaluatedTenants;
        }
        if (this.cellSelector.isRateLimitReached()) {
            int tenantsSkippedCount = sortedCandidateTenantIds.size() - totalEvaluatedTenants;
            this.tenantStripingMetrics.recordRateLimitedTenantsCount(tenantsSkippedCount);
        } else {
            this.tenantStripingMetrics.recordRateLimitedTenantsCount(0);
        }
        this.executeMetadataUpdate(tenantStripingInfoMap, tenantStripeSummary);
        this.tenantStripingMetrics.recordStripesPlacedCount(stripesPlacedForCurrentRound);
        LOG.info("TenantAutoScaler execution summary: {}\nTotal Stripes placed for current round : {}, Total number of tenants striped : {}, Dry-run mode: {}", new Object[]{tenantStripeSummary, stripesPlacedForCurrentRound, totalStripedTenants, isDryRunMode});
        this.cellSelector.clearCurrentRoundState();
    }

    void executeMetadataUpdate(Map<String, TenantStripingInfo> tenantStripingInfoMap, StringBuilder tenantStripeSummary) {
        if (!this.cellSelector.getTenantToCellAssignments().isEmpty() && !this.isDryRunMode()) {
            Map<String, List<Integer>> tenantToCellAssignments = this.cellSelector.getTenantToCellAssignments();
            Map<String, List<Integer>> existingTenantStripes = this.getExistingTenantStripes(tenantStripingInfoMap);
            try {
                Map<String, List<Integer>> successfulTenantAssignments = this.updateTenantMetadata(tenantToCellAssignments, existingTenantStripes, tenantStripeSummary);
                this.updateSharedState(successfulTenantAssignments, existingTenantStripes);
                LOG.info("TenantAutoScaler successfully updated tenant metadata for tenants: {}", (Object)String.join((CharSequence)", ", successfulTenantAssignments.keySet()));
            }
            catch (Exception e) {
                LOG.error("TenantAutoScaler failed to update tenant metadata for tenants: {}", (Object)String.join((CharSequence)", ", tenantToCellAssignments.keySet()), (Object)e);
            }
        }
    }

    void updateSharedState(Map<String, List<Integer>> tenantToCellAssignments, Map<String, List<Integer>> existingTenantStripes) {
        for (Map.Entry<String, List<Integer>> entry : tenantToCellAssignments.entrySet()) {
            String tenantId = entry.getKey();
            List<Integer> selectedCells = entry.getValue();
            List<Integer> existingStripes = existingTenantStripes.getOrDefault(tenantId, Collections.emptyList());
            this.tenantStripingSharedState.addOngoingTenants(tenantId);
            this.tenantStripingSharedState.addExcludedCells(selectedCells, existingStripes);
            LOG.debug("TenantAutoScaler added tenant {} to ongoing assignments with new cells {}", (Object)tenantId, selectedCells);
        }
    }

    private Map<String, TenantStripingInfo> populateTenantStripesAndUsages(ClusterModelCellView cellView) {
        HashMap<String, TenantStripingInfo> tenantStripingInfoMap = new HashMap<String, TenantStripingInfo>();
        cellView.tenantsById().forEach((tenantId, tenant) -> {
            int stripeCount = Optional.ofNullable(tenant.cellIds()).orElse(Collections.emptyList()).size();
            TenantStripingInfo tenantStripingInfo = new TenantStripingInfo((String)tenantId);
            tenantStripingInfo.stripeCount(stripeCount);
            tenantStripingInfoMap.put((String)tenantId, tenantStripingInfo);
            for (TenantResource tenantResource : TenantResource.cachedValues()) {
                tenantStripingInfo.latestResourceUsages().addResourceMapping(tenantResource, tenant.getResourceUsage(tenantResource.resource()));
            }
            ImmutableList existingStripesForTenant = ImmutableList.copyOf(cellView.tenantsById().get(tenantId).cellIds());
            tenantStripingInfo.existingStripes((ImmutableList<Integer>)existingStripesForTenant);
        });
        return tenantStripingInfoMap;
    }

    List<Integer> maybeSelectCells(TenantStripingInfo tenantStripingInfo, StringBuilder tenantStripeSummary) {
        List<Integer> cells = new ArrayList<Integer>();
        int maxDesiredStripeFactor = this.desiredStripeFactorCalculator.calculateMaxDesiredStripeFactor(tenantStripingInfo.spikedResourceUsages(), tenantStripeSummary);
        String tenantId = tenantStripingInfo.tenantId();
        tenantStripeSummary.append(" Tenant : ").append(tenantId).append(" Max Desired Stripe Factor : ").append(maxDesiredStripeFactor).append("\n");
        ImmutableList<Integer> existingStripesForTenant = tenantStripingInfo.existingStripes();
        if (existingStripesForTenant.size() >= maxDesiredStripeFactor) {
            this.tenantOverloadDetector.removeFromCandidates(tenantId);
        } else {
            cells = this.cellSelector.selectCells(maxDesiredStripeFactor, tenantStripingInfo);
        }
        return cells;
    }

    @Override
    public void close() {
        this.tenantStripingSharedState.clearSharedState();
        this.tenantOverloadDetector.close();
        this.cellSelector.close();
    }

    @Override
    public void updateTenantReassignmentState(Set<String> tenants) {
        this.tenantStripingSharedState.removeOngoingTenantAssignments(tenants);
    }

    public TenantStripingSharedState tenantStripingSharedState() {
        return this.tenantStripingSharedState;
    }

    private boolean isDryRunMode() {
        return this.configSupplier.getConfig().getBoolean("tenant.striping.enable.dry.run.mode");
    }

    Map<String, List<Integer>> updateTenantMetadata(Map<String, List<Integer>> tenantToCellAssignments, Map<String, List<Integer>> existingTenantStripes, StringBuilder tenantStripingSummary) throws Exception {
        LOG.info("TenantAutoScaler updating tenant cell mapping for tenants {} with updated stripes", tenantToCellAssignments.keySet());
        List<AssignTenantsToCellRequestData.TenantToCellAssignment> assignments = this.createTenantAssignments(tenantToCellAssignments, existingTenantStripes);
        Set<Object> failedAssignments = new HashSet();
        HashMap<String, List<Integer>> successfulTenantAssignments = new HashMap<String, List<Integer>>();
        try {
            AssignTenantsToCellResult result = this.cloudAdmin.assignTenantsToCells(assignments);
            int apiTimeout = this.configSupplier.getConfig().getInt("default.api.timeout.ms");
            List tenantAssignmentErrors = (List)result.value().get((long)apiTimeout, TimeUnit.MILLISECONDS);
            if (!tenantAssignmentErrors.isEmpty()) {
                this.populateTenantAssignmentErrors(tenantStripingSummary, tenantAssignmentErrors);
                failedAssignments = tenantAssignmentErrors.stream().map(AssignTenantsToCellResponseData.TenantAssignmentErrors::tenantId).collect(Collectors.toSet());
                failedAssignments.forEach(this.tenantStripingMetrics::recordTenantMetadataUpdateFailure);
            } else {
                tenantStripingSummary.append("All tenant assignments completed successfully.").append(System.lineSeparator());
            }
        }
        catch (TimeoutException te) {
            String message = "Timed out waiting for tenant assignments to complete";
            LOG.error(message, (Throwable)te);
            tenantStripingSummary.append(message).append(System.lineSeparator());
            tenantToCellAssignments.keySet().forEach(this.tenantStripingMetrics::recordTenantMetadataUpdateFailure);
            throw te;
        }
        catch (Exception e) {
            String message = "Error assigning tenants to cells";
            LOG.error(message, (Throwable)e);
            tenantStripingSummary.append(message).append(": ").append(e.getMessage()).append(System.lineSeparator());
            tenantToCellAssignments.keySet().forEach(this.tenantStripingMetrics::recordTenantMetadataUpdateFailure);
            throw e;
        }
        LOG.info("TenantAutoScaler successfully completed tenant to cell mapping update for tenants {}", tenantToCellAssignments.keySet());
        for (String tenantId : tenantToCellAssignments.keySet()) {
            if (failedAssignments.contains(tenantId)) continue;
            successfulTenantAssignments.put(tenantId, tenantToCellAssignments.get(tenantId));
            this.tenantStripingMetrics.clearTenantMetadataUpdateFailure(tenantId);
        }
        return successfulTenantAssignments;
    }

    private List<AssignTenantsToCellRequestData.TenantToCellAssignment> createTenantAssignments(Map<String, List<Integer>> tenantToCellAssignments, Map<String, List<Integer>> existingTenantStripes) {
        ArrayList<AssignTenantsToCellRequestData.TenantToCellAssignment> assignments = new ArrayList<AssignTenantsToCellRequestData.TenantToCellAssignment>();
        for (Map.Entry<String, List<Integer>> tenantStripe : tenantToCellAssignments.entrySet()) {
            String tenantId = tenantStripe.getKey();
            List existingCellIds = existingTenantStripes.getOrDefault(tenantId, Collections.emptyList());
            List<Integer> selectedCellIds = tenantStripe.getValue();
            ArrayList assignedCellIds = new ArrayList(existingCellIds);
            assignedCellIds.addAll(selectedCellIds);
            assignedCellIds.sort(Integer::compareTo);
            assignments.add(new AssignTenantsToCellRequestData.TenantToCellAssignment().setTenantId(tenantId).setCellIds(assignedCellIds));
        }
        return assignments;
    }

    private void populateTenantAssignmentErrors(StringBuilder tenantStripingSummary, List<AssignTenantsToCellResponseData.TenantAssignmentErrors> tenantAssignmentErrors) {
        tenantStripingSummary.append("Tenant Assignment Errors:").append(System.lineSeparator());
        tenantAssignmentErrors.forEach(tenantAssignmentError -> tenantStripingSummary.append(tenantAssignmentError.toString()).append(System.lineSeparator()));
    }

    private Map<String, List<Integer>> getExistingTenantStripes(Map<String, TenantStripingInfo> tenantStripingInfoMap) {
        return tenantStripingInfoMap.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> ((TenantStripingInfo)entry.getValue()).existingStripes()));
    }

    class TenantSortingHelper {
        TenantSortingHelper() {
        }

        private Set<String> getSortedTenants(Map<String, TenantStripingInfo> stripeFactorMap, Map<String, TenantResourceUsageMappings> consistentlySpikedTenants) {
            TenantResourceUsageComparator tenantResourceUsageComparator = new TenantResourceUsageComparator(stripeFactorMap, TenantAutoScaler.this.desiredStripeFactorCalculator);
            return consistentlySpikedTenants.entrySet().stream().map(TenantSortingHelper::getTenantResourceEntries).flatMap(Collection::stream).sorted(tenantResourceUsageComparator).map(TenantResourceUsage::tenantId).collect(Collectors.toCollection(LinkedHashSet::new));
        }

        private static List<TenantResourceUsage> getTenantResourceEntries(Map.Entry<String, TenantResourceUsageMappings> resourceUsages) {
            return resourceUsages.getValue().resourceEntries().stream().map(resourceEntry -> new TenantResourceUsage((String)resourceUsages.getKey(), (TenantResource)((Object)((Object)resourceEntry.getKey())), (Double)resourceEntry.getValue())).collect(Collectors.toList());
        }
    }
}

