/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.model;

import com.linkedin.kafka.cruisecontrol.common.Resource;
import com.linkedin.kafka.cruisecontrol.common.ResourceVisitor;
import com.linkedin.kafka.cruisecontrol.model.Broker;
import com.linkedin.kafka.cruisecontrol.model.Load;
import com.linkedin.kafka.cruisecontrol.model.Rack;
import com.linkedin.kafka.cruisecontrol.model.Replica;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.kafka.common.TopicPartition;

public class Partition
implements Serializable {
    private final TopicPartition tp;
    private final List<Replica> replicas;
    private Replica leader;
    private final Set<Broker> brokersWithBadDisk;
    static final double FOLLOWER_CPU_EXP_RATIO = 0.3;

    public Partition(TopicPartition tp) {
        this.tp = tp;
        this.replicas = new ArrayList<Replica>();
        this.leader = null;
        this.brokersWithBadDisk = new HashSet<Broker>();
    }

    public TopicPartition topicPartition() {
        return this.tp;
    }

    void addFollower(Replica follower, int index) {
        if (follower.isLeader()) {
            throw new IllegalArgumentException("Inconsistent leadership information. Trying to add follower replica " + String.valueOf(follower) + " while it is a leader.");
        }
        if (!follower.topicPartition().equals((Object)this.tp)) {
            throw new IllegalArgumentException("Inconsistent topic partition. Trying to add follower replica " + String.valueOf(follower) + " to partition " + String.valueOf(this.tp) + ".");
        }
        this.replicas.add(index, follower);
    }

    void deleteReplica(int brokerId) {
        this.replicas.removeIf(r -> r.broker().id() == brokerId);
    }

    public List<Replica> replicas() {
        return Collections.unmodifiableList(this.replicas);
    }

    public List<Replica> followers() {
        ArrayList<Replica> followers = new ArrayList<Replica>();
        this.replicas.forEach(r -> {
            if (!r.isLeader()) {
                followers.add((Replica)r);
            }
        });
        return followers;
    }

    public List<Replica> onlineFollowers() {
        ArrayList<Replica> onlineFollowers = new ArrayList<Replica>();
        for (Replica follower : this.followers()) {
            if (follower.isCurrentOffline()) continue;
            onlineFollowers.add(follower);
        }
        return onlineFollowers;
    }

    public Replica leader() {
        return this.leader;
    }

    Replica replica(long brokerId) {
        for (Replica replica : this.replicas) {
            if ((long)replica.broker().id() != brokerId) continue;
            return replica;
        }
        throw new IllegalArgumentException("Requested replica " + brokerId + " is not a replica of partition " + String.valueOf(this.tp));
    }

    public List<Broker> followerBrokers() {
        ArrayList<Broker> followerBrokers = new ArrayList<Broker>();
        this.replicas.forEach(r -> {
            if (!r.isLeader()) {
                followerBrokers.add(r.broker());
            }
        });
        return followerBrokers;
    }

    public List<Broker> onlineFollowerBrokers() {
        ArrayList<Broker> onlineFollowerBrokers = new ArrayList<Broker>();
        this.replicas.forEach(r -> {
            if (!r.isLeader() && !r.isCurrentOffline()) {
                onlineFollowerBrokers.add(r.broker());
            }
        });
        return onlineFollowerBrokers;
    }

    public void swapFollowerPositions(int index1, int index2) {
        Replica follower1 = this.replicas.get(index1);
        Replica follower2 = this.replicas.get(index2);
        if (follower1.isLeader() || follower2.isLeader()) {
            throw new IllegalArgumentException(String.format("%s is not a follower.", follower1.isLeader() ? follower1 : follower2));
        }
        this.replicas.set(index2, follower1);
        this.replicas.set(index1, follower2);
    }

    public void swapReplicaPositions(int index1, int index2) {
        Replica replica1 = this.replicas.get(index1);
        Replica replica2 = this.replicas.get(index2);
        this.replicas.set(index2, replica1);
        this.replicas.set(index1, replica2);
    }

    public void moveReplicaToEnd(Replica replica) {
        if (!this.replicas.remove(replica)) {
            throw new IllegalStateException(String.format("Did not find replica %s for partition %s.", replica, this.tp));
        }
        this.replicas.add(replica);
    }

    public Set<Broker> partitionBrokers() {
        HashSet<Broker> partitionBrokers = new HashSet<Broker>();
        this.replicas.forEach(r -> partitionBrokers.add(r.broker()));
        return partitionBrokers;
    }

    public Set<Broker> partitionObserverBrokers() {
        return this.replicas.stream().filter(Replica::isObserver).map(Replica::broker).collect(Collectors.toSet());
    }

    public Set<Broker> partitionSyncBrokers() {
        return this.replicas.stream().filter(replica -> !replica.isObserver()).map(Replica::broker).collect(Collectors.toSet());
    }

    public Set<Broker> partitionSyncBrokersOfStateOtherThan(Broker.Strategy state) {
        return this.replicas.stream().filter(replica -> !replica.isObserver()).map(Replica::broker).filter(broker -> broker.strategy() != state).collect(Collectors.toSet());
    }

    public boolean containsRack(Rack rack) {
        return this.replicas.stream().anyMatch(r -> r.broker().rack().equals(rack));
    }

    void addLeader(Replica leader, int index) {
        if (this.leader != null) {
            throw new IllegalArgumentException(String.format("Partition %s already has a leader replica %s. Cannot add a new leader replica %s", this.tp, this.leader, leader));
        }
        if (!leader.isLeader()) {
            throw new IllegalArgumentException("Inconsistent leadership information. Trying to set " + String.valueOf(leader.broker()) + " as the leader for partition " + String.valueOf(this.tp) + " while the replica is not marked as a leader.");
        }
        this.leader = leader;
        this.replicas.add(index, leader);
    }

    Replica findReplicaWithMedianCPULoad() {
        if (this.replicas.isEmpty()) {
            throw new RuntimeException(String.format("findReplicaWithMedianCPULoad called for Partition: (%s) without replicas but should only be called on partitions with at least one replica.", this));
        }
        Comparator<Replica> replicaCpuLoadComparator = Comparator.comparingDouble(r -> r.load().expectedUtilizationFor(Resource.CPU));
        long nonZeroCpuReplicasCount = (long)this.replicas.size() - this.replicas.stream().filter(r -> r.load().expectedUtilizationFor(Resource.CPU) == 0.0).count();
        int medianIdx = (int)Math.floor((double)nonZeroCpuReplicasCount / 2.0);
        return this.replicas.stream().filter(r -> r.load().expectedUtilizationFor(Resource.CPU) > 0.0).sorted(replicaCpuLoadComparator).skip(medianIdx).findFirst().orElse(this.leader());
    }

    void relocateLeadership(Replica prospectiveLeader) {
        int leaderPos = this.replicas.indexOf(prospectiveLeader);
        this.swapReplicaPositions(0, leaderPos);
        this.leader = prospectiveLeader;
    }

    public Load estimatedNewReplicaLoad() {
        if (this.replicas.isEmpty()) {
            throw new RuntimeException(String.format("estimatedNewReplicaLoad called for Partition: (%s) without replicas but should only be called on partitions with at least one replica.", this));
        }
        Load newReplicaLoad = new Load();
        Load leaderLoad = this.leader.load();
        newReplicaLoad.initializeMetricValues(leaderLoad.loadByWindows(), leaderLoad.windows());
        if (this.partitionBrokers().size() == 1) {
            newReplicaLoad.multiplyLoadFor(Resource.CPU, 0.3);
        } else {
            newReplicaLoad.copyLoadFrom(Resource.CPU, this.findReplicaWithMedianCPULoad().load());
        }
        ResourceVisitor<Boolean> skipResource = new ResourceVisitor<Boolean>(){

            @Override
            public Boolean visitCpuResource() {
                return false;
            }

            @Override
            public Boolean visitNetworkInResource() {
                return false;
            }

            @Override
            public Boolean visitNetworkOutResource() {
                return true;
            }

            @Override
            public Boolean visitDiskResource() {
                return false;
            }

            @Override
            public Boolean visitProduceInResource() {
                return true;
            }

            @Override
            public Boolean visitConsumeOutResource() {
                return true;
            }

            @Override
            public Boolean visitMirrorInResource() {
                return true;
            }

            @Override
            public Boolean visitReplicationInResource() {
                return true;
            }

            @Override
            public Boolean visitRackBasedConsumeOutResource() {
                return true;
            }

            @Override
            public Boolean visitRackLessConsumeOutResource() {
                return true;
            }
        };
        Arrays.stream(Resource.values()).filter(resource -> (Boolean)resource.visit(skipResource)).forEach(newReplicaLoad::clearLoadFor);
        return newReplicaLoad;
    }

    public void clear() {
        this.replicas.clear();
        this.leader = null;
    }

    public String toString() {
        StringBuilder partition = new StringBuilder().append(String.format("%n<Partition>%n<Leader>%s</Leader>%n", this.leader));
        for (Replica replica : this.replicas) {
            if (!replica.isLeader() && !replica.isObserver()) {
                partition.append(String.format("<Follower>%s</Follower>%n", replica));
            }
            if (!replica.isObserver()) continue;
            partition.append(String.format("<Observer>%s</Observer>%n", replica));
        }
        return partition.append(String.format("</Partition>%n", new Object[0])).toString();
    }

    public void addBadDiskBroker(Broker ineligibleBroker) {
        this.brokersWithBadDisk.add(ineligibleBroker);
    }

    public boolean canAssignReplicaToBroker(Broker candidateBroker) {
        return !this.brokersWithBadDisk.contains(candidateBroker);
    }

    public double saturatedReplicaMoveCount(Resource resource) {
        return this.replicas().stream().filter(replica -> replica.originalBroker() != replica.broker() && replica.isSaturatedForResource(resource)).count();
    }
}

