/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.gateway.common.util;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileWatcher
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileWatcher.class);
    private final Path watchedFile;
    private final Consumer<String> onFileChange;
    private final AtomicReference<String> lastKnownContent = new AtomicReference();
    private final AtomicBoolean isRunning = new AtomicBoolean(false);
    private final ExecutorService executorService;
    private final WatchService watchService;
    private volatile Future<?> watchTask;
    private final AtomicLong lastModifiedTime = new AtomicLong(0L);

    public FileWatcher(Path watchedFile, Consumer<String> onFileChange) throws IOException {
        this.watchedFile = watchedFile;
        this.onFileChange = onFileChange;
        this.executorService = Executors.newSingleThreadExecutor(r -> {
            Thread t = new Thread(r, "file-watcher-" + String.valueOf(watchedFile.getFileName()));
            t.setDaemon(true);
            return t;
        });
        this.lastKnownContent.set(this.readFileContent());
        this.initializeFileMetadata();
        this.watchService = watchedFile.getFileSystem().newWatchService();
        Path parentDir = watchedFile.getParent();
        if (parentDir != null) {
            parentDir.register(this.watchService, StandardWatchEventKinds.ENTRY_MODIFY);
        }
        LOGGER.info("Created file watcher for: {}", (Object)watchedFile);
    }

    public void start() {
        if (this.isRunning.compareAndSet(false, true)) {
            this.watchTask = this.executorService.submit(this::watchForChanges);
            LOGGER.info("Started file watcher for: {}", (Object)this.watchedFile);
        }
    }

    public void stop() {
        if (this.isRunning.compareAndSet(true, false)) {
            if (this.watchTask != null && !this.watchTask.isDone()) {
                this.watchTask.cancel(true);
            }
            LOGGER.info("Stopped file watcher for: {}", (Object)this.watchedFile);
        }
    }

    private void watchForChanges() {
        LOGGER.debug("Starting file watch loop for: {}", (Object)this.watchedFile);
        while (this.isRunning.get() && !Thread.currentThread().isInterrupted()) {
            try {
                WatchKey key = this.watchService.poll(5L, TimeUnit.SECONDS);
                if (key == null) continue;
                key.pollEvents().forEach(event -> {
                    WatchEvent.Kind kind = event.kind();
                    if (kind == StandardWatchEventKinds.OVERFLOW) {
                        LOGGER.warn("Watch service overflow occurred for: {}", (Object)this.watchedFile);
                        return;
                    }
                    if (kind == StandardWatchEventKinds.ENTRY_MODIFY) {
                        WatchEvent ev = event;
                        Path filename = (Path)ev.context();
                        LOGGER.debug("Watch event triggered: kind={}, file={}", (Object)kind.name(), (Object)filename);
                        if (this.watchedFile.getFileName().equals(filename)) {
                            this.handleFileChange();
                        }
                    }
                });
                boolean valid = key.reset();
                if (valid) continue;
                LOGGER.warn("Watch key is no longer valid for: {}", (Object)this.watchedFile);
                break;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.debug("File watcher interrupted for: {}", (Object)this.watchedFile);
                break;
            }
            catch (Exception e) {
                LOGGER.error("Error in file watcher for: {}", (Object)this.watchedFile, (Object)e);
            }
        }
        LOGGER.debug("File watch loop ended for: {}", (Object)this.watchedFile);
    }

    private void handleFileChange() {
        try {
            if (!this.hasFileChanged()) {
                LOGGER.debug("No metadata changes detected for: {}", (Object)this.watchedFile);
                return;
            }
            String newContent = this.readFileContent();
            if (newContent != null) {
                LOGGER.info("Detected content change in file: {}", (Object)this.watchedFile);
                this.lastKnownContent.set(newContent);
                try {
                    this.onFileChange.accept(newContent);
                }
                catch (Exception e) {
                    LOGGER.error("Error in file change callback for: {}", (Object)this.watchedFile, (Object)e);
                }
            } else {
                LOGGER.warn("File metadata changed but content could not be read for: {}", (Object)this.watchedFile);
            }
        }
        catch (Exception e) {
            LOGGER.error("Error handling file change for: {}", (Object)this.watchedFile, (Object)e);
        }
    }

    private String readFileContent() {
        try {
            return Files.readString(this.watchedFile);
        }
        catch (IOException e) {
            LOGGER.error("Failed to read file: {}", (Object)this.watchedFile, (Object)e);
            return null;
        }
    }

    private void initializeFileMetadata() {
        try {
            if (Files.exists(this.watchedFile, new LinkOption[0])) {
                this.lastModifiedTime.set(Files.getLastModifiedTime(this.watchedFile, new LinkOption[0]).toMillis());
                LOGGER.debug("Initialized file metadata - modified: {} for: {}", (Object)this.lastModifiedTime.get(), (Object)this.watchedFile);
            }
        }
        catch (IOException e) {
            LOGGER.warn("Failed to initialize file metadata for: {}", (Object)this.watchedFile, (Object)e);
            this.lastModifiedTime.set(0L);
        }
    }

    private boolean hasFileChanged() {
        try {
            boolean changed;
            if (!Files.exists(this.watchedFile, new LinkOption[0])) {
                return this.lastModifiedTime.get() != 0L;
            }
            long currentModTime = Files.getLastModifiedTime(this.watchedFile, new LinkOption[0]).toMillis();
            boolean bl = changed = currentModTime != this.lastModifiedTime.get();
            if (changed) {
                LOGGER.debug("File metadata changed - old: [modified={}], new: [modified={}] for: {}", new Object[]{this.lastModifiedTime.get(), currentModTime, this.watchedFile});
                this.lastModifiedTime.set(currentModTime);
            }
            return changed;
        }
        catch (IOException e) {
            LOGGER.warn("Failed to check file metadata for: {}", (Object)this.watchedFile, (Object)e);
            return true;
        }
    }

    public String getLastKnownContent() {
        return this.lastKnownContent.get();
    }

    public boolean isRunning() {
        return this.isRunning.get();
    }

    @Override
    public void close() {
        this.stop();
        try {
            if (this.watchService != null) {
                this.watchService.close();
            }
        }
        catch (IOException e) {
            LOGGER.warn("Error closing watch service for: {}", (Object)this.watchedFile, (Object)e);
        }
        if (this.executorService != null) {
            this.executorService.shutdown();
            try {
                if (!this.executorService.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.executorService.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.executorService.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        LOGGER.info("Closed file watcher for: {}", (Object)this.watchedFile);
    }

    @Generated
    public Path getWatchedFile() {
        return this.watchedFile;
    }
}

