/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.gateway.common.util;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.fasterxml.jackson.dataformat.yaml.YAMLGenerator;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.Map;
import java.util.Set;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileUtil.class);
    private static final String GATEWAY_TEMP_DIR = ".gateway-secure-temp";
    private static final String POSIX_FILE_PERMISSIONS = "rw-------";
    private static final String POSIX_DIR_PERMISSIONS = "rwx------";

    private FileUtil() {
    }

    public static boolean isFileReadable(String filePath) {
        if (StringUtils.isEmpty((CharSequence)filePath)) {
            return false;
        }
        Path file = Paths.get(filePath, new String[0]);
        if (!Files.exists(file, new LinkOption[0])) {
            return false;
        }
        if (Files.isDirectory(file, new LinkOption[0])) {
            return false;
        }
        return Files.isReadable(file);
    }

    public static void writeToFile(String filePath, String content) {
        if (StringUtils.isEmpty((CharSequence)filePath) || StringUtils.isEmpty((CharSequence)content)) {
            throw new IllegalArgumentException("File path and content must not be empty");
        }
        Path file = Paths.get(filePath, new String[0]);
        try {
            if (file.getParent() != null) {
                Files.createDirectories(file.getParent(), new FileAttribute[0]);
            }
            Files.writeString(file, (CharSequence)content, new OpenOption[0]);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to write to file: " + filePath, e);
        }
    }

    public static void writeYamlFile(String filePath, Map<String, Object> yaml) {
        String yamlString;
        if (yaml == null) {
            throw new IllegalArgumentException("YAML cannot be null");
        }
        YAMLFactory yamlFactory = new YAMLFactory().disable(YAMLGenerator.Feature.WRITE_DOC_START_MARKER).disable(YAMLGenerator.Feature.USE_NATIVE_TYPE_ID).enable(YAMLGenerator.Feature.INDENT_ARRAYS_WITH_INDICATOR);
        ObjectMapper mapper = new ObjectMapper((JsonFactory)yamlFactory).setDefaultPropertyInclusion(JsonInclude.Include.NON_NULL);
        try {
            yamlString = mapper.writeValueAsString(yaml);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to write configuration to file: " + filePath, e);
        }
        Path path = Path.of(filePath, new String[0]);
        if (!Files.exists(path, new LinkOption[0])) {
            try {
                Files.createFile(path, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to create configuration file: " + filePath, e);
            }
        }
        FileUtil.writeToFile(path.toString(), yamlString);
    }

    public static Path createSecureTempFile(String prefix, String suffix) throws IOException {
        return FileUtil.createSecureTempFile(prefix, suffix, false);
    }

    public static Path createSecureTempFile(String prefix, String suffix, boolean deleteOnExit) throws IOException {
        Path secureDir = FileUtil.getOrCreateSecureTempDirectory(deleteOnExit);
        Path tempFile = SystemUtils.IS_OS_UNIX ? FileUtil.createUnixSecureTempFile(secureDir, prefix, suffix) : FileUtil.createWindowsSecureTempFile(secureDir, prefix, suffix);
        if (deleteOnExit) {
            tempFile.toFile().deleteOnExit();
            LOGGER.debug("Scheduled temp file for deletion on exit: {}", (Object)tempFile);
        }
        return tempFile;
    }

    public static Path createSecureTempDirectory(String prefix) throws IOException {
        return FileUtil.createSecureTempDirectory(prefix, false);
    }

    public static Path createSecureTempDirectory(String prefix, boolean deleteOnExit) throws IOException {
        Path parentSecureDir = FileUtil.getOrCreateSecureTempDirectory();
        Path tempDir = SystemUtils.IS_OS_UNIX ? FileUtil.createUnixSecureTempDirectory(parentSecureDir, prefix) : FileUtil.createWindowsSecureTempDirectory(parentSecureDir, prefix);
        if (deleteOnExit) {
            tempDir.toFile().deleteOnExit();
            LOGGER.debug("Scheduled temp directory for deletion on exit: {}", (Object)tempDir);
        }
        return tempDir;
    }

    private static Path createUnixSecureTempDirectory(Path parentDir, String prefix) throws IOException {
        FileAttribute<Set<PosixFilePermission>> attr = PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString(POSIX_DIR_PERMISSIONS));
        Path tempDir = Files.createTempDirectory(parentDir, prefix, attr);
        LOGGER.debug("Created secure temp directory (unix): {}", (Object)tempDir);
        return tempDir;
    }

    private static Path createWindowsSecureTempDirectory(Path parentDir, String prefix) throws IOException {
        Path tempDir = Files.createTempDirectory(parentDir, prefix, new FileAttribute[0]);
        File dir = tempDir.toFile();
        if (!FileUtil.setWindowsOwnerOnlyPermissions(dir)) {
            Files.deleteIfExists(tempDir);
            throw new IOException("Failed to set secure permissions on temp directory. Directory deleted for security.");
        }
        LOGGER.debug("Created secure temp directory (windows): {}", (Object)tempDir);
        return tempDir;
    }

    private static Path createUnixSecureTempFile(Path secureDir, String prefix, String suffix) throws IOException {
        FileAttribute<Set<PosixFilePermission>> attr = PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString(POSIX_FILE_PERMISSIONS));
        Path tempPath = Files.createTempFile(secureDir, prefix, suffix, attr);
        LOGGER.debug("Created secure temp file (unix): {}", (Object)tempPath);
        return tempPath;
    }

    private static Path createWindowsSecureTempFile(Path secureDir, String prefix, String suffix) throws IOException {
        Path tempPath = Files.createTempFile(secureDir, prefix, suffix, new FileAttribute[0]);
        File file = tempPath.toFile();
        if (!FileUtil.setWindowsOwnerOnlyPermissions(file)) {
            Files.deleteIfExists(tempPath);
            throw new IOException("Failed to set secure permissions on temp file. File deleted for security.");
        }
        LOGGER.debug("Created secure temp file (windows): {}", (Object)tempPath);
        return tempPath;
    }

    private static synchronized Path getOrCreateSecureTempDirectory() throws IOException {
        return FileUtil.getOrCreateSecureTempDirectory(false);
    }

    private static synchronized Path getOrCreateSecureTempDirectory(boolean deleteOnExit) throws IOException {
        Path secureDir = Paths.get(System.getProperty("java.io.tmpdir"), GATEWAY_TEMP_DIR);
        boolean directoryCreated = false;
        if (!Files.exists(secureDir, new LinkOption[0])) {
            if (SystemUtils.IS_OS_UNIX) {
                FileUtil.createUnixSecureDirectory(secureDir);
            } else {
                FileUtil.createWindowsSecureDirectory(secureDir);
            }
            directoryCreated = true;
            LOGGER.info("Created secure temp directory: {}", (Object)secureDir.toAbsolutePath());
        }
        if (deleteOnExit && directoryCreated) {
            secureDir.toFile().deleteOnExit();
            LOGGER.debug("Scheduled temp directory for deletion on exit: {}", (Object)secureDir);
        }
        return secureDir;
    }

    private static void createUnixSecureDirectory(Path secureDir) throws IOException {
        FileAttribute<Set<PosixFilePermission>> attr = PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString(POSIX_DIR_PERMISSIONS));
        Files.createDirectories(secureDir, attr);
    }

    private static void createWindowsSecureDirectory(Path secureDir) throws IOException {
        Files.createDirectories(secureDir, new FileAttribute[0]);
        File dir = secureDir.toFile();
        if (!FileUtil.setWindowsOwnerOnlyPermissions(dir)) {
            Files.deleteIfExists(secureDir);
            throw new IOException("Failed to set secure permissions on temp directory. Directory deleted for security.");
        }
    }

    private static boolean setWindowsOwnerOnlyPermissions(File file) {
        boolean success = file.setReadable(false, false);
        success &= file.setWritable(false, false);
        success &= file.setExecutable(false, false);
        success &= file.setReadable(true, true);
        success &= file.setWritable(true, true);
        if (!(success &= file.setExecutable(true, true))) {
            LOGGER.error("Failed to set owner-only permissions on: {}", (Object)file.getAbsolutePath());
        }
        return success;
    }
}

