/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.jdbc.sink;

import io.confluent.connect.jdbc.dialect.DatabaseDialect;
import io.confluent.connect.jdbc.sink.BufferedRecords;
import io.confluent.connect.jdbc.sink.DbStructure;
import io.confluent.connect.jdbc.sink.JdbcSinkConfig;
import io.confluent.connect.jdbc.sink.TableAlterOrCreateException;
import io.confluent.connect.jdbc.util.CachedConnectionProvider;
import io.confluent.connect.jdbc.util.TableId;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.sink.SinkRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcDbWriter {
    private static final Logger log = LoggerFactory.getLogger(JdbcDbWriter.class);
    private final JdbcSinkConfig config;
    private final DatabaseDialect dbDialect;
    private final DbStructure dbStructure;
    final CachedConnectionProvider cachedConnectionProvider;

    JdbcDbWriter(JdbcSinkConfig config, DatabaseDialect dbDialect, DbStructure dbStructure) {
        this.config = config;
        this.dbDialect = dbDialect;
        this.dbStructure = dbStructure;
        this.cachedConnectionProvider = this.connectionProvider(config.connectionAttempts, config.connectionBackoffMs);
    }

    protected CachedConnectionProvider connectionProvider(int maxConnAttempts, long retryBackoff) {
        return new CachedConnectionProvider(this.dbDialect, maxConnAttempts, retryBackoff){

            @Override
            protected void onConnect(Connection connection) throws SQLException {
                log.info("JdbcDbWriter Connected");
                connection.setAutoCommit(false);
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void write(Collection<SinkRecord> records) throws SQLException, TableAlterOrCreateException {
        Connection connection = this.cachedConnectionProvider.getConnection();
        String schemaName = this.getSchemaSafe(connection).orElse(null);
        String catalogName = this.getCatalogSafe(connection).orElse(null);
        try {
            BufferedRecords buffer;
            TableId tableId;
            HashMap<TableId, BufferedRecords> bufferByTable = new HashMap<TableId, BufferedRecords>();
            for (SinkRecord sinkRecord : records) {
                tableId = this.destinationTable(sinkRecord.topic(), schemaName, catalogName);
                buffer = (BufferedRecords)bufferByTable.get(tableId);
                if (buffer == null) {
                    buffer = new BufferedRecords(this.config, tableId, this.dbDialect, this.dbStructure, connection);
                    bufferByTable.put(tableId, buffer);
                }
                buffer.add(sinkRecord);
            }
            for (Map.Entry entry : bufferByTable.entrySet()) {
                tableId = (TableId)entry.getKey();
                buffer = (BufferedRecords)entry.getValue();
                log.debug("Flushing records in JDBC Writer for table ID: {}", (Object)tableId);
                buffer.flush();
                buffer.close();
            }
            log.trace("Committing transaction");
            connection.commit();
        }
        catch (TableAlterOrCreateException | SQLException e) {
            log.error("Error during write operation. Attempting rollback.", (Throwable)e);
            try {
                connection.rollback();
                log.info("Successfully rolled back transaction");
            }
            catch (SQLException sqle) {
                log.error("Failed to rollback transaction", (Throwable)sqle);
                ((Throwable)e).addSuppressed(sqle);
            }
            finally {
                throw e;
            }
        }
        log.info("Completed write operation for {} records to the database", (Object)records.size());
    }

    void closeQuietly() {
        this.cachedConnectionProvider.close();
    }

    TableId destinationTable(String topic, String schemaName, String catalogName) {
        String tableName = this.config.tableNameFormat.replace("${topic}", topic);
        if (tableName.isEmpty()) {
            throw new ConnectException(String.format("Destination table name for topic '%s' is empty using the format string '%s'", topic, this.config.tableNameFormat));
        }
        TableId parsedTableId = this.dbDialect.parseTableIdentifier(tableName);
        String finalCatalogName = parsedTableId.catalogName() != null ? parsedTableId.catalogName() : catalogName;
        String finalSchemaName = parsedTableId.schemaName() != null ? parsedTableId.schemaName() : schemaName;
        return new TableId(finalCatalogName, finalSchemaName, parsedTableId.tableName());
    }

    private Optional<String> getSchemaSafe(Connection connection) {
        try {
            return Optional.ofNullable(connection.getSchema());
        }
        catch (AbstractMethodError | SQLException e) {
            log.warn("Failed to get schema: {}", (Object)e.getMessage());
            return Optional.empty();
        }
    }

    private Optional<String> getCatalogSafe(Connection connection) {
        try {
            return Optional.ofNullable(connection.getCatalog());
        }
        catch (AbstractMethodError | SQLException e) {
            log.warn("Failed to get catalog: {}", (Object)e.getMessage());
            return Optional.empty();
        }
    }
}

