/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafkarest.resources.v3;

import com.google.common.annotations.VisibleForTesting;
import io.confluent.kafkarest.config.ConfigModule;
import io.confluent.kafkarest.exceptions.RateLimitGracePeriodExceededException;
import java.time.Clock;
import java.time.Duration;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import javax.inject.Inject;

final class ProduceRateLimiter {
    private static final int ONE_SECOND_MS = 1000;
    private final int maxRequestsPerSecond;
    private final int maxBytesPerSecond;
    private final long gracePeriod;
    private final boolean rateLimitingEnabled;
    private final Clock clock;
    private final AtomicInteger rateCounterSize = new AtomicInteger(0);
    private final AtomicLong byteCounterSize = new AtomicLong(0L);
    private final ConcurrentLinkedDeque<TimeAndSize> rateCounter = new ConcurrentLinkedDeque();
    private final AtomicLong gracePeriodStart = new AtomicLong(-1L);

    @Inject
    ProduceRateLimiter(@ConfigModule.ProduceGracePeriodConfig Duration produceGracePeriodConfig, @ConfigModule.ProduceRateLimitCountConfig Integer produceRateLimitCountConfig, @ConfigModule.ProduceRateLimitBytesConfig Integer produceRateLimitBytesConfig, @ConfigModule.ProduceRateLimitEnabledConfig Boolean produceRateLimitEnabledConfig, Clock clock) {
        this.maxRequestsPerSecond = Objects.requireNonNull(produceRateLimitCountConfig);
        this.maxBytesPerSecond = Objects.requireNonNull(produceRateLimitBytesConfig);
        this.gracePeriod = produceGracePeriodConfig.toMillis();
        this.rateLimitingEnabled = Objects.requireNonNull(produceRateLimitEnabledConfig);
        this.clock = Objects.requireNonNull(clock);
    }

    Optional<Duration> calculateGracePeriodExceeded(long requestSize) throws RateLimitGracePeriodExceededException {
        if (!this.rateLimitingEnabled) {
            return Optional.empty();
        }
        long nowMs = this.clock.millis();
        TimeAndSize thisMessage = new TimeAndSize(nowMs, requestSize);
        this.addToRateLimiter(thisMessage);
        Optional<Duration> waitFor = this.getWaitFor(this.rateCounterSize.get(), this.byteCounterSize.get());
        if (!waitFor.isPresent()) {
            this.resetGracePeriodStart();
            return Optional.empty();
        }
        if (this.isOverGracePeriod(nowMs)) {
            throw new RateLimitGracePeriodExceededException(this.maxRequestsPerSecond, this.maxBytesPerSecond, Duration.ofMillis(this.gracePeriod));
        }
        return waitFor;
    }

    @VisibleForTesting
    void clear() {
        this.rateCounter.clear();
        this.rateCounterSize.set(0);
    }

    @VisibleForTesting
    void resetGracePeriodStart() {
        this.gracePeriodStart.set(-1L);
    }

    private boolean isOverGracePeriod(Long nowMs) {
        if (this.gracePeriodStart.get() < 0L && this.gracePeriod != 0L) {
            this.gracePeriodStart.set(nowMs);
            return false;
        }
        return this.gracePeriod == 0L || this.gracePeriod < nowMs - this.gracePeriodStart.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addToRateLimiter(TimeAndSize thisMessage) {
        this.rateCounter.add(thisMessage);
        this.rateCounterSize.incrementAndGet();
        this.byteCounterSize.addAndGet(thisMessage.size);
        ConcurrentLinkedDeque<TimeAndSize> concurrentLinkedDeque = this.rateCounter;
        synchronized (concurrentLinkedDeque) {
            if (this.rateCounter.peekLast().time < thisMessage.time - 1000L) {
                this.rateCounter.clear();
                this.rateCounterSize.set(0);
            } else {
                while (this.rateCounter.peek().time < thisMessage.time - 1000L) {
                    TimeAndSize messageToRemove = this.rateCounter.poll();
                    this.byteCounterSize.addAndGet(-messageToRemove.size);
                    this.rateCounterSize.decrementAndGet();
                }
            }
        }
    }

    private Optional<Duration> getWaitFor(int currentCountRate, long currentByteRate) {
        if (currentCountRate <= this.maxRequestsPerSecond && currentByteRate <= (long)this.maxBytesPerSecond) {
            return Optional.empty();
        }
        double waitForMs = currentCountRate > this.maxRequestsPerSecond ? ((double)currentCountRate / (double)this.maxRequestsPerSecond - 1.0) * 1000.0 : ((double)currentByteRate / (double)this.maxBytesPerSecond - 1.0) * 1000.0;
        return Optional.of(Duration.ofMillis((long)waitForMs));
    }

    private static final class TimeAndSize {
        private long time;
        private long size;

        TimeAndSize(long time, long size) {
            this.time = time;
            this.size = size;
        }
    }
}

