/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafkarest.controllers;

import com.google.common.base.Preconditions;
import io.confluent.kafka.schemaregistry.ParsedSchema;
import io.confluent.kafka.schemaregistry.SchemaProvider;
import io.confluent.kafka.schemaregistry.client.SchemaMetadata;
import io.confluent.kafka.schemaregistry.client.SchemaRegistryClient;
import io.confluent.kafka.schemaregistry.client.rest.entities.Schema;
import io.confluent.kafka.schemaregistry.client.rest.exceptions.RestClientException;
import io.confluent.kafka.serializers.subject.strategy.SubjectNameStrategy;
import io.confluent.kafkarest.Errors;
import io.confluent.kafkarest.controllers.SchemaManager;
import io.confluent.kafkarest.entities.EmbeddedFormat;
import io.confluent.kafkarest.entities.RegisteredSchema;
import io.confluent.kafkarest.exceptions.BadRequestException;
import io.confluent.rest.exceptions.RestException;
import java.io.IOException;
import java.util.Collections;
import java.util.Objects;
import java.util.Optional;
import javax.ws.rs.core.Response;
import org.apache.avro.SchemaParseException;

final class SchemaManagerImpl
implements SchemaManager {
    private final SchemaRegistryClient schemaRegistryClient;
    private final SubjectNameStrategy defaultSubjectNameStrategy;

    SchemaManagerImpl(SchemaRegistryClient schemaRegistryClient, SubjectNameStrategy defaultSubjectNameStrategy) {
        this.schemaRegistryClient = Objects.requireNonNull(schemaRegistryClient);
        this.defaultSubjectNameStrategy = Objects.requireNonNull(defaultSubjectNameStrategy);
    }

    @Override
    public RegisteredSchema getSchema(String topicName, Optional<EmbeddedFormat> format, Optional<String> subject, Optional<SubjectNameStrategy> subjectNameStrategy, Optional<Integer> schemaId, Optional<Integer> schemaVersion, Optional<String> rawSchema, boolean isKey) {
        if (schemaId.isPresent()) {
            SchemaManagerImpl.checkArgumentWrapper(!format.isPresent());
            SchemaManagerImpl.checkArgumentWrapper(!schemaVersion.isPresent());
            SchemaManagerImpl.checkArgumentWrapper(!rawSchema.isPresent());
            return this.getSchemaFromSchemaId(topicName, subject, subjectNameStrategy, schemaId.get(), isKey);
        }
        if (schemaVersion.isPresent()) {
            SchemaManagerImpl.checkArgumentWrapper(!format.isPresent());
            SchemaManagerImpl.checkArgumentWrapper(!rawSchema.isPresent());
            return this.getSchemaFromSchemaVersion(topicName, subject, subjectNameStrategy, schemaVersion.get(), isKey);
        }
        if (rawSchema.isPresent()) {
            SchemaManagerImpl.checkArgumentWrapper(format.isPresent());
            return this.getSchemaFromRawSchema(topicName, format.get(), subject, subjectNameStrategy, rawSchema.get(), isKey);
        }
        SchemaManagerImpl.checkArgumentWrapper(!format.isPresent());
        return this.findLatestSchema(topicName, subject, subjectNameStrategy, isKey);
    }

    private RegisteredSchema getSchemaFromSchemaId(String topicName, Optional<String> subject, Optional<SubjectNameStrategy> subjectNameStrategy, int schemaId, boolean isKey) {
        ParsedSchema schema;
        try {
            schema = this.schemaRegistryClient.getSchemaById(schemaId);
        }
        catch (RestClientException | IOException e) {
            throw SchemaManagerImpl.handleRestClientException((Exception)e, "Error when fetching schema by id. schemaId = %d", schemaId);
        }
        String actualSubject = subject.orElse(subjectNameStrategy.orElse(this.defaultSubjectNameStrategy).subjectName(topicName, isKey, schema));
        int schemaVersion = this.getSchemaVersion(actualSubject, schema);
        return RegisteredSchema.create(actualSubject, schemaId, schemaVersion, schema);
    }

    private int getSchemaVersion(String subject, ParsedSchema schema) {
        try {
            return this.schemaRegistryClient.getVersion(subject, schema);
        }
        catch (RestClientException | IOException e) {
            throw SchemaManagerImpl.handleRestClientException((Exception)e, "Error when fetching schema version. subject = %s, schema = %s", subject, schema.canonicalString());
        }
    }

    private RegisteredSchema getSchemaFromSchemaVersion(String topicName, Optional<String> subject, Optional<SubjectNameStrategy> subjectNameStrategy, int schemaVersion, boolean isKey) {
        ParsedSchema parsedSchema;
        SchemaProvider schemaProvider;
        Schema schema;
        String actualSubject = subject.orElse(this.getSchemaSubjectUnsafe(topicName, isKey, subjectNameStrategy));
        try {
            schema = this.schemaRegistryClient.getByVersion(actualSubject, schemaVersion, false);
        }
        catch (RuntimeException e) {
            throw new BadRequestException(String.format("Schema does not exist for subject: %s, version: %s", actualSubject, schemaVersion), e);
        }
        try {
            schemaProvider = EmbeddedFormat.forSchemaType(schema.getSchemaType()).getSchemaProvider();
        }
        catch (UnsupportedOperationException e) {
            throw new BadRequestException(String.format("Schema version not supported for %s", schema.getSchemaType()), e);
        }
        try {
            parsedSchema = (ParsedSchema)schemaProvider.parseSchema(schema.getSchema(), schema.getReferences(), false).orElseThrow(() -> Errors.invalidSchemaException(String.format("Error when fetching schema by version. subject = %s, version = %d", actualSubject, schemaVersion)));
        }
        catch (SchemaParseException e) {
            throw new BadRequestException(String.format("Error parsing schema for %s", schema.getSchemaType()), e);
        }
        return RegisteredSchema.create(schema.getSubject(), schema.getId(), schemaVersion, parsedSchema);
    }

    private RegisteredSchema getSchemaFromRawSchema(String topicName, EmbeddedFormat format, Optional<String> subject, Optional<SubjectNameStrategy> subjectNameStrategy, String rawSchema, boolean isKey) {
        int schemaId;
        ParsedSchema schema;
        SchemaProvider schemaProvider;
        try {
            Preconditions.checkArgument((boolean)format.requiresSchema(), (String)"%s does not support schemas.", (Object)((Object)format));
        }
        catch (IllegalArgumentException e) {
            throw new BadRequestException(e.getMessage(), e);
        }
        try {
            schemaProvider = format.getSchemaProvider();
        }
        catch (UnsupportedOperationException e) {
            throw new BadRequestException(String.format("Raw schema not supported with format = %s", new Object[]{format}), e);
        }
        try {
            schema = (ParsedSchema)schemaProvider.parseSchema(rawSchema, Collections.emptyList(), true).orElseThrow(() -> Errors.invalidSchemaException(String.format("Error when parsing raw schema. format = %s, schema = %s", new Object[]{format, rawSchema})));
        }
        catch (SchemaParseException e) {
            throw new BadRequestException(String.format("Error parsing schema with format = %s", new Object[]{format}), e);
        }
        String actualSubject = subject.orElse(subjectNameStrategy.orElse(this.defaultSubjectNameStrategy).subjectName(topicName, isKey, schema));
        try {
            try {
                schemaId = this.schemaRegistryClient.getId(actualSubject, schema);
            }
            catch (RestClientException | IOException e) {
                schemaId = this.schemaRegistryClient.register(actualSubject, schema);
            }
        }
        catch (RestClientException | IOException e) {
            throw SchemaManagerImpl.handleRestClientException((Exception)e, "Error when registering schema. format = %s, subject = %s, schema = %s", new Object[]{format, actualSubject, schema.canonicalString()});
        }
        int schemaVersion = this.getSchemaVersion(actualSubject, schema);
        return RegisteredSchema.create(actualSubject, schemaId, schemaVersion, schema);
    }

    private RegisteredSchema findLatestSchema(String topicName, Optional<String> subject, Optional<SubjectNameStrategy> subjectNameStrategy, boolean isKey) {
        ParsedSchema schema;
        SchemaProvider schemaProvider;
        SchemaMetadata metadata;
        String actualSubject = subject.orElse(this.getSchemaSubjectUnsafe(topicName, isKey, subjectNameStrategy));
        try {
            metadata = this.schemaRegistryClient.getLatestSchemaMetadata(actualSubject);
        }
        catch (RestClientException | IOException e) {
            throw SchemaManagerImpl.handleRestClientException((Exception)e, "Error when fetching latest schema version. subject = %s", actualSubject);
        }
        try {
            schemaProvider = EmbeddedFormat.forSchemaType(metadata.getSchemaType()).getSchemaProvider();
        }
        catch (UnsupportedOperationException e) {
            throw new BadRequestException(String.format("Schema subject not supported for schema type = %s", metadata.getSchemaType()), e);
        }
        try {
            schema = (ParsedSchema)schemaProvider.parseSchema(metadata.getSchema(), metadata.getReferences(), false).orElseThrow(() -> Errors.invalidSchemaException(String.format("Error when fetching latest schema version. subject = %s", actualSubject)));
        }
        catch (SchemaParseException e) {
            throw new BadRequestException(String.format("Error parsing schema type = %s", metadata.getSchemaType()), e);
        }
        return RegisteredSchema.create(actualSubject, metadata.getId(), metadata.getVersion(), schema);
    }

    private String getSchemaSubjectUnsafe(String topicName, boolean isKey, Optional<SubjectNameStrategy> subjectNameStrategy) {
        SubjectNameStrategy strategy = subjectNameStrategy.orElse(this.defaultSubjectNameStrategy);
        String subject = null;
        Exception cause = null;
        try {
            subject = strategy.subjectName(topicName, isKey, null);
        }
        catch (Exception e) {
            cause = e;
        }
        if (subject == null) {
            IllegalArgumentException error = new IllegalArgumentException(String.format("Cannot use%s schema_subject_strategy%s without schema_id or schema.", subjectNameStrategy.map(requestStrategy -> "").orElse(" default"), subjectNameStrategy.map(requestStrategy -> "=" + strategy).orElse("")));
            if (cause != null) {
                error.initCause(cause);
            }
            throw error;
        }
        return subject;
    }

    private static void checkArgumentWrapper(boolean argument) {
        try {
            Preconditions.checkArgument((boolean)argument);
        }
        catch (IllegalArgumentException e) {
            throw new BadRequestException("Unsupported argument: ", e.getMessage(), e);
        }
    }

    private static RuntimeException handleRestClientException(Exception cause, String message, Object ... args) {
        if (cause instanceof RestClientException && ((RestClientException)((Object)cause)).getErrorCode() == 40301) {
            return new RestException(String.format(message, args), Response.Status.FORBIDDEN.getStatusCode(), 40301, (Throwable)cause);
        }
        return Errors.messageSerializationException(String.format(message, args), cause);
    }
}

